
/*
 * File:      size.cpp
 * Purpose:   
 * Author:
 * Created:
 * Updated:
 * Copyright: LGPL.
 *            Traveller is a registered trademark of Far Future Enterprises.
 */

/* rcsid[] = "$RCSfile: size.cpp,v $ $Revision: 1.2 $ $Author: man $ $Date: 1999/04/06 04:25:07 $" */

#include <stdio.h>
#include <math.h>
#include <string.h>
#include <ctype.h>
#include "str_util.h"
#include "size.h"


// ======================================================================
OrbitInfo::OrbitInfo(float orb, float p_mass)
{
	dice = new Dice(6, 2, 0);
	orbit = orb;
	if(p_mass > 0.0)
		CalcPeriod(p_mass);
	else
		period = 0.0;
	GenerateEccen();
}

void
OrbitInfo::GenerateEccen(void)
{
int i;

	i = dice->Roll(6,2);
	if(i < 8)
		eccen = 0.0;
	else if(i < 9)
		eccen = 0.005;
	else if(i < 10)
		eccen = 0.010;
	else if(i < 11)
		eccen = 0.015;
	else if(i < 12)
		eccen = 0.020;
	else 
		{
		dice->Roll(6,1);
		if(i < 2)
			eccen = 0.025;
		else if(i < 3)
			eccen = 0.050;
		else if(i < 4)
			eccen = 0.100;
		else if(i < 5)
			eccen = 0.200;
		else if(i < 6)
			eccen = 0.250;
		else
			eccen = 0.500;
		}
}

void 
OrbitInfo::CalcPeriod(float p_mass)
{
	period = sqrt(pow(GetAUOrbit(), 3) / p_mass) * 365.25;
}

float
OrbitInfo::GetAUOrbit(void)
{
	return(ConvToAUOrbit(orbit));
}

float
OrbitInfo::ConvToAUOrbit(float o)
{
float f,g;
int i,j,orb;

	orb = (int)(o * 10.0);
	if(orb < 0)
		return(0.0);

	i = orb / 10;
	j = orb % 10;

	if(i > 20)
		{
		f = DistTable[20] * (i - 20);
		}
	else
		{
		f = DistTable[i];
		if(j)
			{
			g = DistTable[i+1] - f;
			f += ((g * j) / 10);
			}
		}

	return(f);
}

char *
OrbitInfo::FormatPeriod(char *dest)
{
double f,g,h;

	h = fabs(period);

	if(h > 365.25)
		{
		f = modf((h / 365.25), &g) * 365.25;
		sprintf(dest, "%.0f years %.3f days ", g, f);
		}
	else
		{
		sprintf(dest, "%.3f days ", h);
		}
	if(period < 0.0)
		strcat(dest, "Retrograde ");

	return(dest);
}

float OrbitInfo::DistTable[22] = {
                .2,     .4,     .7,    1.0,     1.6,     2.8,     5.2,
              10.0,   19.6,   38.8,   77.2,    15.4,   307.6,   614.8,
            1229.2, 2458.0, 4915.6, 9830.8, 19661.2, 39322.0, 78644.0,
          157288.0 };

// ======================================================================
StarInfo::StarInfo(int t, int c, int s, float orbit, float p_mass) :
	DetailStar(NULL, t, c, s)
{
	orb = NULL;
	if(orbit > -2.0)
		orb = new OrbitInfo(orbit, p_mass);
}

StarInfo::~StarInfo()
{
	if(orb != NULL)
		delete orb;
}

char *
StarInfo::GetFormatedOrbit(char *buff)
{
	if(NULL == orb)
		sprintf(buff, "Primary");
	else if(orb->GetOrbit() < 0.0)
		sprintf(buff, "Close");
	else
		sprintf(buff, "%.1f", orb->GetOrbit());

	return(buff);
}

// ======================================================================
BaseSize::BaseSize(short s, float orb, float p_mass, int n) :
	size_desc(s)
{
	Init(orb, p_mass, n);
}

//BaseSize::BaseSize(char c, float orb, float p_mass, int n) :
//	size_desc(c)
//{
//	Init(orb, p_mass, n);
//}

void
BaseSize::Init(float orbit, float p_mass, int n)
{
	dice = new Dice(6, 2, 0);
	orb = new OrbitInfo(orbit, p_mass);
	num_sat = n;
}

// ======================================================================
ParentSize::ParentSize(short s, float orb, float p_mass, float e, 
		float m, int u, int n) :
	BaseSize(s, orb, p_mass, n)
{
	Init(m, u);
}

//ParentSize::ParentSize(char c, float orb, float p_mass, float e, 
//		float m, int u, int n) :
//	BaseSize(c, orb, p_mass, n)
//{
//	Init(m, u);
//}

void
ParentSize::Init(float m, int u)
{
	mass = m;
	uwp_size = u;
}

// ======================================================================
BeltSize::BeltSize(int zone_flag, float orb, float p_mass) :
	BaseSize(0, orb, p_mass, -1)
{
	mass = 0.0;
	Generate(zone_flag);
}

void
BeltSize::Generate(int zone_flag)
{
int i,j;

	orb->GenerateEccen();

	// predominate diameter
	i = dice->Roll(6, 2);
	if(i < 3) p_diameter = 0.001;
	else if(i < 4) p_diameter = 0.005;
	else if(i < 5) p_diameter = 0.010;
	else if(i < 6) p_diameter = 0.025;
	else if(i < 7) p_diameter = 0.050;
	else if(i < 8) p_diameter = 0.100;
	else if(i < 9) p_diameter = 0.300;
	else if(i < 10) p_diameter = 1.000;
	else if(i < 11) p_diameter = 5.000;
	else if(i < 12) p_diameter = 50.000;
	else p_diameter = 500.000;

	// max diameter
	switch(dice->Roll(6, 1))
		{
		case 3:
			{
			if(p_diameter > 1.000) m_diameter = p_diameter;
			else m_diameter = 1.000;
			}
		case 4:
			{
			if(p_diameter > 10.000) m_diameter = p_diameter;
			else m_diameter = 10.000;
			}
		case 5:
			{
			if(p_diameter > 100.000) m_diameter = p_diameter;
			else m_diameter = 100.000;
			}
		case 6:
			{
			if(p_diameter > 1000.000) m_diameter = p_diameter;
			else m_diameter = 1000.000;
			}
		default:
			m_diameter = p_diameter;
		}
	
	// zones
	if(zone_flag < HAB_HABIT)
		i = -4;
	else if(zone_flag > HAB_HABIT)
		i = 2;
	else
		i = 0;

	i = dice->Roll(6,2,i);

	if(i < 5) j = 0;
	else if(i < 9) j = 1;
	else j = 2;

	dice->Roll(6, 2, -2);
	n_zone = zone_table[j][0][i];
	m_zone = zone_table[j][1][i];
	c_zone = zone_table[j][2][i];

	i=0;

    if(orb->GetOrbit() < 5.0) i -= 3;
    else if(orb->GetOrbit() < 9.0) i--;
    else if(orb->GetOrbit() < 13.0) i++;
    else i += 2;
    
	// belt width
    i = dice->Roll(6, 2, i);
    if(i < 3) width = 0.01;
    else if(i < 4) width = 0.05;
    else if(i < 6) width = 0.1;
    else if(i < 8) width = 0.5;
    else if(i < 9) width = 1.0;
    else if(i < 10) width = 1.5;
    else if(i < 11) width = 2.0;
    else if(i < 12) width = 5.0;
    else width = 10.0;
}

// misc printing fcns:
char *
BeltSize::FormatD(int which, char *buff)
{
float f;

	if(which) f = m_diameter;
	else f = p_diameter;

	if(f < 1.0)
		sprintf(buff,"%.0fm",f*100.0);
	else
		sprintf(buff, "%.0fKm",f);

	return(buff);
}

int BeltSize::zone_table[3][3][11] =
							   {{{ 40,40,40,40,40,50,50,50,50,60,60 }, /* n */
                                 { 30,40,40,40,40,40,40,40,30,40,40 },
                                 { 30,20,20,20,20,10,10,10,20, 0, 0 }},
                                {{ 20,30,20,20,30,20,10,10,10, 0, 0 }, /* m */
                                 { 50,50,60,60,60,70,70,80,80,80,90 },
                                 { 20,20,20,20,10,10,20,10,10,20,10 }},
                                {{ 20,20,20,10,10,10,10,10, 0, 0, 0 }, /* c */
                                 { 30,30,30,30,30,20,20,10,20,20,10 },
                                 { 50,50,50,60,60,70,70,80,80,80,90 }}};

// ======================================================================
GGSize::GGSize(short s, float orb, float p_mass, int n) :
	SimpleSize(s, orb, p_mass, n)
{
	Generate();
}

//GGSize::GGSize(char c, float orb, float p_mass, int n) :
//	SimpleSize(c, orb, p_mass, n)
//{
//	Generate();
//}

void
GGSize::Generate(void)
{
int i,j;

	// uwp equiv.
	if(get_t_hex_val() == SIZE_LARGE_GG)
		{
		i = dice->Roll(6,3);
		if(i > 7) i += 7;
		else i += 8;
		}
	else                       /* small gg */
		{
		i = dice->Roll(6,2);
		if(i > 9) i -= 2;
		else if (i > 6) i--;
		}
	uwp_size = i * 10;

	// density
	i = dice->Roll(6,3);
	if(i < 8) j = i + 7;
	else if(i < 12) j = i * 2;
	else if(i < 14) j = i + 11;
	else j = i + 12;

	density = j / 100.0;
	mass = density * pow((uwp_size/8), 3);
}

// ======================================================================
WorldSize::WorldSize(short s, float orb, float p_mass, 
		int pre_stress, int atmos, int n, int zone_flag) :
	SimpleSize(s, orb, p_mass, n)
{
	CalcGravity();
	Generate(p_mass, pre_stress, atmos, zone_flag);
}

//WorldSize::WorldSize(char c, float orb, float p_mass, 
//		int pre_stress, int atmos, int n, int zone_flag) :
//	SimpleSize(c, orb, p_mass, n)
//{
//	CalcGravity();
//	Generate(p_mass, pre_stress, atmos, zone_flag);
//}

void
WorldSize::CalcGravity(void)
{
float f;

	if(get_t_hex_val() == SIZE_SMALL)
		f = .6;
	else
		f = get_t_hex_val();
	mass = pow((f / 8), 3) * density;
	gravity = density * f / 8;
}

void
WorldSize::Generate(float p_mass, int pre_stress, int atmos, int zone_flag)
{
int i,j,k,size;
long l,m;
float f,orbit;

	// diameter
	size = get_t_hex_val();
	l = size * 1000l;
	if(size < SIZE_SMALL)
		{
		i = 3;
		m = 100;
		}
	else
		{
		l = 600;
		i = 2;
		m = 10;
		}

	for(j = 0;j < i;j++)
		{
		l += (long)dice->Roll(10, 1, -6) * m;
		m /= 10;
		}
	diameter = (l * 8) / 5;

	// core
	i = 0;
	if(size < 4)
		i++;
	if(size > 5)
		i -= 2;
	if(atmos < 4)
		i++;
	if(atmos > 5)
		i -=2;
	if(zone_flag > HAB_HABIT)
		i += 6;
	i = dice->Roll(6,2,i);
	if(i < 2)
		{
		core = WC_HEAVY;
    	j = 110;
    	k = 5;
		}
	else if(i < 11)
		{
		core = WC_MOLTEN;
    	j = 82;
    	k = 2;
		}
	else if(i < 15)
		{
		core = WC_ROCKY;
    	j = 50;
    	k = 2;
		}
	else
		{
		core = WC_ICY;
    	j = 18;
    	k = 2;
		}

	i = dice->Roll(6, 3, -3);
	density = (((i * k) + j) / 100.0);

	// rot period
	orbit = orb->GetAUOrbit();
	i = dice->Roll(6, 2, -2);
	f = (float)((i * 4) + 5) + (p_mass / orbit);
	if(f > 40.0)
		f = SpecialRot(f);

	rot_period = f;

	// tilt
	i = dice->Roll(6,1,-2);
	j = dice->Roll(6,1);
	k = dice->Roll(6,2);

	if(k < 12)
		tilt =(((k - 2) / 2 ) + i);
	else
		tilt =(30 + (j * 10) + i);

	// stress
	i = pre_stress + dice->Roll(6, 1, -3);
	if(core == WC_HEAVY)
		i += dice->Roll(6, 1, -2);
	else if(core == WC_MOLTEN)
		i += dice->Roll(6, 1, -3);
	if(i < 0)
		stress = 0;
	else
		stress = i;

	if(get_t_hex_val() == SIZE_SMALL)
		f = .6;
	else
		f = get_t_hex_val();
	mass = pow((f / 8), 3) * density;
	gravity = density * f / 8;
}

float
WorldSize::SpecialRot(float rot)
{
int i,j,k;

	i = dice->Roll(10,1);
	j = dice->Roll(6,1) * 24;
	k = dice->Roll(60,1, -1);
	
	switch(dice->Roll(6,2))
		{
         case 2:
         	return(-((j * 10.0) + i + (k / 60.0)));
         case 3:
         	return(((j * 20.0) + i + (k / 60.0)));
         case 4:
         case 10:
         	return(((j * 10.0) + i + (k / 60.0)));
         case 6:
         case 7:
         case 8:
         	return(0.0);
         case 11:
         	return(((j * 50.0) + i + (k / 60.0)));
         case 12:
         	return(-((j * 50.0) + i + (k / 60.0)));
         default:
         	return(rot);
        }
}

char *
WorldSize::FormatRotPeriod(char *buff)
{
	return(PeriodFormat(buff, rot_period));
}

char *
WorldSize::PeriodFormat(char *dest, float per)
{
double f,g,h;

	if(per == 0.0)
		sprintf(dest, "Tidally Locked");
	else
		{
		if(per < 0.0)
			h = fabs(per);
		else
			h = per;
			
		if(h > 24.0)
			{
			f = modf((h / 24.0), &g) * 24.0;
			sprintf(dest, "%.0f days %.3f hours ", g, f);
			}
		else
			{
			f = modf(h, &g);
			sprintf(dest, "%.0f hours %.3f minutes ", g, f * 60);
			}
		if(per < 0.0)
			strcat(dest, "Retrograde ");
		}

	return(dest);
}

char *WorldSize::cores[4] = {"Heavy","Molten","Rocky","Icy"};

// ======================================================================
SatSize::SatSize(short s, float orb, float p_mass, int pre_stress, 
				int atmos, int n, int zone_flag, int par_size, float par_mass) :
	WorldSize(s, orb, p_mass, pre_stress, atmos, n, zone_flag)
{
	Init((int) orb, par_size);
	Generate(par_mass);
}

//SatSize::SatSize(char c, float orb, float p_mass, int pre_stress, 
//					int atmos, int n, int zone_flag, int p_size) :
//	WorldSize(c, orb, p_mass, pre_stress, atmos, n, zone_flag)
//{
//	Init((int) orb, p_size);
//	Generate(p_mass);
//}

void
SatSize::Init(int sorb, int par_size)
{
	if(par_size == 11)
		km_orbit = 600.0;
	else
		km_orbit = par_size * 1000;
	km_orbit *= ((8.0 * sorb) / 5.0);

	sat_orbit = sorb;
}

void
SatSize::Generate(float par_mass)
{
// FIXME: bad period calc
	rot_period = ((dice->Roll(6, 2, -2) * 4) + 5) + 
			(par_mass / (km_orbit / 400000.0));
	if(rot_period > (40.0 * 24))
		rot_period = SpecialRot(rot_period);

	sat_period = sqrt(pow(km_orbit/400000.0, 3) * 763.64 / par_mass);
//fprintf(stderr, "\nkm:%f", km_orbit);
//fprintf(stderr, "\npar:%f", par_mass);
//fprintf(stderr, "\nrot:%f", rot_period);
//fprintf(stderr, "\nsat:%f", sat_period);
}

float
SatSize::GetEffPeriod(void)
{
float f,g;

	f = sat_period * 24.0;
	if(rot_period == 0.0) return(f);
	g = .001 * f;
	if((rot_period > (f - .05)) &&
		(rot_period < (f + .05)))
		return(0.0);
	else return(rot_period);
}

char *
SatSize::FormatEffPeriod(char *buff)
{
	return(PeriodFormat(buff, GetEffPeriod()));
}

char *
SatSize::FormatSatPeriod(char *buff)
{
	return(PeriodFormat(buff, sat_period * 24.0));
}

// ======================================================================
// ======================================================================
// ======================================================================

