/*
 * File:      	hardtimes.cpp
 * Purpose:   	apply hardtimes to sector(s)
 * Author:		Mark A. Nordstrand
 * Created:	
 * Updated:	
 * Copyright:	LGPL
Traveller is a registered trademark of Far Future Enterprises.
Portions based upon material Copyright 1977-2002 Far Future Enterprises.
 */

/* rcsid[] = "$RCSfile: hardtimes.cpp,v $ $Revision: 1.5 $ $Author: man $ $Date: 2002/06/05 04:51:07 $" */

#include <stdio.h>
#include <time.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <libgen.h>
#include "hardtimes.h"
#include "t_dice.h"

#define BUFFER_SIZE			256

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// misc table stuff taken from other program modified to be less error prone
static int isol[2][18]={
	{ -1,-1, 0, 1, 1, 0,-2,-2,-2,-1, 0, 1, 2, 2, 2, 1, 0,-1 },
	{ -1, 0, 1, 0,-1,-1,-1, 0, 1, 1, 2, 1, 1, 0,-1,-2,-2,-2 }
};

static int initgov[16]={
	0, 7, 1, 8, 2, 6, 7, 3, 4, 5,10,11, 9,11,12,12
};

static int thegov[17]={
	0, 2, 4, 7, 8, 9, 5, 1, 3,12,10,11,14, 0, 6,13,15
};

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// ugly globals
static FILE *log1=stdout;		// for now.......

// ============================================================================
// some common stuff....
static int 
variance(int v)
{
	if(v <= 2) 
		return(0);
	return((int)(((v - 2.0) / 2.0) - 0.5));
}

static float isolated(DetailSector *s, int x, int y)
{
int i,n=0,m=0;
HexData *hd;

	for(i = 0;i < 18;i++) {
		if((x + isol[0][i] <= 31 && x + isol[0][i] >= 0) &&
			(y + isol[1][i] <= 39 && y + isol[1][i] >= 0)) {
			if(((hd = s->GetHex(x + isol[0][i], y + isol[1][i])) != NULL) &&
					(hd->world != NULL)) {
				n++;
			}
		} else m++;
	}
	if(n > 0)
		return(0.0);
	if(m > 0)
		return(0.5);
	return(1.0);
}

static void depop_world(main_world *d_mw)
{
	fprintf(log1, "World depopulated (%de%d).\n", 
			d_mw->GetPopNdx(), d_mw->get_pop());
	d_mw->SetPop('0');
	d_mw->SetPopNdx(0);
	d_mw->SetGovt('0');
	d_mw->SetLaw('0');
	d_mw->SetTech('0');
}

static void open_log(char *dir)
{
char buffer[BUFFER_SIZE];

	sprintf(buffer, "%s/desc/sector.txt", dir);
	log1 = fopen(buffer, "a");
}

static void log_world_header(HT_AREA area, HT_WAR war, main_world *mw)
{
char buffer[20];

	fprintf(log1, "%s (%02d%02d) %s (%c):  ",
		mw->GetName(), mw->GetLocX(), mw->GetLocY(),
		mw->GetUWPStr(buffer), mw->GetZone());

	switch(war) {
		case HTW_PEACE:
			fprintf(log1, "Peace, ");
			break;
		case HTW_WAR:
			fprintf(log1, "War, ");
			break;
		case HTW_INTENSE:
			fprintf(log1, "Intense War, ");
			break;
		case HTW_BLACK:
			fprintf(log1, "Black War, ");
			break;
		default:
			fprintf(log1, "Unknown, ");
			break;
	}
	switch(area) {
		case HTA_SAFE:
			fprintf(log1, "Safe\n");
			break;
		case HTA_FRONTIER:
			fprintf(log1, "Frontier\n");
			break;
		case HTA_OUTLANDS:
			fprintf(log1, "Outlands\n");
			break;
		case HTA_WILDS:
			fprintf(log1, "Wilds\n");
			break;
		default:
			fprintf(log1, "Unknown\n");
			break;
	}
}

static int calc_port_var(int o, int n)
{
	if(n > 4)
		n = 5;
	return(n - o);
}

// ============================================================================
// the 'map' class
HardTimes::HardTimes(char *dir)
{
char buffer[BUFFER_SIZE];
int x,y,x1;
FILE *fp;

	for(x = 0;x < MAX_X_HEX;x++) {
		for(y = 0;y < MAX_Y_HEX;y++) {
			area[x][y] = HTA_SAFE;
			war[x][y] = HTW_PEACE;
		}
	}

	sprintf(buffer, "%s/%s.ht", dir, basename(dir));
	if((fp = fopen(buffer, "r")) != NULL) {
		// read area map
		y = 0;
		while(fgets(buffer, BUFFER_SIZE, fp) != NULL) {
			if(strlen(buffer) > MAX_X_HEX)
				x1 = MAX_X_HEX;
			else
				x1 = strlen(buffer);
			for(x = 0;x < x1;x++) {
				switch(buffer[x]) {
					default:
					case 'S':
					case 's':
						area[x][y] = HTA_SAFE;
						break;
					case 'F':
					case 'f':
						area[x][y] = HTA_FRONTIER;
						break;
					case 'O':
					case 'o':
						area[x][y] = HTA_OUTLANDS;
						break;
					case 'W':
					case 'w':
						area[x][y] = HTA_WILDS;
						break;
				}
			}
			y++;
			if(y >= MAX_Y_HEX)
				break;
		}
		// read war map
		y = 0;
		while(fgets(buffer, BUFFER_SIZE, fp) != NULL) {
			if(strlen(buffer) > MAX_X_HEX)
				x1 = MAX_X_HEX;
			else
				x1 = strlen(buffer);
			for(x = 0;x < x1;x++) {
				switch(buffer[x]) {
					default:
					case 'P':
					case 'p':
						war[x][y] = HTW_PEACE;
						break;
					case 'W':
					case 'w':
						war[x][y] = HTW_WAR;
						break;
					case 'I':
					case 'i':
						war[x][y] = HTW_INTENSE;
						break;
					case 'B':
					case 'b':
						war[x][y] = HTW_BLACK;
						break;
				}
			}
			y++;
			if(y >= MAX_Y_HEX)
				break;
		}
	}
}

#if 0
void
HardTimes::Dump(void)
{
int x,y;

	for(y = 0;y < MAX_Y_HEX;y++) {
		for(x = 0;x < MAX_X_HEX;x++) {
			switch(area[x][y]) {
				case HTA_SAFE:
					fprintf(stderr, "S");
					break;
				case HTA_FRONTIER:
					fprintf(stderr, "F");
					break;
				case HTA_OUTLANDS:
					fprintf(stderr, "O");
					break;
				case HTA_WILDS:
					fprintf(stderr, "W");
					break;
			}
		}
		fprintf(stderr, "\n");
	}
	for(y = 0;y < MAX_Y_HEX;y++) {
		for(x = 0;x < MAX_X_HEX;x++) {
			switch(war[x][y]) {
				case HTW_PEACE:
					fprintf(stderr, "P");
					break;
				case HTW_WAR:
					fprintf(stderr, "W");
					break;
				case HTW_INTENSE:
					fprintf(stderr, "I");
					break;
				case HTW_BLACK:
					fprintf(stderr, "B");
					break;
			}
		}
		fprintf(stderr, "\n");
	}
//exit(0);
}
#endif

HT_AREA 
HardTimes::GetArea(int x, int y)
{
	if((x < 0) || (x >= MAX_X_HEX) || (y < 0) || (y >= MAX_Y_HEX)) 
		return(HTA_SAFE);
	return(area[x][y]);
}

HT_WAR 
HardTimes::GetWar(int x, int y)
{
	if((x < 0) || (x >= MAX_X_HEX) || (y < 0) || (y >= MAX_Y_HEX)) 
		return(HTW_PEACE);
	return(war[x][y]);
}

// ============================================================================
void
full_depop_world(main_world *d_mw)
{
	fprintf(log1, 
		" World cannot support Life.  Social statistics reduced to 0.\n");
	fprintf(log1, "   Port (%c) reduced to X.  ", d_mw->get_port());
	d_mw->SetPort('X');
	d_mw->SetPop('0');
	d_mw->SetGovt('0');
	d_mw->SetLaw('0');
	d_mw->SetTech('0');
	d_mw->SetPopNdx(0);
	if(d_mw->GetBase() != ' ') {
		fprintf(log1, "Base (%c) removed.", d_mw->GetBase());
		d_mw->SetBase(' ');
	}
	fprintf(log1, "\n");
}

// ----------------------------------------------------------------------------
// actual workers
bool
hard_times(HT_AREA area, HT_WAR war, float iso_mod, main_world *d_mw)
{
bool gone=FALSE,balk=FALSE,isol=FALSE;
bool need_nl=FALSE;
char buffer[20];
int dm=0,dm2=0,biodam=0,pop_red=0,tech_red=0,xeno=0,env_dm=0;
// XXX since the uwp class has ports 'ABCDEFGXY' instead of just 'ABCDEX'
//  these are handy replacements
int port_ndx,port_var=0,o_tech;
Dice d;

	if((HTA_SAFE == area) && (HTW_PEACE == war))
		// nothing to do
		return(FALSE);

	log_world_header(area, war, d_mw);

	if(d_mw->get_port() > 'E')
		port_ndx = 5;
	else
		port_ndx = d_mw->get_port() - 'A';
	o_tech = d_mw->get_tech_val();

	// biosphere damage
	if((dm += (int) war) > 1) {
		if(d_mw->get_port() == 'A') dm++;
		if(d_mw->get_pop() >= '9') dm++;

		if(war > HTW_INTENSE) dm2 = 2;
   
		biodam = d.Roll(6, 2) + dm;
		if(biodam > 13)
			biodam = d.Roll(6, 2) + dm2;
		else
			biodam = 0;
	}

	if(biodam > 0) {
		bool taint=FALSE;

		fprintf(log1, " Biosphere damage has occured:\n");
		fprintf(log1, "  Starport %c destroyed. ", d_mw->get_port());
		port_var = calc_port_var(d_mw->get_port_val(), 8);
		d_mw->SetPort('X');
		if(biodam < 4)
			fprintf(log1, "Temporary temperature decrease.");
		else if(biodam < 6) {
			dm2 = d.Roll(6, 1) + 6;
			fprintf(log1, "Permanent temperature decrease of %dC", dm2);
		} else if(biodam < 11) {
			if(d_mw->get_atmos() == '5') {
				d_mw->SetAtmos('4');
				taint = TRUE;
			}
			if(d_mw->get_atmos() == '6') {
				d_mw->SetAtmos('7');
				taint = TRUE;
			}
			if(d_mw->get_atmos() == '8') {
				d_mw->SetAtmos('9');
				taint = TRUE;
			}
			if(taint)
				fprintf(log1, "Atmosphere accuires a taint. ");

			if(biodam > 8) {
				fprintf(log1, "Widespread destruction.");
				pop_red = 1;
				tech_red = 3;
			} 
			fprintf(log1, "\n");
		} else if(biodam < 13) {
			if(d_mw->get_atmos() != '0') {
				fprintf(log1, "Atmosphere becomes poisoned. ");
				d_mw->SetAtmos('C');
			}
			fprintf(log1, "Extensive devastation.");
			pop_red = 2;
			tech_red = 6;
		} else {
			if(d_mw->get_atmos() != '0') {
				fprintf(log1, "Atmosphere becomes poisoned. ");
				d_mw->SetAtmos('C');
			}
			depop_world(d_mw);
			gone = TRUE;
		}
		fprintf(log1, "\n");
	}

	// starport reduction
	dm = d.Roll(6, 1);
	dm2 = 0;

	if(port_ndx < 4) {
		switch(area) {
			case HTA_SAFE:
				break;
			case HTA_FRONTIER:
				if(0 == port_ndx)
					dm += 2;
				break;
			case HTA_OUTLANDS:
				dm += (port_ndx - 3);
				break;
			case HTA_WILDS:
				if(port_ndx < 2)
					dm += 3;
				else if(port_ndx < 3)
					dm += 2;
				else
					dm++;
				break;
		}
		switch(war) {
			case HTW_PEACE:
				break;
			case HTW_WAR:
				if(port_ndx < 3)
					dm++;
				break;
			case HTW_INTENSE:
				if(port_ndx < 2)
					dm += 2;
				if(port_ndx < 3)
					dm++;
				break;
			case HTW_BLACK:
				if(port_ndx < 1)
					dm += 3;
				if(port_ndx < 3)
					dm += 2;
				else
					dm++;
				break;
		}
		switch(port_ndx) {
			case 0:
				dm += (int)(iso_mod * 2.0);
				if(d_mw->get_pop_val() < 2)
					dm += 2;
				else if(d_mw->get_pop_val() < 3)
					dm++;
				if(d_mw->get_tech_val() < 5)
					dm += 8;
				else if(d_mw->get_tech_val() < 7)
					dm += 5;
				else if(d_mw->get_tech_val() < 8)
					dm += 3;
				else if(d_mw->get_tech_val() < 11)
					dm++;
				break;
			case 1:
				dm += (int)(iso_mod * 3.0);
				if(d_mw->get_pop_val() < 2)
					dm += 2;
				else if(d_mw->get_pop_val() < 3)
					dm++;
				if(d_mw->get_tech_val() < 5)
					dm += 7;
				else if(d_mw->get_tech_val() < 7)
					dm += 4;
				else if(d_mw->get_tech_val() < 8)
					dm++;
				break;
			case 2:
				dm += (int)(iso_mod * 4.0);
				if(d_mw->get_pop_val() < 2)
					dm++;
				if(d_mw->get_tech_val() < 5)
					dm += 5;
				else if(d_mw->get_tech_val() < 7)
					dm += 3;
				break;
			case 3:
				dm += (int)(iso_mod * 1.0);
				if(d_mw->get_tech_val() < 5)
					dm += 3;
				else if(d_mw->get_tech_val() < 7)
					dm++;
				break;
		}

		dm2 = d_mw->get_port_val() + variance(dm);
		// XXX again, port has more values
		if(dm2 > 4)
			dm2 = 8;

		if(dm2 > d_mw->get_port_val()) {
			fprintf(log1, " Starport reduced %c ", d_mw->get_port());
			port_var = calc_port_var(d_mw->get_port_val(), dm2);
			d_mw->SetPortVal(dm2);
			fprintf(log1, "-> %c (-%d).", d_mw->get_port(), variance(dm));
			need_nl = TRUE;
			if(dm2 > 4)
				port_ndx = 5;
			else
				port_ndx = dm2;
		}
	}

	// base destruction 
	if(d_mw->GetBase() != ' ') {
		if((d.Roll(6, 1) > 3) && ((port_var > 1) || 
				(d_mw->get_port_val() > 4) || (area = HTA_WILDS))) {
			fprintf(log1, " Bases destroyed (%c).", d_mw->GetBase());
			d_mw->SetBase(' ');
			need_nl = TRUE;
		}
	}

	// recession in planetary economies
	dm = d.Roll(6, 1);
	dm2 = 0;

	dm += port_ndx;
	if(HTA_WILDS == area)
		dm += 3;
	if(HTA_OUTLANDS == area)
		dm += 2;
	if(d_mw->get_atmos_val() == 12)
		dm += 2;
	else if((d_mw->get_atmos_val() < 4) || (d_mw->get_atmos_val() > 9))
		dm++;
	if((d_mw->get_hydro_val() < 2) || (d_mw->get_hydro_val() > 9))
		dm++;
	if((d_mw->get_atmos_val() > 9) && (d_mw->get_hydro_val() > 0))
		dm++;
	if(d_mw->get_pop_val() < 5)
		dm += 4;
	else if(d_mw->get_pop_val() < 6)
		dm += 2;
	else if(d_mw->get_pop_val() < 7)
		dm++;
	if((d_mw->get_govt_val() == 5) || (d_mw->get_govt_val() == 6))
		dm--;
	else if((d_mw->get_govt_val() == 0) || (d_mw->get_govt_val() == 2) ||
		(d_mw->get_govt_val() == 3) || (d_mw->get_govt_val() == 7) ||
		(d_mw->get_govt_val() > 10))
		dm++;
	if(d_mw->get_govt_val() > 17)
		dm += 9;
	else if(d_mw->get_govt_val() > 15)
		dm += 7;
	else if(d_mw->get_govt_val() > 13)
		dm += 5;
	else if(d_mw->get_govt_val() > 11)
		dm += 3;
	else if(d_mw->get_govt_val() > 9)
		dm += 1;
	else if(d_mw->get_govt_val() == 1)
		dm -= 10;
	else if(d_mw->get_govt_val() == 2)
		dm -= 8;
	else if(d_mw->get_govt_val() == 3)
		dm -= 6;
	else if((d_mw->get_govt_val() == 4) || (d_mw->get_govt_val() == 5))
		dm -= 4;
	else if((d_mw->get_govt_val() == 6) || (d_mw->get_govt_val() == 7))
		dm -= 2;

	dm2 = variance(dm) + tech_red;
	tech_red = dm2;

	if(dm2 != 0) {
		fprintf(log1, " TL reduced %c -> ", d_mw->get_tech());
		if((d_mw->get_tech_val() - dm2) < 0)
			d_mw->SetTechVal(0);
		else
			d_mw->SetTechVal((d_mw->get_tech_val() - dm2));

		fprintf(log1, " %c (-%d).", d_mw->get_tech(), dm2);
		need_nl = TRUE;
	}

	// do carriage control
	if(need_nl) 
		fprintf(log1, "\n");
	need_nl = FALSE;

	// xenophobia and isolationism
	if(!gone) {
		// xenophobia
		dm = d.Roll(3, 1);
		dm2 = 0;

		dm += port_var;
		if(biodam)
			dm += 6;
		if(HTA_WILDS == area)
			dm += 3;
		else if(HTA_OUTLANDS == area)
			dm++;
		if(d_mw->get_atmos_val() < 2)
			dm++;
		if(d_mw->get_hydro_val() < 2)
			dm++;
		if((d_mw->get_atmos_val() > 9) && (d_mw->get_hydro_val() > 0))
			dm++;
		if(d_mw->get_pop_val() < 3)
			dm--;
		else if(d_mw->get_pop_val() > 8)
			dm += 2;
		else if(d_mw->get_pop_val() > 5)
			dm++;
		if(d_mw->get_govt_val() > 10)
			dm++;

		xeno = variance(dm);

		if(xeno) {
			// isolationism
			dm = 0;
			if(port_ndx < 1)
				dm -= 5;
			else if(port_ndx < 2)
				dm -= 4;
			else if(port_ndx < 3)
				dm -= 2;
			else if(port_ndx < 4)
				dm--;
			if((d_mw->get_atmos_val() == 5) || (d_mw->get_atmos_val() == 6) ||
					(d_mw->get_atmos_val() == 8))
				dm++;

			if(!((d_mw->get_atmos_val() > 9) && (d_mw->get_hydro_val() > 2)))
				dm++;
			if(d_mw->get_tech_val() < 5)
				dm += 7;
			else if(d_mw->get_tech_val() < 6)
				dm += 3;
			else if(d_mw->get_tech_val() < 7)
				dm += 2;
			else if(d_mw->get_tech_val() < 8)
				dm += 1;

			if((d.Roll(6, 2) + dm) > 10)
				isol = TRUE;
		}

		// doomed/failing
		dm = 0;
		dm2 = 0;

		if(d_mw->get_tech_val() < 9) {
			int o_atm,o_hyd;

			o_atm = d_mw->get_atmos_val();
			o_hyd = d_mw->get_hydro_val();
			switch(d_mw->get_tech_val()) {
				case 0:
				case 1:
				case 2:
					if((4 == o_atm) || (7 == o_atm) || (8 == o_atm))
						dm += 3;
					else if((o_atm < 4) || (9 == o_atm) || 
								(10 == o_atm) || (11 == o_atm))
						dm += 5;
					if(o_hyd < 1)
						dm += 5;
					else if(o_hyd < 2)
						dm += 3;
					else if((o_hyd < 3) || (o_hyd >9))
						dm += 2;
					break;
				case 3:
				case 4:
					if((4 == o_atm) || (7 == o_atm) || (8 == o_atm))
						dm += 1;
					else if((o_atm < 4) || (9 == o_atm) || (10 == o_atm))
						dm += 2;
					else if(11 == o_atm)
						dm += 4;
					if(o_hyd < 1)
						dm += 4;
					else if(o_hyd < 2)
						dm += 2;
					else if((o_hyd < 3) || (o_hyd >9))
						dm += 1;
					break;
				case 5:
				case 6:
					if((o_atm < 4) || (9 == o_atm) || (10 == o_atm))
						dm += 1;
					else if(11 == o_atm)
						dm += 2;
					if(o_hyd < 1)
						dm += 2;
					else if(o_hyd < 2)
						dm += 1;
					break;
					break;
				case 7:
				case 8:
					if(11 == o_atm)
						dm += 1;
					break;
			}
			if((d_mw->get_atmos_val() > 9) && (d_mw->get_hydro_val() > 0)) {
				if(d_mw->get_tech_val() < 3)
					dm += 5;
				else if(d_mw->get_tech_val() < 4)
					dm += 3;
				else if(d_mw->get_tech_val() < 5)
					dm += 2;
				else if(d_mw->get_tech_val() < 6)
					dm += 1;
			}
		}

		if(d_mw->get_pop_val() < 3) {
			dm -= 3;
			if(war > HTW_WAR)
				dm++;
		} else if(d_mw->get_pop_val() < 6) {
			dm -= 2;
			dm += (int) war;
		} else if(d_mw->get_pop_val() > 8) {
			dm++;
			if(war > HTW_INTENSE)
				dm += 3;
			else if(war > HTW_PEACE)
				dm++;
		} else {
			if(war > HTW_INTENSE)
				dm += 5;
			else if(war > HTW_WAR)
				dm += 3;
			else if(war > HTW_PEACE)
				dm += 2;
		}
		env_dm = variance(dm);
		dm += d.Roll(3, 1);
		dm2 = variance(dm) + pop_red;
		pop_red = dm2;

		if(dm2) {
			int pop,popmul;

			if(env_dm > 4)
				fprintf(log1, " Doomed world.");
			else if(env_dm > 1)
				fprintf(log1, " Failing world.");
			pop = d_mw->get_pop_val();
			popmul = d_mw->GetPopNdx();
			fprintf(log1, " Population reduced %de%d -> ", popmul, pop);
			while((popmul = popmul - dm2) <= 0) {
				pop--;
				dm2 = 0 - popmul;
				popmul = 9;
			}

			if(pop < 1) {
				gone = TRUE;
				depop_world(d_mw);
				d_mw->SetBase(' ');
			} else {
				d_mw->SetPopVal(pop);
				d_mw->SetPopNdx(popmul);
			}

			fprintf(log1, "%de%d (-%d).\n", d_mw->GetPopNdx(),
					d_mw->get_pop_val(), pop_red);
		}
	}

	// government change
	if(!gone && (pop_red || tech_red)) {
		int i,gov,law;

		dm = d.Roll(6, 1);
		dm2 = 0;

		if(o_tech - d_mw->get_tech_val() - 2 > 0)
			dm += o_tech - d_mw->get_tech_val() - 2;
		if(pop_red > 1)
			dm += pop_red - 1;

		i = initgov[d_mw->get_tech_val()] + variance(dm);
		if(i > 13)
			dm2 = 13;
		else
			dm2 = i;

		if(dm2 == 7) {
			if(d.Roll(6, 1) > 3)
				dm2 = 7;
			else
				dm2 = 14;
		} else if(dm2 == 11) {
			if(d.Roll(6, 1) > 3)
				dm2 = 11;
			else
				dm2 = 15;
		} else if(dm2 == 12) {
			if(d.Roll(6, 1) > 3)
				dm2 = 12;
			else
				dm2 = 16;
		}

		gov = thegov[dm2];

		if(7 == gov)
			balk = TRUE;

		if(gov != d_mw->get_govt_val()) {
			char buff[80];

			fprintf(log1, " Government: %c [%s] -> ",
				d_mw->get_govt(), d_mw->GetGovtDesc(buff));
			d_mw->SetGovtVal(gov);
			fprintf(log1, "%c [%s].\n",
				d_mw->get_govt(), d_mw->GetGovtDesc(buff));
		}

		// law level change
		law = d.Roll(6, 2) - 7 + d_mw->get_govt_val();
		if(law < 0)
			law = 0;

		if((law != d_mw->get_govt_val()) || (xeno)) {
			fprintf(log1, " Lawlevel change: %c -> ", d_mw->get_law());
			d_mw->SetLawVal(law);
			fprintf(log1, "%c", d_mw->get_law());
			if(xeno) {
				fprintf(log1, 
					" [Xenophobia: +%d]", xeno);
				if(isol) {
					fprintf(log1, " Isolationist");
				} 
			}
			fprintf(log1, ".\n");
		}
	}

	// change alleg....
	if(HTA_WILDS == area)
		d_mw->SetAlleg("Wi");

	fprintf(log1, "  New UWP: %s\n", d_mw->GetUWPStr(buffer));

	return(TRUE);
}

bool
collapse(HT_AREA area, HT_WAR war, float iso_mod, main_world *d_mw)
{
bool gone=FALSE;
char o_base;
int msp=10,tmp;
int o_port,o_pop,o_govt,o_law,o_tech,o_pop_ndx;
Dice dice;

	if((HTA_SAFE == area) && (HTW_PEACE == war))
		// nothing to do
		return(FALSE);

	log_world_header(area, war, d_mw);

	o_port = d_mw->get_port_val();
	o_pop = d_mw->get_pop_val();
	o_govt = d_mw->get_govt_val();
	o_law = d_mw->get_law_val();
	o_tech = d_mw->get_tech_val();
	o_pop_ndx = d_mw->GetPopNdx();
	o_base = d_mw->GetBase();

	// max. sus. pop
	if(d_mw->get_size_val() < 5)
		msp--;
	else if(d_mw->get_size_val() < 8)
		msp -= 2;
	if((d_mw->get_atmos_val() == 5) || (d_mw->get_atmos_val() == 7) ||
			(d_mw->get_atmos_val() == 9))
		msp--;
	if(d_mw->get_atmos_val() == 4)
		msp -= 2;
	if(d_mw->get_hydro_val() == 0)
		msp -= 2;
	else if((d_mw->get_hydro_val() < 3) || (d_mw->get_hydro_val() > 9))
		msp--;
	if((d_mw->get_atmos_val() < 4) || (d_mw->get_atmos_val() > 9))
		msp = 0;

	if(msp < 1) {
		full_depop_world(d_mw);
		gone = TRUE;
	} else if(msp < o_pop) {
		tmp = dice.Roll(9, 1);
		fprintf(log1, " Population decline: %de%d -> %de%d\n", 
				o_pop_ndx, o_pop, tmp, msp);
		d_mw->SetPopVal(msp);
		d_mw->SetPopNdx(tmp);

	}

	if(!gone) {
		// tech decline
		int dm,dm2,t_pop,t_pop_ndx;

		tmp = d_mw->get_tech_val();
		if(tmp < 9)
			dm = dice.Roll(6, 1) - 3;
		else if(tmp < 11)
			dm = dice.Roll(6, 1);
		else if(tmp < 14)
			dm = dice.Roll(6, 2);
		else 
			dm = dice.Roll(6, 3);
		if(dm < 0)
			dm = 0;

		if(dm > 0) {
			fprintf(log1, " Tech Level decline %c -> ", d_mw->get_tech());
			d_mw->SetTechVal(tmp - dm);
			fprintf(log1, "%c.\n", d_mw->get_tech());
		}

		// actual pop
		dm2 = (int)(((float)dm / 4.0) + 0.5);
		if(dm2) {
			t_pop = o_pop;
			t_pop_ndx = o_pop_ndx;
			while((t_pop_ndx = (t_pop_ndx - dm2)) <= 0) {
				t_pop--;
				dm = -t_pop_ndx;
				t_pop_ndx = 9;
			}

			if((t_pop < 1) && (t_pop_ndx < 1)) {
				full_depop_world(d_mw);
				gone = TRUE;
			} else if((t_pop < d_mw->get_pop_val()) || 
				(t_pop_ndx < d_mw->GetPopNdx())) {
				fprintf(log1, "  Population decline %de%d -> %de%d.\n",
					d_mw->GetPopNdx(), d_mw->get_pop_val(), t_pop_ndx, t_pop);
				d_mw->SetPopNdx(t_pop_ndx);
				d_mw->SetPopVal(t_pop);
			}
		}
	}

	// low pop
	if((!gone) && (d_mw->get_pop_val() < 6) && (o_pop > 5)) {
		int t_pop_ndx,t_pop,t_tech;

		fprintf(log1, " Internal population decline.  ");
		t_pop = d_mw->get_pop_val();
		t_pop_ndx = d_mw->GetPopNdx() / 2;
		if(0 == t_pop_ndx) {
			t_pop--;
			t_pop_ndx = 5;
		}

		if(t_pop < 1) {
			full_depop_world(d_mw);
			gone = TRUE;
		} else {
			t_tech = d_mw->get_tech_val() - 1;
			if(t_tech < 0)
				t_tech = 0;
			else {
				fprintf(log1, "Tech: %c --> ", d_mw->get_tech());
				d_mw->SetTechVal(t_tech);
				fprintf(log1, "%c.  ", d_mw->get_tech());
			}
			fprintf(log1, "Pop: %de%d -> %de%d.\n", 
				d_mw->GetPopNdx(), d_mw->get_pop_val(), t_pop_ndx, t_pop);
			d_mw->SetPopNdx(t_pop_ndx);
			d_mw->SetPopVal(t_pop);
		}
	}

	// starport
	if(d_mw->get_port() != 'X') {
		bool destroy_base=FALSE;
		int tmp,t_port,tech_dec;

		tmp = dice.Roll(6, 1);
		tech_dec = o_tech - d_mw->get_tech_val();
		t_port = d_mw->get_port_val();
		if(tmp > tech_dec) {
			t_port++;
		}
		else if(tmp == tech_dec) {
			t_port += 2;
			destroy_base = TRUE;
		}
		else  {
			t_port = 8;
			destroy_base = TRUE;
		}
		if(t_port > 4) {
			t_port = 8;
			destroy_base = TRUE;
		}

		fprintf(log1, "  Starport reduced %c --> ", d_mw->get_port());
		d_mw->SetPortVal(t_port);
		fprintf(log1, "%c.  ", d_mw->get_port());
		if(d_mw->GetBase() != ' ') {
			if(!destroy_base) {
				switch(d_mw->GetBase()) {
					case 'S':
						if(dice.Roll(10, 1) < 8)
							destroy_base = TRUE;
						break;
					case 'W':
					case 'D':
						if(dice.Roll(10, 1) != 10)
							destroy_base = TRUE;
						break;
					default:			// navy
						if(dice.Roll(10, 1) < 9)
							destroy_base = TRUE;
						break;
				}
			}
			if(destroy_base) {
				fprintf(log1, "Base (%c) destroyed.", d_mw->GetBase());
				d_mw->SetBase(' ');
			}
		}
		fprintf(log1, "\n");
	}
	
	// balkinization
	// XXX since none of mtu supports an alternate govt type, this
	//  seems a bit pointless....

	return(TRUE);
}

// ----------------------------------------------------------------------------
void
process(HARD_TIMES_METHOD method, TradeTable *tt,
		DetailSector *b_sect, DetailSector *w_sect, DetailSector *d_sect, 
		HardTimes *ht)
{
HT_AREA area;
HT_WAR war;
char u[20];
int x, y;
float iso_mod;
HexData *b_hd,*w_hd,*d_hd;
main_world *b_mw,*w_mw,*d_mw;

	for(x = 0;x < MAX_X_HEX;x++) {
		for(y = 0;y < MAX_Y_HEX;y++) {
			b_mw = w_mw = d_mw = NULL;
			if((b_hd = b_sect->GetHex(x, y)) != NULL)
				b_mw = b_hd->world;
			if((w_hd = w_sect->GetHex(x, y)) != NULL)
				w_mw = w_hd->world;
			if((NULL == b_hd) && (NULL == w_hd))
				// nothing to do
				continue;

			// sanity check....
			if((NULL == b_mw) && (w_mw != NULL)) {
				fprintf(stderr, "Warning (%02d%02d): no baseline data using working data.\n", x, y);
				b_mw = w_mw;
				b_hd = w_hd;
			} else if((b_mw != NULL) && (NULL == w_mw)) {
				fprintf(stderr, "Warning (%02d%02d): no working data, using basline data.\n", x, y);
				w_mw = b_mw;
				w_hd = b_hd;
			}
			d_hd = d_sect->CopyHex(x, y, w_hd);
			if((NULL == b_mw) && (NULL == w_mw))
				continue;
			d_mw = d_hd->world;

			area = ht->GetArea(x, y);
			war = ht->GetWar(x, y);
			iso_mod = isolated(d_sect, x, y);
			// XXX RCS has other methods (partially implemented)
			//   for now, I only need hard times
			if(HTM_HARD_TIMES == method) {
				hard_times(area, war, iso_mod, d_mw);
			}
			else if(HTM_COLLAPSE == method) {
				collapse(area, war, iso_mod, d_mw);
			}
			d_mw->GetUWPStr(u);
			d_mw->SetTradeCodes(tt->CheckTrade(u));
		}
	}
}


// ============================================================================
// helpers
DetailSector *
create_ht_sector(TCodes *c, char *dir)
{
char buffer[BUFFER_SIZE];
DetailSector *s;

	s = new DetailSector(c);
	sprintf(buffer, "%s/%s.sec", dir, basename(dir));
	s->SetName(buffer);
	return(s);
}

DetailSector *
create_sector(TCodes *c, char *dir)
{
char buffer[BUFFER_SIZE];
DetailSector *s;

	s = new DetailSector(c);
	sprintf(buffer, "%s/%s.sec", dir, basename(dir));
	s->LoadFile(buffer);
	return(s);
}

// ============================================================================
// main program
int
main(int argc, char *argv[])
{
char *b_name=NULL,*w_name=NULL,*d_name=NULL;
int err=0,opt;
HARD_TIMES_METHOD method=HTM_HARD_TIMES;
time_t t;
struct tm *l_tm;
TCodes *codes=NULL;
DetailSector *b_sect=NULL,*w_sect=NULL,*d_sect=NULL;
HardTimes *ht;

	// read in args
	do {
		opt = getopt(argc, argv, "tlrhc:b:w:d:");
		switch(opt) {
			case '?':
			case 'h':
				err = 1;
				break;
			case 't':
				method = HTM_HARD_TIMES;
				break;
			case 'l':
				method = HTM_COLLAPSE;
				fprintf(stderr, "Warning: collapse not fully imlemented yet.\n\n");
				break;
			case 'r':
				method = HTM_REBUILD;
				fprintf(stderr, "Warning: rebuild not implemented yet.\n\n");
				break;
			case 'c':
				codes = new TCodes(optarg);
				break;
			case 'b':
				b_name = new char[strlen(optarg) + 1];
				strcpy(b_name, optarg);
				break;
			case 'w':
				w_name = new char[strlen(optarg) + 1];
				strcpy(w_name, optarg);
				break;
			case 'd':
				d_name = new char[strlen(optarg) + 1];
				strcpy(d_name, optarg);
				break;
		}
	} while(opt > 0);

	// verify names
	if(0 == err) {
		if(NULL == b_name) {
			fprintf(stderr, "Error:  No base name given.\n");
			err = 2;
		}
		if(NULL == w_name) {
			fprintf(stderr, "Error:  No working name given.\n");
			err = 3;
		}
		if(NULL == d_name) {
			fprintf(stderr, "Error:  No destination name given.\n");
			err = 4;
		}
		if(NULL == codes) {
			fprintf(stderr, "Error:  No codes directory given.\n");
			err = 5;
		}
	}

	// load things up
	if(0 == err) {
		b_sect = create_sector(codes, b_name);
		w_sect = create_sector(codes, w_name);
		d_sect = create_ht_sector(codes, d_name);
		open_log(d_name);
		ht = new HardTimes(d_name);
		process(method, codes->GetTradeTable(), b_sect, w_sect, d_sect, ht);
		d_sect->WriteSec();
		t = time(&t);
		l_tm = localtime(&t);
		fprintf(log1, "\nHard Times applied on %s\n", asctime(l_tm));
	// if any config errors, give some help
	} else if(err) {
		fprintf(stderr, "%s: Apply Hard Times to a sector.\n", argv[0]);
		fprintf(stderr, "\nUsage:\n");
		fprintf(stderr, 
			"%s [-tlr] -c<codes> -b<base> -w<working> -d<destination>\n", 
			argv[0]);
		fprintf(stderr, "\nWhere:\n");
		fprintf(stderr, "  -t apply Hard Times rules (default)\n");
		fprintf(stderr, "  -l apply Collapse rules\n");
		fprintf(stderr, "  -r apply Rebuild rules\n");
		fprintf(stderr, "  <codes> is the codes directory\n");
		fprintf(stderr, "  <base> is the baseline (pre-rebellion) sector\n");
		fprintf(stderr, "  <working> is the sector to work on\n");
		fprintf(stderr, "  <destination> is the sector for output\n\n");
		fprintf(stderr, "'sector' is actually the mtu dir holding the sector information.\n");
		fprintf(stderr, "The only possible modifications will be to the UWP, bases, trade codes,\n");
		fprintf(stderr, "p of pbg, and if wilds, the allegence.  All else will be copied verbatam\n");
		fprintf(stderr, "to destination.  The <dir>/desc/sector.txt will contain a report of changes\n");
		fprintf(stderr, "made.\n");
	}

	// we're outa' here
	return(err);
}

