//
// File:		legend.cpp
// Purpose:		handle drawing of the sector legend
// Author:		Mark A. Nordstrand
// Created:
// Updated:	
// Copyright:	LGPL
// Traveller is a registered trademark of Far Future Enterprises.
// Portions based upon material Copyright 1977-2002 Far Future Enterprises.
//
// rcsid[] = "$RCSfile: legend.cpp,v $ $Revision: 1.3 $ $Author: man $ $Date: 2002/10/04 02:44:36 $"

#include "legend.h"

#define LEGEND_BORDER_OFFSET		5
#define LEGEND_INNER				4

#define LEGEND_X_OFFSET				10
#define LEGEND_Y_OFFSET				10
#define MIN_LEGEND_MAP_X			40
#define MIN_LEGEND_MAP_Y			60

#define LEGEND_TEXT_Y_INCR			13
#define LEGEND_TEXT_X_INCR			13
#define LEGEND_TEXT_OFFSET			10
#define LEGEND_LINE_LENGTH			10

static wxFont *RealBigFont=NULL;
static wxFont *LegendFont=NULL;
static wxFont *LegendBoldFont=NULL;

// helpers
static int line_pos[MAX_LAYOUT_Y][MAX_LAYOUT_X][2] = {
	{ { 80,  15 }, { 120,  20 }, { 190,  15 } },
	{ { 80,  55 }, {  80,  35 }, { 190,  35 } },
	{ { 80,  75 }, { 190,  55 }, { 190,  75 } },
	{ { 80,  95 }, { 190, 115 }, { 190,  95 } },
	{ { 80, 115 }, {  80, 135 }, { 190, 135 } }
};

static int text_pos[MAX_LAYOUT_Y][MAX_LAYOUT_X][2] = {
	{ { 10,  10 }, { 110,  10 }, { 193,  10 } },
	{ { 10,  50 }, {  10,  30 }, { 193,  30 } },
	{ { 10,  70 }, { 193,  50 }, { 193,  70 } },
	{ { 10,  90 }, { 193, 110 }, { 193,  90 } },
	{ { 10, 110 }, {  10, 130 }, { 193, 130 } }
};

static int text_offset[MAX_HEX_DISPLAY] = {
	38,		// HD_NONE=0,
	0,		// HD_NUMBER,
	0,		// HD_NAME,
	0,		// HD_NAME_POP,
	40,		// HD_UWP,
	40,		// HD_UWP_NO_PORT,
	40,		// HD_UWP_FILTERED,
	0,		// HD_TRADE_CODES,
	45,		// HD_PORT,
	45,		// HD_SIZE,
	35,		// HD_ATMOS,
	35,		// HD_HYDRO,
	45,		// HD_POP,
	40,		// HD_GOVT,
	45,		// HD_LAW,
	40,		// HD_TECH,
	7,		// HD_BASE_CODE,
	40,		// HD_ALLEG,
	23,		// HD_GG_NUM,
	16,		// HD_BELTS_NUM,
	10,		// HD_POP_NDX,
	15,		// HD_NUM_STAR,
	28,		// HD_BASE_BM1,
	28,		// HD_BASE_BM2,
	50,		// HD_GG_BM,
	42,		// HD_BELTS_BM,
	35,		// HD_STAR_BM1,
	35,		// HD_STAR_BM2,
	35,		// HD_STAR_BM3,
	0,		// HD_MW_BM,
	0,		// HD_MW_PRESENT,
	0,		// HD_MW_HYDRO_BM
};

// NOTE:  we can't do the following stuff until wxWindows is up and 
// ready for business, so we have a hokey check in the common entry
// point, instead of trying to do this in the ctor.
void 
Legend::Init(void) 
{
	RealBigFont = new wxFont(18, wxSWISS, wxNORMAL, wxBOLD);
	LegendFont = new wxFont(12, wxSWISS, wxNORMAL, wxBOLD);
	LegendBoldFont = new wxFont(13, wxSWISS, wxNORMAL, wxBOLD);
}

void
Legend::DrawLegendBox(wxDC& dc, int x, int y, int w, int h)
{
wxPen *p;

//fprintf(stderr, "---> x:%ld y:%ld w:%ld h:%ld r_x:%ld r_y:%ld\n",
	p = new wxPen("black", 2, 2);
	dc.SetPen(*p);
	dc.DrawLine(x,
				y,
				x,
				y + h);
	dc.DrawLine(x,
				y + h,
				x + w,
				y + h);
	dc.DrawLine(x + w,
				y + h,
				x + w,
				y);
	dc.DrawLine(x + w,
				y,
				x,
				y);
	dc.SetPen(*wxBLACK_PEN);
	delete p;

	p = new wxPen("black", 1, 1);
	dc.SetPen(*p);
	dc.DrawLine(x + LEGEND_INNER,
				y + LEGEND_INNER,
				x + LEGEND_INNER,
				y + h - LEGEND_INNER);
	dc.DrawLine(x + LEGEND_INNER,
				y + h - LEGEND_INNER,
				x + w - LEGEND_INNER,
				y + h - LEGEND_INNER);
	dc.DrawLine(x + w - LEGEND_INNER,
				y + h - LEGEND_INNER,
				x + w - LEGEND_INNER,
				y + LEGEND_INNER);
	dc.DrawLine(x + w - LEGEND_INNER,
				y + LEGEND_INNER,
				x + LEGEND_INNER,
				y + LEGEND_INNER);
	dc.SetPen(*wxBLACK_PEN);
	delete p;
}

void 
Legend::DrawWorldBM(wxDC& dc, char *desc, char *uwp, bool draw, 
				int x, int y, int *max_x)
{
long w,h,descent;

	if(draw) {
		dc.DrawText(desc, 
					x + (2 * LEGEND_TEXT_OFFSET) + HEX_CENTER_WIDTH, y);
		drawn_hex.DrawMWBM(dc, uwp, x + LEGEND_TEXT_OFFSET + 6, y + 6);
	}
	dc.GetTextExtent(desc, &w, &h, &descent);
	w += (3 * LEGEND_TEXT_OFFSET) + HEX_CENTER_WIDTH;
	if(w > *max_x)
		*max_x = w;
}

void 
Legend::DrawLegendMap(wxDC& dc, bool draw, int x1, int y1, int x2, int y2,
			int ss_ndx, int *max_x, int *max_y)
{
char buff[2];
int i,x,y;

//fprintf(stderr, "Map(%d): (%d %d) (%d %d)\n", draw, x1, y1, x2, y2);
	if(x2 < (MIN_LEGEND_MAP_X * 4))
		x2 = MIN_LEGEND_MAP_X * 4;
	if(y2 < (MIN_LEGEND_MAP_Y * 4))
		y2 = MIN_LEGEND_MAP_Y * 4;

	x = x2 / 4;
	y = y2 / 4;
	if(draw) {
		dc.SetBrush(*wxTRANSPARENT_BRUSH);
		dc.SetFont(*RealBigFont);
		for(i = 0;i < MAX_SS;i++) {
			dc.DrawRectangle(x1 + ((i % 4) * x),
						y1 + ((i / 4) * y),
						x + 1, y + 1);
			// shade in current ss
			if(ss_ndx == i) {
				dc.SetBrush(*wxGREY_BRUSH);
				dc.DrawRectangle(x1 + ((i % 4) * x),
						y1 + ((i / 4) * y),
						x + 1, y + 1);
				dc.SetBrush(*wxTRANSPARENT_BRUSH);
			}
			sprintf(buff, "%c", i + 'A');
			dc.DrawText(buff, 
					x1 + ((i % 4) * x) + (x / 2) - LEGEND_X_OFFSET,
					y1 + ((i / 4) * y) + (y / 2) - LEGEND_Y_OFFSET);
		}
	
	}

	*max_x = (x * 4);
	*max_y = (y * 4);
//fprintf(stderr, "Map(r): (%d %d)\n", *max_x, *max_y);
}

void 
Legend::DrawSSList(wxDC& dc, bool draw, int x1, int y1, int x2, int y2,
			BaseSector *sect, int *max_x, int *max_y)
{
char *s,buff[120];
int i;
long w,h,descent;

//fprintf(stderr, "SSList(%d): (%d %d) (%d %d)\n", draw, x1, y1, x2, y2);
	for(i = 0;i < MAX_SS;i++) {
		sprintf(buff, "%c", 'A' + i);
		if(draw)
			dc.DrawText(buff, x1 + LEGEND_TEXT_OFFSET,
				   y1 + (i * LEGEND_TEXT_Y_INCR) + LEGEND_TEXT_OFFSET);

		if((s = sect->GetSSName(i)) == NULL)
			sprintf(buff, "Subsector %c", 'A' + i);
		else
			sprintf(buff, "%s", s);

		if(draw)
			dc.DrawText(buff, 
				x1 + LEGEND_TEXT_OFFSET + (2 * LEGEND_TEXT_X_INCR),
				y1 + (i * LEGEND_TEXT_Y_INCR) + LEGEND_TEXT_OFFSET);
		
		dc.GetTextExtent(buff, &w, &h, &descent);
		w += (2 * LEGEND_TEXT_X_INCR);
		if(w > *max_x)
			*max_x = w;
	}

	*max_y = (MAX_SS * LEGEND_TEXT_Y_INCR) + (2 * LEGEND_TEXT_OFFSET);
	*max_x += 2 * LEGEND_TEXT_OFFSET;

	if(*max_x > x2)
		x2 = *max_x;
	if(*max_y > y2)
		y2 = *max_y;
	if(draw)
		DrawLegendBox(dc, x1, y1, x2, y2);
//fprintf(stderr, "SSList(r): (%d %d)\n", *max_x, *max_y);
}

void 
Legend::DrawBorder(wxDC& dc, bool draw, int x1, int y1, int x2, int y2,
			ColorTable *ct, ColorArray *colors,
			int *max_x, int *max_y)
{
	DrawLegendRoutesBorders(dc, TRUE, draw, x1, y1, x2, y2,
					ct, colors, max_x, max_y);
}

void 
Legend::DrawRoutes(wxDC& dc, bool draw, int x1, int y1, int x2, int y2,
			ColorTable *ct, ColorArray *colors,
			int *max_x, int *max_y)
{
	DrawLegendRoutesBorders(dc, FALSE, draw, x1, y1, x2, y2,
					ct, colors, max_x, max_y);
}

void 
Legend::DrawLegendRoutesBorders(wxDC& dc, bool border, bool draw,
			int x, int y, int x2, int y2,
			ColorTable *ct, ColorArray *colors,
			int *max_x, int *max_y)
{
char buff[120];
short color_ndx;
int y1;
long w,h,descent;
ColorTableEntry *cte;
ListNode *ln;
wxPen *p;

//fprintf(stderr, "RB(%d): (%d %d) (%d %d)\n", draw, x, y, x2, y2);
	y1 = y + LEGEND_TEXT_OFFSET;
	dc.SetFont(*LegendBoldFont);
	if(border)
		sprintf(buff, "Borders");
	else
		sprintf(buff, "Routes");
	if(draw)
		dc.DrawText(buff, x + LEGEND_TEXT_OFFSET, y1);

	dc.GetTextExtent(buff, &w, &h, &descent);
	*max_x = w;
	y1 += 16;

	dc.SetFont(*LegendFont);
	
	ln = ct->First();
	while(ln != NULL) {
		cte = (ColorTableEntry *)ln->Data();
		color_ndx = cte->GetColor();
		cte->GetFormated(buff);

		if(draw) {
			p = new wxPen(*colors->GetColor(color_ndx),
						  BORDER_WIDTH, BORDER_STYLE);
			dc.SetPen(*p);
			dc.DrawLine(x + LEGEND_TEXT_OFFSET, y1 + 6,
						x + LEGEND_TEXT_OFFSET + LEGEND_LINE_LENGTH, y1 + 6);
			dc.DrawText(buff,
						x + (2 * LEGEND_TEXT_OFFSET) + LEGEND_LINE_LENGTH, y1);
		}

		y1 += LEGEND_TEXT_Y_INCR;
		dc.GetTextExtent(buff, &w, &h, &descent);
		if(w > *max_x)
			*max_x = w;

		if(draw) {
			dc.SetPen(*wxBLACK_PEN);
			delete p;
		}
		ln = ct->Next();
	}

	*max_y = y1 + LEGEND_TEXT_OFFSET - y;
	*max_x += (3 * LEGEND_TEXT_OFFSET) + LEGEND_LINE_LENGTH;

	if(*max_x > x2)
		x2 = *max_x;
	if(*max_y > y2)
		y2 = *max_y;
	if(draw)
		DrawLegendBox(dc, x, y, x2, y2);
//fprintf(stderr, "RB(r): (%d %d)\n", *max_x, *max_y);
}

void 
Legend::DrawHexLegend(wxDC& dc, bool draw, int x, int y, int ww, int hh,
			BitmapList *bitmaps, TCodes *codes, ColorArray *colors,
			HexLayout *layout, int *max_x, int *max_y)
{
HEX_DISPLAY hd;
bool world_char=FALSE,world_pop=FALSE,base_bm=FALSE;
bool star_bm=FALSE,mw_bm=FALSE;
char *s;
char buff[120];
int i,j,line_x,line_y,offset_x,x1,y1,x2,y2;
long w,h,descent;
BaseTable *bt;
ListNode *ln;
BaseTableEntry *bte;
wxPen *p;
// special debug code
#ifdef DEBUG_LAYOUT_CODE
// set initial index here:
int layout_ndx=21;
#endif

//fprintf(stderr, "Hex(%d): (%d %d) (%d %d)\n", draw, x, y, ww, hh);
	drawn_hex.GetHexCorner(2, &line_x, &line_y);

	x1 = x + LEGEND_TEXT_OFFSET;
	y1 = y + LEGEND_TEXT_OFFSET;
	y2 = y + LEGEND_TEXT_OFFSET + 18;
	x2 = x + ((227 - (4 * LEGEND_TEXT_OFFSET)) - line_x);
//fprintf(stderr, "--> (%d %d) (%d %d) (%d %d)\n", x, y, x1, y1, x2, y2);
// the hex
	if(draw){
		drawn_hex.DrawHex(dc, x2, y2);
		drawn_hex.Start(dc);
	}

// hex information
// XXX note:  no practical way to handle layout of individual hexes
	for(i = 0;i < MAX_LAYOUT_X;i++) {
		for(j = 0;j < MAX_LAYOUT_Y;j++) {
			drawn_hex.GetTextOffsets(i, j, &line_x, &line_y);
			hd = layout->GetCode(i, j);
#ifndef DEBUG_LAYOUT_CODE
			if(HD_NONE == hd)
				continue;
#endif
			if(HD_MW_HYDRO_BM == hd)
				world_char = TRUE;
			if(HD_MW_BM == hd)
				mw_bm = TRUE;
			if(HD_NAME_POP == hd)
				world_pop = TRUE;
			if((HD_BASE_BM1 == hd) || (HD_BASE_BM2 == hd))
				base_bm = TRUE;
			if((HD_STAR_BM1 == hd) || (HD_STAR_BM1 == hd) ||
				(HD_STAR_BM1 == hd))
				star_bm = TRUE;
			if(draw) {
				int draw_x,draw_y;

				draw_x = line_x + x2;
				draw_y = line_y + y2;
				dc.DrawLine(line_pos[j][i][0] + x,
							line_pos[j][i][1] + y,
							line_x + x2, 
							line_y + y2);
				s = layout->GetString(hd, TRUE);
				offset_x = text_pos[j][i][0] + x;
				// XXX should be a better way...
				if(text_pos[j][i][0] < 20) {
#ifdef DEBUG_LAYOUT_CODE
					s = layout->GetString((HEX_DISPLAY) layout_ndx, TRUE);
					offset_x += text_offset[layout_ndx];
					layout_ndx++;
#else
					offset_x += text_offset[hd];
#endif
				}
				dc.DrawText(s, offset_x, text_pos[j][i][1] + y);
			}
		}
	}

	if(draw)
		drawn_hex.DrawDummy(dc, layout, bitmaps, codes, x2, y2);

	drawn_hex.GetHexCorner(3, &line_x, &line_y);
	y1 += line_y;
	y1 += (5 * LEGEND_TEXT_OFFSET);

	// world char.
	if((world_char) || (mw_bm)) {
		dc.SetFont(*LegendBoldFont);
		sprintf(buff, "World Characteristics");
		if(draw) {
			dc.DrawText(buff, x1, y1);
		}
		y1 += 20;
		dc.SetFont(*LegendFont);
	}

	if(world_char) {
		// belts
		sprintf(buff, "Asteroid Belt");
		if(draw) {
			dc.DrawText(buff, 
						x + (2 * LEGEND_TEXT_OFFSET) + HEX_CENTER_WIDTH, y1);
			drawn_hex.DrawAstMW(dc, x + LEGEND_TEXT_OFFSET + 6, y1 + 6);
		}
		y1 += 15;
		dc.GetTextExtent(buff, &w, &h, &descent);
		w += (3 * LEGEND_TEXT_OFFSET) + HEX_CENTER_WIDTH;
		if(w > *max_x)
			*max_x = w;
	
		// no water
		sprintf(buff, "No Water Present");
		if(draw) {
			dc.DrawText(buff, 
						x + (2 * LEGEND_TEXT_OFFSET) + HEX_CENTER_WIDTH, y1);
			drawn_hex.DrawDryMW(dc, x + LEGEND_TEXT_OFFSET + 6, y1 + 6);
		}
		y1 += 15;
		dc.GetTextExtent(buff, &w, &h, &descent);
		w += (3 * LEGEND_TEXT_OFFSET) + HEX_CENTER_WIDTH;
		if(w > *max_x)
			*max_x = w;

		// water
		sprintf(buff, "Water Present");
		if(draw) {
			dc.DrawText(buff, 
						x + (2 * LEGEND_TEXT_OFFSET) + HEX_CENTER_WIDTH, y1);
			drawn_hex.DrawMW(dc, x + LEGEND_TEXT_OFFSET + 6, y1 + 6);
		}
		y1 += 20;
		dc.GetTextExtent(buff, &w, &h, &descent);
		w += (3 * LEGEND_TEXT_OFFSET) + HEX_CENTER_WIDTH;
		if(w > *max_x)
			*max_x = w;
	
#if 0
// cp
		sprintf(buff, "Communication Point");
		if(draw) {
			dc.DrawText(buff, x + (2 * LEGEND_TEXT_OFFSET) + HEX_CENTER_WIDTH, y1);
			dc.SetBrush(*wxWHITE_BRUSH);
			dc.SetPen(*wxBLACK_PEN);
			dc.DrawEllipse(x + LEGEND_TEXT_OFFSET, y1, 
						HEX_CENTER_WIDTH + 1, HEX_CENTER_WIDTH + 1);
	
			dc.DrawLine(HEX_IMAGE_X + alt_x + 2, 
						HEX_IMAGE_Y + alt_y + 2, 
						HEX_IMAGE_X + alt_x + HEX_CENTER_WIDTH - 2, 
						HEX_IMAGE_Y + alt_y + HEX_CENTER_WIDTH - 2);
			dc.DrawLine(HEX_IMAGE_X + alt_x + 2, 
						HEX_IMAGE_Y + alt_y + HEX_CENTER_WIDTH - 2, 
						HEX_IMAGE_X + alt_x + HEX_CENTER_WIDTH - 2, 
						HEX_IMAGE_Y + alt_y + 2);
		}
		y1 += 20;
		dc.GetTextExtent(buff, &w, &h, &descent);
		w += (3 * LEGEND_TEXT_OFFSET) + HEX_CENTER_WIDTH;
		if(w > *max_x)
			*max_x = w;
#endif
	}
	if(mw_bm) {
		DrawWorldBM(dc, "Size 0", "A-0000000", draw, x, y1, max_x);
		y1 += 15;
		DrawWorldBM(dc, "Hydro 0, Atmos 0", "A-8000000", draw, x, y1, max_x);
		y1 += 15;
		DrawWorldBM(dc, "Hydro 0, Atmos 5+", "A-8500000", draw, x, y1, max_x);
		y1 += 15;
		DrawWorldBM(dc, "Hydro 5+, Atmos 0", "A-8050000", draw, x, y1, max_x);
		y1 += 15;
		DrawWorldBM(dc, "Hydro 5+, Atmos 5+", "A-8550000", draw, x, y1, max_x);
		y1 += 15;
		DrawWorldBM(dc, "Hydro 9+", "A-88A0000", draw, x, y1, max_x);
		y1 += 15;
		DrawWorldBM(dc, "Atmos A+", "A-8A00000", draw, x, y1, max_x);
		y1 += 20;
	}

	// bases
	if(base_bm) {
		dc.SetFont(*LegendBoldFont);
		sprintf(buff, "Bases");
		if(draw) {
			dc.DrawText(buff, x1, y1);
		}
		y1 += 20;

		dc.SetFont(*LegendFont);

		// loop thru the bases
		bt = codes->GetBaseTable();
		ln = bt->First();
		while(ln != NULL) {
			wxBitmap *bm1,*bm2;

			bte = (BaseTableEntry *)ln->Data();
			bm1 = bitmaps->NthBM(bte->GetIndex(0), TRUE);
			bm2 = bitmaps->NthBM(bte->GetIndex(1), TRUE);
			// XXX most any codes with 2 bitmaps just reflect other bases
			if(NULL == bm2) {
				s = bte->GetDesc();
				sprintf(buff, s);

				if(draw) {
					drawn_hex.DrawBM(dc, bm1, x1 + 6, y1 + 6);
					dc.DrawText(buff, x + (2 * LEGEND_TEXT_OFFSET) + 15, y1);
				}

				y1 += 18;
				dc.GetTextExtent(buff, &w, &h, &descent);
				w += (3 * LEGEND_TEXT_OFFSET) + 15;
				if(w > *max_x)
					*max_x = w;
			}
			ln = bt->Next();
		}

		y1 += 5;
	}

	// draw Zones
	dc.SetFont(*LegendBoldFont);
	sprintf(buff, "Travel Zones");
	if(draw) {
		dc.DrawText(buff, x1, y1);
	}
	y1 += 20;
	dc.SetFont(*LegendFont);

	// amber
	sprintf(buff, "Amber Zone");
	if(draw) {
		p = new wxPen(*colors->GetColor(AMBER_COLOUR_NDX),
					  BORDER_WIDTH, BORDER_STYLE);
		dc.SetPen(*p);
		dc.DrawLine(x + LEGEND_TEXT_OFFSET, y1 + 6,
					x + LEGEND_TEXT_OFFSET + LEGEND_LINE_LENGTH, y1 + 6);
		dc.DrawText(buff,
					x + (2 * LEGEND_TEXT_OFFSET) + LEGEND_LINE_LENGTH, y1);
	}

	y1 += LEGEND_TEXT_Y_INCR;
	dc.GetTextExtent(buff, &w, &h, &descent);
	w += (3 * LEGEND_TEXT_OFFSET) + LEGEND_LINE_LENGTH;
	if(w > *max_x)
		*max_x = w;

	if(draw) {
		dc.SetPen(*wxBLACK_PEN);
		delete p;
	}

	// red
	sprintf(buff, "Red Zone");
	if(draw) {
		p = new wxPen(*colors->GetColor(RED_COLOUR_NDX), BORDER_WIDTH,
					  BORDER_STYLE);
		dc.SetPen(*p);
		dc.DrawLine(x + LEGEND_TEXT_OFFSET, y1 + 6,
					x + LEGEND_TEXT_OFFSET + LEGEND_LINE_LENGTH, y1 + 6);
		dc.DrawText(buff,
					x + (2 * LEGEND_TEXT_OFFSET) + LEGEND_LINE_LENGTH, y1);
	}

	y1 += LEGEND_TEXT_Y_INCR;
	dc.GetTextExtent(buff, &w, &h, &descent);
	w += (3 * LEGEND_TEXT_OFFSET) + LEGEND_LINE_LENGTH;
	if(w > *max_x)
		*max_x = w;

	if(draw) {
		dc.SetPen(*wxBLACK_PEN);
		delete p;
	}

	// draw population
	if(world_pop) {
		dc.SetFont(*LegendBoldFont);
		sprintf(buff, "Population");
		if(draw) {
			dc.DrawText(buff, x1, y1);
		}
		y1 += 20;

		// low
		dc.SetFont(*LegendFont);
		sprintf(buff, "Secondus");
		if(draw) {
			dc.DrawText(buff, x1, y1);
		}
		sprintf(buff, "Under a billion");
		if(draw) {
			dc.DrawText(buff, x1 + 60, y1);
		}

		y1 += LEGEND_TEXT_Y_INCR;
		dc.GetTextExtent(buff, &w, &h, &descent);
		w += (2 * LEGEND_TEXT_OFFSET) + 60;
		if(w > *max_x)
			*max_x = w;

		// high
		dc.SetFont(*LegendBoldFont);
		sprintf(buff, "Primus");
		if(draw) {
			dc.DrawText(buff, x1, y1);
		}
		dc.SetFont(*LegendFont);
		sprintf(buff, "A billion or more");
		if(draw) {
			dc.DrawText(buff, x1 + 60, y1);
		}
	
		y1 += 18;
		dc.GetTextExtent(buff, &w, &h, &descent);
		w += (2 * LEGEND_TEXT_OFFSET) + 60;
		if(w > *max_x)
			*max_x = w;
	}

	if(*max_x < 275)
		*max_x = 275;

	*max_y = y1 + LEGEND_TEXT_OFFSET - y;

	if(*max_x > ww)
		ww = *max_x;
	if(*max_y > hh)
		hh = *max_y;
	if(draw)
		DrawLegendBox(dc, x, y, ww, hh);
//fprintf(stderr, "Hex(r): (%d %d)\n", *max_x, *max_y);
}

void
Legend::DrawLegend(wxDC& dc, int init_x, int init_y, int *max_x, int *max_y,
		bool draw, int ss_ndx, BaseSector *sect, BitmapList *bitmaps, 
		TCodes *codes, ColorTable *bt, ColorTable *rt, ColorArray *colors,
		HexLayout *layout)
{
char *s,*ss;
char buff[120];
int start_x[5],start_y[5];
int end_x[5],end_y[5];
int temp_x,temp_y;
int i;

	if(NULL == RealBigFont)
		Init();

	dc.SetPen(*wxBLACK_PEN);
	temp_x = temp_y = 0;
	for(i = 0;i < 5;i++)
		start_x[i] = start_y[i] = end_x[i] = end_y[i] = 0;

	// verify sizes use dummy coords of (0,0) (200,200)
	DrawLegendMap(dc, FALSE, 0, 0, 200, 200, ss_ndx, &end_x[0], &end_y[0]);
	DrawHexLegend(dc, FALSE, 0, 0, 200, 200, bitmaps, codes, colors,
					layout, &end_x[1], &end_y[1]);
	DrawBorder(dc, FALSE, 0, 0, 200, 200, bt, colors, &end_x[2], &end_y[2]);
	DrawRoutes(dc, FALSE, 0, 0, 200, 200, rt, colors, &end_x[3], &end_y[3]);
	DrawSSList(dc, FALSE, 0, 0, 200, 200, sect, &end_x[4], &end_y[4]);

	// set up width and height
	if(ss_ndx < 0) {
		start_x[0] = start_x[2] = start_x[4] = init_x;
		temp_x = end_x[0];
		if(end_x[2] > temp_x)
			temp_x = end_x[2];
		if(end_x[4] > temp_x)
			temp_x = end_x[4];
		end_x[0] = end_x[2] = end_x[4] = temp_x;
		start_x[1] = start_x[3] = init_x + temp_x + (LEGEND_X_OFFSET);
		if(end_x[3] > end_x[1])
			end_x[1] = end_x[3];
		else
			end_x[3] = end_x[1];

		start_y[0] = init_y + LEGEND_TEXT_Y_INCR + LEGEND_BORDER_OFFSET;
		start_y[4] = start_y[0] + end_y[0] + LEGEND_Y_OFFSET;
		start_y[2] = start_y[4] + end_y[4] + LEGEND_Y_OFFSET;
		start_y[1] = init_y + LEGEND_TEXT_Y_INCR + LEGEND_BORDER_OFFSET;
		start_y[3] = start_y[1] + end_y[1] + LEGEND_Y_OFFSET;

		*max_x = start_x[1] + end_x[1];
		if((start_y[3] + end_y[3]) > (start_y[2] + end_y[2]))
			*max_y = (start_y[3] + end_y[3]);
		else
			*max_y = (start_y[2] + end_y[2]);
	} else {
		start_y[0] = init_y + LEGEND_TEXT_Y_INCR + LEGEND_BORDER_OFFSET;
		start_y[1] = start_y[0] + end_y[0] + LEGEND_Y_OFFSET;
		start_y[2] = start_y[1] + end_y[1];
		start_y[3] = start_y[2] + end_y[2];

		temp_x = 0;
		for(i = 0;i < 4;i++) {
			start_x[i] = init_x;
			
			if(end_x[i] > temp_x)
				temp_x = end_x[i];
		}
		for(i = 0;i < 4;i++) {
			end_x[i] = temp_x;
		}
		*max_x = init_x + temp_x;
		*max_y = start_y[3] + end_y[3];
	}

	if(!draw)
		return;

	// now draw the legend
	dc.SetFont(*LegendFont);
	if(ss_ndx > -1) {
		s = sect->GetName();
		ss = sect->GetSSName(ss_ndx);
		if(s != NULL) {
			if(ss != NULL) 
				sprintf(buff, "%s -- SubSector (%c) of the %s Sector.", 
						ss, ss_ndx + 'A', s);
			else
				sprintf(buff, "SubSector %c of the %s Sector.", 
					ss_ndx + 'A', s);
		} else {
			if(ss != NULL) 
				sprintf(buff, "%s -- SubSector (%c).", ss, ss_ndx + 'A');
			else
				sprintf(buff, "SubSector %c.", ss_ndx + 'A');
		}
	} else {
		s = sect->GetName();
		if(s != NULL)
			sprintf(buff, "Legend for the %s sector.", s);
		else
			sprintf(buff, "Legend for sector map.");
	}
	dc.DrawText(buff, init_x, init_y);
	temp_x = temp_y = 0;
	DrawLegendMap(dc, TRUE, start_x[0], start_y[0],
					end_x[0], end_y[0], ss_ndx, &temp_x, &temp_y);
	temp_x = temp_y = 0;
	DrawHexLegend(dc, TRUE, start_x[1], start_y[1],
					end_x[1], end_y[1], bitmaps, codes, colors,
					layout, &temp_x, &temp_y);
	temp_x = temp_y = 0;
	DrawBorder(dc, TRUE, start_x[2], start_y[2],
					end_x[2], end_y[2], bt, colors, &temp_x, &temp_y);
	temp_x = temp_y = 0;
	DrawRoutes(dc, TRUE, start_x[3], start_y[3],
					end_x[3], end_y[3], rt, colors, &temp_x, &temp_y);
	temp_x = temp_y = 0;
	if(ss_ndx < 0)
		DrawSSList(dc, TRUE, start_x[4], start_y[4],
					end_x[4], end_y[4], sect, &temp_x, &temp_y);
}

