/*
 * File:      mtu.cpp
 * Purpose:   MTU menu program
 * Author:    Mark A. Nordstrand
 * Created:   8/1/99
 * Updated:
 * Copyright: LGPL
Traveller is a registered trademark of Far Future Enterprises.
Portions based upon material Copyright 1977-2002 Far Future Enterprises.
 */

/* rcsid[] = "$RCSfile: mtu.cpp,v $ $Revision: 1.16 $ $Author: man $ $Date: 2002/10/06 16:32:37 $" */


// For compilers that support precompilation, includes "wx/wx.h".
#include "wx/wxprec.h"


#ifndef WX_PRECOMP
#include "wx/wx.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <math.h>
#include "llist.h"
#include "mtu.h"
#include "t_res.h"
#include "wx/confbase.h"
#include "wx/fileconf.h"


#if defined(__WXGTK__) || defined(__WXMOTIF__)
#include "mtu.xpm"
#include "bitmaps/config.xpm"
#include "bitmaps/eye.xpm"
#include "bitmaps/exit.xpm"
#include "bitmaps/help.xpm"
#include "bitmaps/mag.xpm"
#include "bitmaps/mini-x2.xpm"
#include "bitmaps/new.xpm"
#include "bitmaps/open.xpm"
#include "bitmaps/print.xpm"
#include "bitmaps/save.xpm"
#include "bitmaps/ssearch.xpm"
#include "bitmaps/subsect.xpm"
#endif

#define USER_ID_OFFSET		10000

enum {
	 ID_TOOLBAR=500,
	 ID_RESOURCE,
	 ID_CFG,
	 ID_NEW,
	 ID_OPEN,
	 ID_COMBO,
	 ID_SEARCH,
	 ID_PRINT,
	 ID_VIEW,
	 ID_SEL,
	 ID_DETAIL,
	 ID_HELP,
	 ID_EXIT,
	 ID_USER1,								// dummy menu event
	 ID_USER2=(ID_USER1 + USER_ID_OFFSET)	// real menu events
};

#define TWO_PI				(2 * 3.1415927)

#define DRAW_OFFSET			5
#define DRAW_MULT			4
#define DRAW_SIZE_X			(MAX_X_HEX * DRAW_MULT)
#define DRAW_SIZE_Y			(MAX_Y_HEX * DRAW_MULT)

#define MAX_STATUS_BOX		3

IMPLEMENT_APP(MTU)

MainFrame* frame=NULL;
// The `main program' equivalent, creating the windows and returning the
// main frame
bool MTU::OnInit(void)
{
int status_widths[MAX_STATUS_BOX] =
		{ -1, 172, 128 };
	// Create the main frame window
	frame = new MainFrame();

	// Give it a status line
	frame->CreateStatusBar(MAX_STATUS_BOX);
	frame->SetStatusWidths(MAX_STATUS_BOX, status_widths);

	// Give it an icon
	frame->SetIcon(wxICON(mtu));
#if 0
	// Make a menubar
	wxMenu *fileMenu = new wxMenu;
	fileMenu->Append(wxID_EXIT, "E&xit", "Quit toolbar sample" );

	wxMenu *helpMenu = new wxMenu;
	helpMenu->Append(wxID_HELP, "&About", "About toolbar sample");

	wxMenuBar* menuBar = new wxMenuBar( wxMB_DOCKABLE );

	menuBar->Append(fileMenu, "&File");
	menuBar->Append(helpMenu, "&Help");

	// Associate the menu bar with the frame
	frame->SetMenuBar(menuBar);
#endif
	// Create the toolbar
	frame->CreateToolBar(wxNO_BORDER | wxHORIZONTAL | wxTB_DOCKABLE, 
			ID_TOOLBAR);
	
	frame->GetToolBar()->SetMargins( 2, 2 );

	frame->InitToolbar();
	frame->SetButtons(frame->GetToolBar(), FALSE);

	frame->canvas = new MTUCanvas(frame);
	frame->canvas->SetBackgroundColour(*wxWHITE);

	// Force a resize. This should probably be replaced by a call to a wxFrame
	// function that lays out default decorations and the remaining content window.
	wxSizeEvent event(wxSize(-1, -1), frame->GetId());
	frame->OnSize(event);
	frame->Show(TRUE);

	frame->SetStatusText("", 0);
	frame->SetStatusText("", 1);
	frame->SetStatusText("", 2);
	
	SetTopWindow(frame);

	return(TRUE);
}

#define MAX_BUTTONS		12
// Set up toolbar
//bool MTU::InitToolbar(MainFrame *frame)
bool MainFrame::InitToolbar(void)
{
wxToolBar *toolBar;
wxBitmap* bm[MAX_BUTTONS];
#ifdef __WXMSW__
int width = 24;
#else
int width = 16;
#endif
int currentX = 5;

	toolBar = GetToolBar();

#ifdef __WXMSW__
	bm[0] = new wxBitmap("icon1");
	bm[1] = new wxBitmap("icon2");
	bm[2] = new wxBitmap("icon3");
	bm[3] = new wxBitmap("icon4");
	bm[4] = new wxBitmap("icon5");
	bm[5] = new wxBitmap("icon6");
	bm[6] = new wxBitmap("icon7");
	bm[7] = new wxBitmap("icon8");
	bm[8] = new wxBitmap("icon9");
	bm[9] = new wxBitmap("icon10");
	bm[10] = new wxBitmap("icon11");
	bm[11] = new wxBitmap("icon12");
#else
	bm[0] = new wxBitmap(mini_x2_xpm);
	bm[1] = new wxBitmap(config_xpm);
	bm[2] = new wxBitmap(new_xpm);
	bm[3] = new wxBitmap(open_xpm);
	bm[4] = new wxBitmap(save_xpm);
	bm[5] = new wxBitmap(ssearch_xpm);
	bm[6] = new wxBitmap(print_xpm);
	bm[7] = new wxBitmap(eye_xpm);
	bm[8] = new wxBitmap(subsect_xpm);
	bm[9] = new wxBitmap(mag_xpm);
	bm[10] = new wxBitmap(help_xpm);
	bm[11] = new wxBitmap(exit_xpm);
#endif

	toolBar->AddTool(ID_RESOURCE, *(bm[0]), wxNullBitmap, 
			FALSE, currentX, -1, (wxObject *) NULL, "Resources");
	currentX += width + 5;
	toolBar->AddTool(ID_CFG, *(bm[1]), wxNullBitmap, 
			FALSE, currentX, -1, (wxObject *) NULL, "Codes");
	currentX += width + 5;
	toolBar->AddSeparator();
#ifdef USE_NEW_FUNCTIONS
	toolBar->AddTool(ID_NEW, *(bm[2]), wxNullBitmap, 
			FALSE, currentX, -1, (wxObject *) NULL, "New");
	currentX += width + 5;
#endif

#if (wxUSE_TOOLBAR_NATIVE && !USE_GENERIC_TBAR) && !defined(__WXMOTIF__)
// I guess this won't work w/ the generic nor Motif native toolbars...
	{
	char *ptr;
	int i=0;
	UListData *uld;

		combo = new wxComboBox(toolBar, ID_COMBO, wxEmptyString,
				wxDefaultPosition, wxDefaultSize,
				0, NULL, wxCB_READONLY);

		// need a dummy one for the initial state (I guess....)
		combo->Append("");
		while((uld = (UListData *)u_list->NthData(i)) != NULL) {
			ptr = uld->GetDesc();
			combo->Append(ptr);
			i++;
		}
		toolBar->AddControl(combo);
	}
#else
	toolBar->AddTool(ID_OPEN, *(bm[3]), wxNullBitmap, 
			FALSE, currentX, -1, (wxObject *) NULL, "Open");
	currentX += width + 5;
#endif
//	toolBar->AddTool(ID_SAVE, *(bm[4]), wxNullBitmap, 
//			FALSE, currentX, -1, (wxObject *) NULL, "Save");
//	currentX += width + 5;
	toolBar->AddSeparator();
	toolBar->AddTool(ID_SEARCH, *(bm[5]), wxNullBitmap, 
			FALSE, currentX, -1, (wxObject *) NULL, "Search");
	currentX += width + 5;
	toolBar->AddTool(ID_PRINT, *(bm[6]), wxNullBitmap, 
			FALSE, currentX, -1, (wxObject *) NULL, "Print");
	currentX += width + 5;
	toolBar->AddTool(ID_VIEW, *(bm[7]), wxNullBitmap, 
			FALSE, currentX, -1, (wxObject *) NULL, "View");
	currentX += width + 5;
	toolBar->AddTool(ID_SEL, *(bm[8]), wxNullBitmap, 
			FALSE, currentX, -1, (wxObject *) NULL, "Select");
	currentX += width + 5;
	toolBar->AddTool(ID_DETAIL, *(bm[9]), wxNullBitmap, 
			FALSE, currentX, -1, (wxObject *) NULL, "Detail");
	currentX += width + 5;
	toolBar->AddSeparator();
	toolBar->AddTool(ID_HELP, *(bm[10]), wxNullBitmap, 
			FALSE, currentX, -1, (wxObject *) NULL, "Help");
	currentX += width + 5;
	toolBar->AddSeparator();
	toolBar->AddTool(ID_EXIT, *(bm[11]), wxNullBitmap, 
			FALSE, currentX, -1, (wxObject *) NULL, "Exit");
	
	toolBar->EnableTool( wxID_PRINT, FALSE );

	toolBar->Realize();

	// Can delete the bitmaps since they're reference counted
	int i;
	for (i = 0; i < 8; i++)
		delete bm[i];

	return(TRUE);
}

// wxID_HELP will be processed for the 'About' menu and the toolbar help button.

BEGIN_EVENT_TABLE(MainFrame, wxFrame)
	EVT_MENU(wxID_EXIT, MainFrame::OnQuit)
	EVT_MENU(wxID_HELP, MainFrame::OnAbout)

	EVT_MENU_RANGE(ID_USER2, ID_USER2 + USER_ID_OFFSET, MainFrame::OnUser)

	EVT_CLOSE(MainFrame::OnCloseWindow)
	EVT_TOOL_RANGE(ID_RESOURCE-1, ID_EXIT, MainFrame::OnToolLeftClick)
	EVT_TOOL_ENTER(ID_TOOLBAR, MainFrame::OnToolEnter)
	EVT_COMBOBOX(ID_COMBO, MainFrame::OnCombo)
END_EVENT_TABLE()

// Define my frame constructor
MainFrame::MainFrame() :
	wxFrame((wxFrame *) NULL, -1, (const wxString) "My Traveller Universe",
	   wxPoint(100, 100), wxSize(450, 300))
{
	// simple initialization
	choose_dialog = NULL;
	code_dialog = NULL;
	resource_dialog = NULL;
	view_dialog = NULL;
	view3d_dialog = NULL;
	print_dialog = NULL;
	print_dialog3 = NULL;
	search_dialog = NULL;
#ifdef USE_NEW_FUNCTIONS
	new_dialog = NULL;
	import_dialog = NULL;
	d3_dialog = NULL;
	d2_dialog = NULL;
#endif
	sectors = NULL;
	sd = NULL;
	star = NULL;
	codes = NULL;
	colors = NULL;

	view = new View3D();

	// set up the help system
	help = new THelp();

	dirs = new MTUFile();
	LoadLocalResources();
	htv = HTV_NONE;
	LoadData();

	star_font = new wxFont(8, wxSWISS, wxNORMAL, wxNORMAL);

	u_list = new UList(dirs->GetMTUDir());

	user_menu = NULL;
}

MainFrame::~MainFrame()
{
char str[MAX_FILE_LENGTH],str1[MAX_FILE_LENGTH];
wxString ptr;
wxConfig *config;

	config = dirs->GetConfig();
	// save each flag
	WriteFlags(config, ENTRY_DISPLAY_NAME, FLAG_SECT_NAME);
	WriteFlags(config, ENTRY_DISPLAY_ROUTES, FLAG_SECT_ROUTES);
	WriteFlags(config, ENTRY_DISPLAY_BORDERS, FLAG_SECT_BORDERS);
	WriteFlags(config, ENTRY_DISPLAY_NAME3D, FLAG_SECT_3D_NAME);
	WriteFlags(config, ENTRY_DISPLAY_DETAIL3D, FLAG_SECT_3D_DETAILED);
	WriteFlags(config, ENTRY_DISPLAY_ROUTES3D, FLAG_SECT_3D_ROUTES);

	// save 3d scale
	sprintf(str, "/%s/%s", SECTION_DISPLAY, ENTRY_DISPLAY_SCALE);
	sprintf(str1, "%.2f", GetScale());
	config->Write((const wxString) str, (const wxString) str1);
	
	// save 3d rotation
	sprintf(str, "/%s/%s", SECTION_DISPLAY, ENTRY_DISPLAY_ROT_X);
	sprintf(str1, "%d", GetXRot());
	config->Write((const wxString) str, (const wxString) str1);

	sprintf(str, "/%s/%s", SECTION_DISPLAY, ENTRY_DISPLAY_ROT_Y);
	sprintf(str1, "%d", GetYRot());
	config->Write((const wxString) str, (const wxString) str1);

	sprintf(str, "/%s/%s", SECTION_DISPLAY, ENTRY_DISPLAY_ROT_Z);
	sprintf(str1, "%d", GetZRot());
	config->Write((const wxString) str, (const wxString) str1);
	
	config->Flush();
	delete wxConfigBase::Set((wxConfigBase *) NULL);

	if(colors != NULL)
		delete colors;

	if(u_list != NULL)
		delete u_list;
//	if(sectors != NULL)
//		delete sectors;
	if(codes != NULL)
		delete codes;
	if(dirs != NULL)
		delete dirs;

	delete star_font;
	delete help;
	delete view;
}

void 
MainFrame::OnUser(wxCommandEvent& event)
{
	if(user_menu != NULL)
		user_menu->HandleEvent(dirs, event.GetId());
}

void 
MainFrame::OnQuit(wxCommandEvent& WXUNUSED(event))
{
	// we're outa' here
	Destroy();
//	Close(TRUE);
}

void 
MainFrame::OnAbout(wxCommandEvent& WXUNUSED(event))
{
	DoAbout();
}

void
MainFrame::OnCombo(wxCommandEvent& event)
{
	OpenU(event.GetSelection() - 1);
}

// Define the behaviour for the frame closing
// - must delete all frames except for the main one.
void 
MainFrame::OnCloseWindow(wxCloseEvent& WXUNUSED(event))
{
	Destroy();
}

static char *msgs[] = {
	"Edit Resources",
	"Edit Codes",
	"Create New Universe",
#if (wxUSE_TOOLBAR_NATIVE && !USE_GENERIC_TBAR) && !defined(__WXMOTIF__)
	"",
#endif
	"Open Existing Universe",
	"Search Current Universe",
	"Print Current Universe",
	"Configure View of Current Universe",
	"Select Sector",
	"Detail",
	"Get Some Help",
	"Exit Program"
};

void 
MainFrame::OnToolLeftClick(wxCommandEvent& event)
{
	switch(event.GetId()) {
		case ID_OPEN:
			DoOpen();
			break;
		case ID_RESOURCE:
			{
			wxConfig *config;

			config = dirs->GetConfig();
			if(NULL == resource_dialog)
				resource_dialog = new ResourceDialog(this, config);
			if(resource_dialog->GetResources()) {
				LoadResources();
			}
			}
			break;
		case ID_CFG:
			{
				BitmapList *bl;
				wxConfig *config;

				config = dirs->GetConfig();
				bl = new BitmapList(config);

				if(NULL == code_dialog)
					code_dialog = new TCodeDialog(this, FALSE);
				if(code_dialog->EditCodes(codes, bl)) {
					LoadData();
				}
				delete bl;
			}
			break;
		case ID_VIEW:
			if(!sectors->Get3D()) {
				if(NULL == view_dialog)
					view_dialog = new ViewDialog(this);
				if(view_dialog->GetView(GetFlags(), htv)) {
				}
			} else {
				if(NULL == view3d_dialog)
					view3d_dialog = new View3DDialog(this);
				if(view3d_dialog->GetView(GetFlags(), htv,
						GetScale(), GetXRot(), 
						GetYRot(), GetZRot())) {
				}
			}
			UpdateStatus();
			break;
		case ID_PRINT:
			if(!sectors->Get3D()) {
				if(NULL == print_dialog)
					print_dialog = new PrintDialog(this);
				if(print_dialog->DoPrintOut(sectors,DRAW_SIZE_X,DRAW_SIZE_Y)) {
				}
			} else {
				long x,y;

				if(NULL == print_dialog3)
					print_dialog3 = new Print3DDialog(this);
				frame->GetMax3DView(&x, &y);
				if(print_dialog3->DoPrintOut(sectors, x, y+20)) {
				}
			}
			break;
		case ID_HELP:
			DoAbout();
			break;
		case ID_SEL:
			DoSelect();
			break;
		case ID_DETAIL:
			{
			char buff[MAX_FILE_LENGTH * 3],buff2[MAX_FILE_LENGTH],*dir=NULL;
			char *program;
				
				if(sd != NULL) {
					BaseSector *bs;

					program = dirs->GetSector();
					bs = sd->GetSector();
					if(bs != NULL) {
						strcpy(buff2, bs->GetSecName());
						dir = buff2;
						sprintf(buff, "%s %s", program, bs->GetSecName());
					}
				} else if(star != NULL) {
					int i=0;
					Sector3D *s;

					s = sectors->Get3DSector();
					program = dirs->GetSystem();
					if(s != NULL) {
						char *ptr,buff6[MAX_FILE_LENGTH];
						ptr = dirs->GetSectSysDir();
	
						sprintf(buff, 
						 "%s -n %s -S %s -X%d -Y%d -Z%d -o %s/x%dy%dz%d.sys",
							program, 
							star->GetStar(0)->GetName(buff6), 
							sectors->GetDesc(),
							star->GetX(), star->GetY(), star->GetZ(),
							ptr, star->GetX(), star->GetY(), star->GetZ());
						for(i = 0;i < MAX_3D_STARS;i++) {
							char buff4[24],buff5[24];

							if(star->GetStar(i) != NULL) {
								sprintf(buff5, " -r %s", 
									star->GetStar(i)->GetStarStr(buff4));
								strcat(buff, buff5);
							} else {
								strcat(buff, " -r N");
							}
						}
						strcpy(buff2, s->GetFileName());
						dir = buff2;
					}
				}

				if(dir != NULL) {
					dirs->VerifyDirs();
					wxExecute(buff, FALSE);
				}
			}
			break;
		case ID_NEW:
#ifdef USE_NEW_FUNCTIONS
			DoNew();
#endif
			break;
		case ID_SEARCH:
			if(NULL == search_dialog)
				search_dialog = new SearchDialog(this);
			search_dialog->Search(NULL, sectors, codes, dirs);
			break;
		case ID_EXIT:
			Close(TRUE);
			break;
	}
}

void
MainFrame::OnToolEnter(wxCommandEvent& event)
{
int sel;

	sel = event.GetSelection();
	if((sel >= ID_RESOURCE) && (sel <= ID_EXIT)) {
		SetStatusText(msgs[sel - ID_RESOURCE]);
	}
	else
		SetStatusText("");
}

// -----------------------------------------------------------------------
void 
MainFrame::SetButtons(wxToolBar *tb, bool v)
{
	tb = GetToolBar();

	tb->EnableTool(ID_SEARCH, v);
	tb->EnableTool(ID_PRINT, v);
	tb->EnableTool(ID_VIEW, v);
	if((sectors != NULL) && (sectors->Get3D()))
		tb->EnableTool(ID_SEL, FALSE);
	else
		tb->EnableTool(ID_SEL, v);
	if((NULL == sd) && (NULL == star))
		tb->EnableTool(ID_DETAIL, FALSE);
	else
		tb->EnableTool(ID_DETAIL, v);
	tb->EnableTool(ID_EXIT, TRUE);
}

void
MainFrame::UpdateStatus(void)
{
char buff[120];

	if(sd != NULL) {
		sprintf(buff, "%s, (%d %d)", sd->GetDesc(), sd->GetX(), sd->GetY());
		SetStatusText(buff, 1);
	} else if(star != NULL) {
		char buff1[120];

		sprintf(buff, "%s (%d %d %d)", star->GetStar(0)->GetName(buff1), 
				star->GetX(), star->GetY(), star->GetZ());
		SetStatusText(buff, 1);
	} else {
		SetStatusText("", 1);
	}

	buff[0] = 0;
	if(sectors != NULL) {
		if(sectors->Get3D()) {
			sprintf(buff, "%.2f %d/%d/%d", scale, x_rot, y_rot, z_rot);
		} else {
			switch(htv) {
				case HTV_NONE:
					sprintf(buff, "None");
					break;
				case HTV_AREA:
					sprintf(buff, "Area");
					break;
				case HTV_WAR:
					sprintf(buff, "War");
					break;
			}
		}
	}


	SetStatusText(buff, 2);
}

// -----------------------------------------------------------------------
bool
MainFrame::CreateDir(char *msg, char *dir)
{
// XXX leave out until everything else is set up
#if 0
char msg_buff[1024];

	if(mkdir(dir, 0755)) {	// possible error
		if(EEXIST == errno) {
			sprintf(msg_buff, 
				"%s\n already exists.\n\nDo you wish to continue?", dir);
			if(wxMessageBox(msg_buff, "Directory Creation Error",
				wxOK | wxCANCEL | wxCENTRE |wxICON_EXCLAMATION) == wxCANCLE)
				return(FALSE);
		} else {
			sprintf(msg_buff, "Error %d occured while creating:\n%s", dir);
			wxMessageBox(msg_buff, "Directory Creation Error",
				wxOK | wxCANCEL | wxCENTRE |wxICON_EXCLAMATION);
			return(FALSE);
		}
	}

#endif
	return(TRUE);			// all ok
}

// -----------------------------------------------------------------------
void
MainFrame::LoadResources(void)
{
wxConfig *config;

	dirs->Load();
	config = dirs->GetConfig();

	if(colors != NULL)
		delete colors;

	colors = new ColorArray(config);
}

void
MainFrame::LoadLocalResources(void)
{
unsigned long f=0l;
int x,y,z;
char str[MAX_FILE_LENGTH];
wxString ptr;
wxConfig *config;

	config = dirs->GetConfig();
	// grab the global stuff
	LoadResources();

	// get flags
	f |= LoadFlags(config, ENTRY_DISPLAY_NAME, FLAG_SECT_NAME);
	f |= LoadFlags(config, ENTRY_DISPLAY_ROUTES, FLAG_SECT_ROUTES);
	f |= LoadFlags(config, ENTRY_DISPLAY_BORDERS, FLAG_SECT_BORDERS);
	f |= LoadFlags(config, ENTRY_DISPLAY_NAME3D, FLAG_SECT_3D_NAME);
	f |= LoadFlags(config, ENTRY_DISPLAY_DETAIL3D, FLAG_SECT_3D_DETAILED);
	f |= LoadFlags(config, ENTRY_DISPLAY_ROUTES3D, FLAG_SECT_3D_ROUTES);
	SetFlags(f);

	// get 3d scale
	SetScale(1.0);
	sprintf(str, "/%s/%s", SECTION_DISPLAY, ENTRY_DISPLAY_SCALE);
	if(config->Read((const wxString) str, &ptr)) {
		SetScale(atof(ptr.GetData()));
	}
	
	// get 3d rotation
	x = y = z = 0;
	sprintf(str, "/%s/%s", SECTION_DISPLAY, ENTRY_DISPLAY_ROT_X);
	if(config->Read((const wxString) str, &ptr)) {
		x = atoi(ptr.GetData());
	}
	sprintf(str, "/%s/%s", SECTION_DISPLAY, ENTRY_DISPLAY_ROT_Y);
	if(config->Read((const wxString) str, &ptr)) {
		y = atoi(ptr.GetData());
	}
	sprintf(str, "/%s/%s", SECTION_DISPLAY, ENTRY_DISPLAY_ROT_Z);
	if(config->Read((const wxString) str, &ptr)) {
		z = atoi(ptr.GetData());
	}
	SetRot(x, y, z);
}

unsigned long
MainFrame::LoadFlags(wxConfig *config, char *entry, unsigned long flag)
{
char str[MAX_FILE_LENGTH],*cfg;
wxString ptr;

	sprintf(str, "/%s/%s", SECTION_DISPLAY, entry);
	if(config->Read((const wxString) str, &ptr)) {
		cfg = (char *)ptr.GetData();
		if(('t' == cfg[0]) || ('T' == cfg[0]) || ('1' == cfg[0]))
			return(flag);
	}
	return(0);
}

void
MainFrame::WriteFlags(wxConfig *config, char *entry, unsigned long flag)
{
char str[MAX_FILE_LENGTH],str1[2];
wxString ptr;

	str1[1] = 0;
	str1[0] = VALUE_FALSE;
	sprintf(str, "/%s/%s", SECTION_DISPLAY, entry);
	if(GetFlags() & flag)
		str1[0] = VALUE_TRUE;
	config->Write((const wxString) str, (const wxString) str1);
}

void
MainFrame::LoadData(void)
{
	if(codes == NULL)
		delete codes;

	codes = new TCodes(dirs->GetMTUDataDir());
}

// -----------------------------------------------------------------------
void
MainFrame::DoAbout(void)
{
AboutDialog *dlg;

	dlg = new AboutDialog(this);
	dlg->ShowAbout("mtu");
	delete dlg;
}

void
MainFrame::DoSelect(void)
{
int i;
SectorList *s;

	// NOTE:  should only be here for 2D
	s = sectors->GetSectors();
	if(choose_dialog == NULL)
		choose_dialog = new ChooseDialog(this);
	if((i = choose_dialog->GetSectorChoice(s)) > -1) {
		SectorData *l_sd;

		l_sd = (SectorData *)s->NthData(i);
		if(l_sd != NULL) {
			sd = l_sd;
			dirs->CrackNames(sd->GetFileName());
			SetButtons(GetToolBar(), TRUE);
			UpdateStatus();
		}
	}
}

void
MainFrame::DoOpen(void)
{
int i;

	if(choose_dialog == NULL)
		choose_dialog = new ChooseDialog(this);
	if((i = choose_dialog->GetUniverseChoice(u_list)) > -1) {
		OpenU(i);
	}
}

#ifdef USE_NEW_FUNCTIONS
// XXX yikes!!!
void
MainFrame::DoNew(void)
{
bool d3;
char name[MAX_FILE_LENGTH],file_name[MAX_FILE_LENGTH];
char full_path[MAX_FILE_LENGTH],buffer[MAX_FILE_LENGTH * 2];
char lbuff[MAX_FILE_LENGTH];
char new_full_path[MAX_FILE_LENGTH];
char new_buffer[MAX_FILE_LENGTH * 2];
int x,y;
FILE *fp;
Universe3DInfo *info_3d;
UniverseInfo *info;
SectorInfo *si;
SectorCornerInfo *sci[4];

	info_3d = new Universe3DInfo();
	info = new UniverseInfo();
	// XXX put this back in:
	if(wxMessageBox("A greater amount of control is\n"
					"available for secgen from the\n"
					"command line.\n"
					"Also, you may need to restart\n"
					"this program before it will\n"
					"see the new universe.\n\n"
					"Do you wish to continue?",
					"Universe Creation Warning",
					wxOK | wxCANCEL | wxCENTRE |wxICON_EXCLAMATION) == wxCANCEL)
						break;
	name[0] = file_name[0] = 0;
	if(NULL == new_dialog)
		new_dialog = new NewDialog(this);
	if(new_dialog->GetNew(name, file_name, &d3)) {
		if(0 == name[0]) {
			wxMessageBox("No name given.\nCannot continue.",
				"Creation Error",
				wxOK | wxCENTRE | wxICON_EXCLAMATION);
			return;
		}

		if(0 == file_name[0])
			strcpy(file_name, name);

fprintf(stderr, "\n====> new:\n");
fprintf(stderr, "name: <%s>  file_name: <%s>\n", name, file_name);
		// common part of wxExec string
		sprintf(buffer, "%s -F%s/%s/", dirs->GetSecGen(),
			dirs->GetMTUDir(), file_name);

		sprintf(full_path, "%s/%s", dirs->GetMTUDir(), file_name);
fprintf(stderr, "mkdir: <%s>\n", full_path);
		if(!CreateDir("Universe", full_path))
			return;

		// 3D only makes one pass.  get the needed info, and run secgen
		if(d3) {
			if(NULL == d3_dialog)
				d3_dialog = new Create3DDialog(this);
			if(d3_dialog->GetSector(info_3d)) {
				char lbuff[MAX_FILE_LENGTH];

				strcat(full_path, "/");
				strcat(full_path, info_3d->GetFileName());
				if(!CreateDir("Sector", full_path))
					return;
fprintf(stderr, "----> 3D:\n");
fprintf(stderr, "mkdir: <%s>\n", full_path);
fprintf(stderr, "name: <%s>  file_name: <%s> lang: <%s>\n", info_3d->GetName(), 
info_3d->GetFileName(), info_3d->GetLangFile());
fprintf(stderr, "d: %d  m: %d  min (%d %d %d)  max (%d %d %d)\n",
info_3d->GetDensity(), info_3d->GetMaturity(), 
info_3d->GetMinX(), info_3d->GetMinY(), info_3d->GetMinZ(),
info_3d->GetMaxX(), info_3d->GetMaxY(), info_3d->GetMaxZ());

fprintf(stderr, "create %s/mtu.install:\n", full_path);
fprintf(stderr, "# install file\n");
fprintf(stderr, "DIR=%s\n", info_3d->GetFileName());
fprintf(stderr, "NAME=\"%s\"\n", info_3d->GetName());
fprintf(stderr, "D3=1\n");
				sprintf(lbuff, "%s/%s", info_3d->GetFileName(), info_3d->GetFileName());
				strcat(buffer, lbuff);
				sprintf(lbuff, " -3 -D%d -M%d ",
						info_3d->GetDensity(), info_3d->GetMaturity());
				strcat(buffer, lbuff);
				sprintf(lbuff, "-n%s ", info_3d->GetName());
				strcat(buffer, lbuff);
				sprintf(lbuff, "-x%d -y%d -z%d -X%d -Y%d -Z%d ",
					info_3d->GetMinX(), info_3d->GetMinY(), info_3d->GetMinZ(),
					info_3d->GetMaxX(), info_3d->GetMaxY(), info_3d->GetMaxZ());
				strcat(buffer, lbuff);
				if(info_3d->GetLangFile() != NULL) {
					sprintf(lbuff, "-L%s ", info_3d->GetLangFile());
					strcat(buffer, lbuff);
				}
fprintf(stderr, "<%s>\n", buffer);
			}
		} else {
		// 2D universe needs how many sectors (x and y)
		//   and presence and maturity flags for each
		//   with this, repeatedly call secgen
			if(NULL == d2_dialog)
				d2_dialog = new Create2DDialog(this);
			if(d2_dialog->GetSector(info)) {
fprintf(stderr, "----> 2D:\n");
fprintf(stderr, "name: <%s>  file_name: <%s> lang: <%s>\n", info->GetName(), 
info->GetFileName(), info->GetLangFile());
fprintf(stderr, "d: %d  m: %d  min (%d %d)  max (%d %d)\n",
info->GetDensity(), info->GetMaturity(), 
info->GetMinX(), info->GetMinY(), info->GetMaxX(), info->GetMaxY());
fprintf(stderr, "u: %d  st: %d  sy: %d  r: %d  z:%d\n",
info->GetUniform(), info->GetGenStars(), info->GetGenSystems(), info->GetGenRoutes(), info->GetGenZones());
				strcat(buffer, " ");
				if(!info->GetGenStars())
					strcat(buffer, "-S ");
				if(info->GetGenSystems())
					strcat(buffer, "-s ");
				if(info->GetGenRoutes())
					strcat(buffer, "-R ");
				if(info->GetGenZones())
					strcat(buffer, "-o ");

				strcpy(new_full_path, full_path);
				strcat(new_full_path, "/mtu.install");
				// XXX create mtu.install
fprintf(stderr, "create %s:\n", new_full_path);
fprintf(stderr, "# install file\n");
fprintf(stderr, "DIR=%s\n", info->GetFileName());
fprintf(stderr, "NAME=\"%s\"\n", info->GetName());
fprintf(stderr, "D3=0\n");
				// XXX open the universe .dat file
				//  and write header info
				strcpy(new_full_path, full_path);
fprintf(stderr, "create %s/%s.dat:\n", full_path, info->GetFileName());
fprintf(stderr, "%s%s\n",
"########################################",
"########################################");
fprintf(stderr, "## MTU Sectors for the %s universe\n", 
info->GetFileName());
				for(x = info->GetMinX();x <= info->GetMaxX();x++) {
					for(y = info->GetMinY();y <= info->GetMaxY();y++) {
						si = info->FindSectorInfo(x, y);
						if(si->GetDetail() == DT_NONE)
							continue;

fprintf(stderr, "to .dat file: %s:%s:%d:%d\n", si->GetFileName(), si->GetName(), si->GetX(), si->GetY());
						if(si->GetDetail() == DT_NAMED)
							continue;

						strcpy(new_buffer, buffer);
						strcpy(new_full_path, full_path);
						strcpy(new_full_path, "/");
						strcat(new_full_path, si->GetFileName());
						if(!CreateDir("Sector", new_full_path))
							continue;
fprintf(stderr, "mkdir: <%s>\n", new_full_path);
						strcat(new_full_path, si->GetFileName());
						sci[0] = info->FindSectorCorner(x, y);
						sci[1] = info->FindSectorCorner(x+1, y);
						sci[2] = info->FindSectorCorner(x, y+1);
						sci[3] = info->FindSectorCorner(x+1, y+1);
						if((sci[0]->GetDensity() == sci[1]->GetDensity()) &&
							(sci[0]->GetDensity() == sci[2]->GetDensity()) &&
							(sci[0]->GetDensity() == sci[3]->GetDensity()))
							sprintf(lbuff, "-D%d ", sci[0]->GetDensity());
						else
							sprintf(lbuff, "-d1%d -d2%d -d3%d -d4%d ",
									sci[0]->GetDensity(),
									sci[1]->GetDensity(),
									sci[2]->GetDensity(),
									sci[3]->GetDensity());
						strcat(new_buffer, lbuff);
						if((sci[0]->GetMaturity() == sci[1]->GetMaturity()) &&
							(sci[0]->GetMaturity() == sci[2]->GetMaturity()) &&
							(sci[0]->GetMaturity() == sci[3]->GetMaturity()))
							sprintf(lbuff, "-M%d ", sci[0]->GetMaturity());
						else
							sprintf(lbuff, "-m1%d -m2%d -m3%d -m4%d ",
								sci[0]->GetMaturity(), sci[1]->GetMaturity(),
								sci[2]->GetMaturity(), sci[3]->GetMaturity());
						strcat(new_buffer, lbuff);

						if(si->GetLangFile() != NULL) {
							sprintf(lbuff, "-L%s ", si->GetLangFile());
							strcat(new_buffer, lbuff);
						}
fprintf(stderr, "<%s>\n", new_buffer);
					}
				}
			}
		}
	}

	delete info_3d;
	delete info;
}
#endif

// -----------------------------------------------------------------------
void
MainFrame::OpenU(int ndx)
{
char title[MAX_FILE_LENGTH];
int x,y;
SectorList *sl;

//	if((sectors != NULL) && (sectors->Edited())) {
//		// save ourselves
//	}
	sectors = (UListData *) u_list->NthData(ndx);
	wxBeginBusyCursor();
	if(sectors != NULL) {
		sectors->ClearList();
	}
	sectors->LoadList(codes, dirs->GetMTUDir());
	dirs->CrackNames(sectors->GetFileName());
	sl = sectors->GetSectors();

	// set the canvas
	if(sectors->Get3D()) {
		Sector3D *s3;

		s3 = sectors->Get3DSector();
		view->CreateView(s3, 
			GetXRot(), GetYRot(), GetZRot());
		x = ((view->max_x - view->min_x) / DRAW_SIZE_X) + 1;
		y = ((view->max_y - view->min_y) / DRAW_SIZE_Y) + 1;
		dirs->CrackNames(s3->GetDirName());
	} else {
		x = sl->GetMaxX() - sl->GetMinX() + 1;
		y = sl->GetMaxY() - sl->GetMinY() + 1;
	}
	canvas->SetScrollbars(DRAW_SIZE_X, DRAW_SIZE_Y, x+1, y+1, 0, 0);

	// setup the user menu
	if(user_menu != NULL) {
		delete user_menu;
		user_menu = NULL;
	}

	// XXX this should have it's own in MTUFile
	sprintf(title, "%s/%s/menu/universe.menu", dirs->GetMTUDir(), 
					sectors->GetFileName());
	user_menu = new Menu(dirs, title, ID_USER2, ID_USER1);

	// set the title
	sprintf(title, "%s: %s", "My Traveller Universe", sectors->GetDesc());
	SetTitle(title);

	sd = NULL;
	star = NULL;
	SetButtons(GetToolBar(), TRUE);
	UpdateStatus();

	// finally, redraw the screen
	canvas->Refresh();
	wxEndBusyCursor();
}

// -----------------------------------------------------------------------
void
MainFrame::Draw(wxDC& dc)
{
	if(sectors != NULL) {
		if(sectors->Get3D())
			Draw3D(dc);
		else
			Draw2D(dc);
	}
}

void
MainFrame::Draw3D(wxDC& dc)
{
char buff[80];
ListNode *n;
View3DStar *v;
DetailStar *ds;
wxBrush *b;
wxPen *p;

//fprintf(stderr, "Max: (%d, %d, %d)\n", view->max_x, view->max_y, view->max_z);
//fprintf(stderr, "Min: (%d, %d, %d)\n", view->min_x, view->min_y, view->min_z);
	dc.SetFont(*star_font);
	n = view->First();
	while(n != NULL) {
		int i,x,y,rad;

		v = (View3DStar *) n->Data();

		x = v->x - view->min_x;
		y = v->y - view->min_y;
		rad = (((v->z - view->min_z) * 10) / (view->max_x - view->min_x)) + 1;
//fprintf(stderr, "---> (%d, %d, %d): %d\n", v->x, v->y, v->z, rad);

		// figure out the color to draw
		ds = v->star->GetStar(0);
		buff[0] = 0;
		ds->GetColor(buff);
		for(i = 0;i < (int) strlen(buff);i ++)
			buff[i] = toupper(buff[i]);

		b = new wxBrush(buff, wxSOLID);
		if(strcmp(buff, "WHITE") == 0)
			p = new wxPen("Black", 1, wxSOLID);
		else
   			p = new wxPen(buff, 1, wxSOLID);

		dc.SetPen(*p);
		dc.SetBrush(*b);

		// draw the star
		dc.DrawEllipse(DRAW_OFFSET + x, 
						DRAW_OFFSET + y, 
						rad, rad);
						
		dc.SetPen(*wxBLACK_PEN);
		dc.SetBrush(*wxTRANSPARENT_BRUSH);
		delete p;
		delete b;

		// add the desc
		if(GetFlags() & FLAG_SECT_3D_NAME) {
			ds->GetName(buff);
			dc.DrawText(buff, x + 10, y + 10);
		}

		n = view->Next();
	}
}

void
MainFrame::Draw2D(wxDC& dc, int min_x, int min_y, int max_x, int max_y)
{
int range_x,range_y,x,y,k;
SectorData *sd;
BaseSector *sect;
HexData *hd;
SectorList *sl;

	if((sl = sectors->GetSectors()) == NULL)
		return;

//fprintf(stderr, "(%d %d) (%d %d)\n", sl->GetMinX(), sl->GetMinY(),
//sl->GetMaxX(), sl->GetMaxY());
//fprintf(stderr, "(%d %d) (%d %d)\n", min_x, min_y, max_x, max_y);
	if(min_x < sl->GetMinX())
		min_x = sl->GetMinX();
	if(max_x > sl->GetMaxX())
		max_x = sl->GetMaxX();
	if(min_y < sl->GetMinY())
		min_y = sl->GetMinY();
	if(max_y > sl->GetMaxY())
		max_y = sl->GetMaxY();

	k = 0;
	while((sd = (SectorData *)sl->NthData(k++)) != NULL) {
		int i,j,hex_x,hex_y,color_ndx;
		wxPen *p=NULL;
		ColorTable *ct;

		if((sd->GetX() < min_x) || (sd->GetX() > max_x) ||
			(sd->GetY() < min_y) || (sd->GetY() > max_y))
			continue;

		x = DRAW_OFFSET + 
//			((sd->GetX() - sl->GetMinX()) * DRAW_SIZE_X);
			((sd->GetX() - min_x) * DRAW_SIZE_X);
		y = DRAW_OFFSET +
//			((sd->GetY() - sl->GetMinY()) * DRAW_SIZE_Y);
			((sd->GetY() - min_y) * DRAW_SIZE_Y);

		if((sect = sd->GetSector()) != NULL) {
			// draw routes
			if(GetFlags() & FLAG_SECT_ROUTES) {
				ct = sect->GetRouteTable();
				for(i = 0;i < MAX_X_HEX;i++) {
					for(j = 0;j < MAX_Y_HEX;j++) {
						if((hd = sect->GetHex(i, j)) != NULL) {
							int x1, y1;
							ListNode *n;
							Route *r;

							n = hd->routes->First();
							hex_x = x + (i * DRAW_MULT) + (DRAW_MULT / 2);
							hex_y = y + (j * DRAW_MULT) + (DRAW_MULT / 2);
							if(i & 1)
								hex_y += (DRAW_MULT / 2);
							while(n != NULL) {
								r = (Route *)n->Data();
								x1 = x + (r->x * DRAW_MULT) + (DRAW_MULT / 2);
								y1 = y + (r->y * DRAW_MULT) + (DRAW_MULT / 2);
								if(r->x & 1)
									y1 += (DRAW_MULT / 2);
	
								color_ndx = ct->GetColor(r->color);
								p = new wxPen(*colors->GetColor(color_ndx),
									1, wxSOLID);
								dc.SetPen(*p);
								dc.DrawLine(hex_x, hex_y, x1, y1);
								dc.SetPen(*wxBLACK_PEN);
								delete p;
								p = NULL;

								n = n->Next();
								}
							}
						}
					}
				}

			// draw borders
			if(GetFlags() & FLAG_SECT_BORDERS) {
				ct = sect->GetBorderTable();
				for(i = 0;i < MAX_X_HEX;i++) {
					for(j = 0;j < MAX_Y_HEX;j++) {
						if((hd = sect->GetHex(i, j)) != NULL) {
							int l;
	
							hex_x = x + (i * DRAW_MULT) + (DRAW_MULT / 2);
							hex_y = y + (j * DRAW_MULT) + (DRAW_MULT / 2);
							if(i & 1)
								hex_y += (DRAW_MULT / 2);
							for(l = 0;l < 6;l++) {
								color_ndx = hd->border_ndx[l];
								if((color_ndx > -1) && 
										(color_ndx < colors->GetMaxColors())) {
									int x1,x2,y1,y2;
	
									color_ndx = ct->GetColor(hd->border_ndx[l]);
									p = new wxPen(*colors->GetColor(color_ndx),
										1, wxSOLID);
									dc.SetPen(*p);
									switch(l) {
										case 0:
											x1 = hex_x;
											x2 = hex_x + DRAW_MULT;
											y1 = hex_y;
											y2 = hex_y;
											break;
										case 1:
											x1 = hex_x + DRAW_MULT;
											x2 = hex_x + DRAW_MULT;
											y1 = hex_y;
											y2 = hex_y + (DRAW_MULT / 2);
											break;
										case 2:
											x1 = hex_x + DRAW_MULT;
											x2 = hex_x + DRAW_MULT;
											y1 = hex_y + (DRAW_MULT / 2);
											y2 = hex_y + DRAW_MULT;
											break;
										case 3:
											x1 = hex_x;
											x2 = hex_x + DRAW_MULT;
											y1 = hex_y + DRAW_MULT;
											y2 = hex_y + DRAW_MULT;
											break;
										case 4:
											x1 = hex_x;
											x2 = hex_x;
											y1 = hex_y + DRAW_MULT;
											y2 = hex_y + (DRAW_MULT / 2);
											break;
										case 5:
											x1 = hex_x;
											x2 = hex_x;
											y1 = hex_y + (DRAW_MULT / 2);
											y2 = hex_y;
											break;
									}
									dc.DrawLine(x1, y1, x2, y2);
									dc.SetPen(*wxBLACK_PEN);
									delete p;
									p = NULL;
								}
							}
						}
					}
				}
			}

			// draw 'dot'
			for(i = 0;i < MAX_X_HEX;i++) {
				for(j = 0;j < MAX_Y_HEX;j++) {
					hex_x = x + (i * DRAW_MULT) + (DRAW_MULT / 2);
					hex_y = y + (j * DRAW_MULT) + (DRAW_MULT / 2);
					if(i & 1)
						hex_y += (DRAW_MULT / 2);
					
					// draw info
					if(htv != HTV_NONE) {
						char *ptr=NULL;
						int ndx;
						wxBrush *b;
						
						if(HTV_AREA == htv)
							ndx = sd->GetArea(i, j);
						else if(HTV_WAR == htv)
							ndx = sd->GetWar(i, j);

						switch(ndx) {
							case 1:
								ptr = "yellow";
								break;
							case 2:
								ptr = "orange";
								break;
							case 3:
								ptr = "red";
								break;
						}

						if(ptr != NULL) {
							b = new wxBrush(ptr, wxSOLID);
							p = new wxPen(ptr, 1, wxSOLID);

							dc.SetPen(*p);
							dc.SetBrush(*b);

							// draw the region
							dc.DrawLine(hex_x - (DRAW_MULT / 2), 
										hex_y - (DRAW_MULT / 2), 
										hex_x + (DRAW_MULT / 2), 
										hex_y + (DRAW_MULT / 2));
										
							dc.SetPen(*wxBLACK_PEN);
							dc.SetBrush(*wxTRANSPARENT_BRUSH);
							delete p;
							delete b;
						}
					}

					// now, draw the hex
					if(((hd = sect->GetHex(i, j)) != NULL) &&
						(hd->world != NULL)) {
//						dc.DrawLine(hex_x, hex_y, hex_x+4, hex_y+4);
						dc.DrawLine(hex_x, hex_y, hex_x, hex_y+1);
//						dc.DrawLine(hex_x+1, hex_y, hex_x+1, hex_y+1);
//fprintf(stderr, "(%d %d) -> (%d, %d)\n", i, j, x, y);
					}
				}
			}
		}

		// draw the name
		if(GetFlags() & FLAG_SECT_NAME) {
			char lbuff[20];
			dc.DrawText(sd->GetDesc(), 
				x + DRAW_OFFSET, y + (DRAW_SIZE_Y / 2) - DRAW_OFFSET);
			if(GetFlags() & FLAG_SECT_COORD) {
				sprintf(lbuff, "(%d %d)", sd->GetX(), sd->GetY());
				dc.DrawText(lbuff,
					x + DRAW_OFFSET, y + (DRAW_SIZE_Y / 2) - DRAW_OFFSET + 12);
			}

		}
	}

	// draw sector borders (the rectangles!)
//	range_x = sl->GetMaxX() - sl->GetMinX() + 1;
//	range_y = sl->GetMaxY() - sl->GetMinY() + 1;
	range_x = max_x - min_x + 1;
	range_y = max_y - min_y + 1;
//fprintf(stderr, "(%d %d) <-- (%d %d) (%d %d)\n", range_x, range_y,
//min_x, min_y, max_x, max_y);
	for(x = 0; x <= range_x; x++)
		dc.DrawLine((x * DRAW_SIZE_X) + DRAW_OFFSET,
					DRAW_OFFSET,
					(x * DRAW_SIZE_X) + DRAW_OFFSET,
					(range_y * DRAW_SIZE_Y) + DRAW_OFFSET);

	for(y = 0; y <= range_y; y++)
		dc.DrawLine(DRAW_OFFSET,
					(y * DRAW_SIZE_Y) + DRAW_OFFSET,
					(range_x * DRAW_SIZE_X) + DRAW_OFFSET,
					(y * DRAW_SIZE_Y) + DRAW_OFFSET);
}

void
MainFrame::SetRot(int x, int y, int z)
{
	x_rot = x;
	y_rot = y;
	z_rot = z;
	if((sectors != NULL) && (sectors->Get3D())) {
		Sector3D *s3;

		s3 = sectors->Get3DSector();
		view->CreateView(s3,
			GetXRot(), GetYRot(), GetZRot());
		x = ((view->max_x - view->min_x) / DRAW_SIZE_X) + 1;
		y = ((view->max_y - view->min_y) / DRAW_SIZE_Y) + 1;
		canvas->SetScrollbars(DRAW_SIZE_X, DRAW_SIZE_Y, x+1, y+1, 0, 0);
	}
}

void 
MainFrame::CalcPixelSizes(long *size_x, long *size_y, 
	int start_x, int start_y, int end_x, int end_y)
{
	*size_x = (end_x - start_x + 1) * DRAW_SIZE_X + (2 * DRAW_OFFSET);
	*size_y = (end_y - start_y + 1) * DRAW_SIZE_Y + (2 * DRAW_OFFSET);
}

void
MainFrame::CalcSectorSize(int *x, int *y)
{
	*x = DRAW_SIZE_X;
	*y = DRAW_SIZE_Y;
}

void
MainFrame::GetMax3DView(long *x, long *y)
{
	*x = ((view->max_x - view->min_x + 1));
	*y = ((view->max_y - view->min_y + 1));
}

#define HIT_THRESHOLD	10

void
MainFrame::DoSelection(int x, int y)
{
	if(sectors != NULL) {
		x -= DRAW_OFFSET;
		y -= DRAW_OFFSET;
		if(sectors->Get3D()) {
			ListNode *n;
			View3DStar *v;
			Star3D *s=NULL;

			x += view->min_x;
			y += view->min_y;
			n = view->First();
			while(n != NULL) {
				// NOTE: go through 'em all to get the best hit....
				v = (View3DStar *) n->Data();
				if((x >= (v->x - HIT_THRESHOLD)) && 
					(x <= (v->x + HIT_THRESHOLD)) && 
					(y >= (v->y - HIT_THRESHOLD)) && 
					(y <= (v->y + HIT_THRESHOLD))) {
					s = v->star;
				}
				n = view->Next();
			}
			if(s != NULL)
				star = s;
		} else {
			SectorList *sl;
			SectorData *sd1;
			ListNode *n;

			x = x / DRAW_SIZE_X;
			y = y / DRAW_SIZE_Y;
			sl = sectors->GetSectors();
			x += sl->GetMinX();
			y += sl->GetMinY();
			n = sl->First();
			while(n != NULL) {
				sd1 = (SectorData *)n->Data();
				if((sd1->GetX() == x) && (sd1->GetY() == y)) {
					sd = sd1;
					dirs->CrackNames(sd->GetFileName());
					break;
				}
				n = sl->Next();
			}
		}
		SetButtons(GetToolBar(), TRUE);
		UpdateStatus();
	}
}
// =======================================================================
BEGIN_EVENT_TABLE(MTUCanvas, wxScrolledWindow)
	EVT_LEFT_DOWN(MTUCanvas::OnLeftMouseDown)
	EVT_MIDDLE_DOWN(MTUCanvas::OnMouseDown)
	EVT_RIGHT_DOWN(MTUCanvas::OnMouseDown)
END_EVENT_TABLE()

MTUCanvas::MTUCanvas(wxFrame *frame) :
	wxScrolledWindow(frame, -1, wxPoint(0, 0), wxSize(480, 300), wxRETAINED)
{
}

void 
MTUCanvas::OnDraw(wxDC& dc)
{
	dc.Clear();
	frame->Draw(dc);
}

void 
MTUCanvas::OnMouseDown(wxMouseEvent &event)
{
int n_x,n_y;

	CalcUnscrolledPosition(event.GetX(), event.GetY(), &n_x, &n_y);
	frame->DoSelection(n_x, n_y);
}

void 
MTUCanvas::OnLeftMouseDown(wxMouseEvent &event)
{
Menu *m;
wxMenu *wx_m;

	if((m = frame->GetUserMenu()) != NULL)
		if((wx_m = m->GetMenu()) != NULL)
			PopupMenu(wx_m, event.GetX(), event.GetY());
}

// =======================================================================
View3DStar::View3DStar(Star3D *s, int n_x, int n_y, int n_z)
{
	star = s;
	x = n_x;
	y = n_y;
	z = n_z;
}

View3DStar::~View3DStar()
{
}

// -----------------------------------------------------------------------
View3D::View3D()
{
}

View3D::~View3D()
{
	Clear();
}

void
View3D::Clear(void)
{
ListNode *n;
View3DStar *v;

	max_x = max_y = max_z = min_x = min_y = min_z = 0;
	n = First();
	while(n != NULL) {
		v = (View3DStar *) n->Data();
		delete v;
		DeleteNode(n);
		n = First();
	}
}

// create a 'view'
//  just a list sorted on the adjusted z dimension
void
View3D::CreateView(Sector3D *sect, int x_rot, int y_rot, int z_rot)
{
int n_x,n_y,n_z,i,j,k;
double x[3][3];
double y[3][3];
double z[3][3];
double t[3][3];
double temp[3][3];
double sum;
double x_ang,y_ang,z_ang;
ListNode *n1,*n2;
View3DStar *v1,*v2;
Star3D *s;

	Clear();

	x_ang = ((double) x_rot * TWO_PI) / 360;
	y_ang = ((double) y_rot * TWO_PI) / 360;
	z_ang = ((double) z_rot * TWO_PI) / 360;

	x[0][0] = 1;
	x[0][1] = 0;
	x[0][2] = 0;
	x[1][0] = 0;
	x[1][1] = cos(x_ang);
	x[1][2] = -sin(x_ang);
	x[2][0] = 0;
	x[2][1] = sin(x_ang);
	x[2][2] = cos(x_ang);

	y[0][0] = cos(y_ang);
	y[0][1] = 0;
	y[0][2] = sin(y_ang);
	y[1][0] = 0;
	y[1][1] = 1;
	y[1][2] = 0;
	y[2][0] = -sin(y_ang);
	y[2][1] = 0;
	y[2][2] = cos(y_ang);

	z[0][0] = cos(z_ang);
	z[0][1] = -sin(z_ang);
	z[0][2] = 0;
	z[1][0] = sin(z_ang);
	z[1][1] = cos(z_ang);
	z[1][2] = 0;
	z[2][0] = 0;
	z[2][1] = 0;
	z[2][2] = 1;

	for(i = 0;i < 3;i++) {
		for(j = 0;j < 3;j++) {
			sum = 0;
			for(k = 0;k < 3;k++) { 
				sum += y[i][k] * x[k][j];
			}
			temp[i][j] = sum;
		}
	}
	for(i = 0;i < 3;i++) {
		for(j = 0;j < 3;j++) {
			sum = 0;
			for(k = 0;k < 3;k++) {
				sum += z[i][k] * temp[k][j];
			}
			t[i][j] = sum;
		}
	}

	n1 = sect->First();
	while(n1 != NULL) {
		bool found;
		double pt[3],new_pt[3];

		s = (Star3D *) n1->Data();
		
		pt[0] = (double) s->GetX();
		pt[1] = (double) s->GetY();
		pt[2] = (double) s->GetZ();

		for(j = 0;j < 3;j++) {
			sum = 0;
			for(k = 0;k < 3;k++) {
				sum += pt[k] * t[k][j];
			}
			new_pt[j] = sum;
		}  

		n_x = (int) new_pt[0];
		n_y = (int) new_pt[1];
		n_z = (int) new_pt[2];

//fprintf(stderr, "---> (%d, %d, %d)\n", n_x, n_y, n_z);
		// figure ranges
		if(n_x < min_x) min_x = n_x;
		if(n_x > max_x) max_x = n_x;
		if(n_y < min_y) min_y = n_y;
		if(n_y > max_y) max_y = n_y;
		if(n_z < min_z) min_z = n_z;
		if(n_z > max_z) max_z = n_z;

		// create the view entry
		v2 = new View3DStar(s, n_x, n_y, n_z);
		// now, place it
		found = FALSE;
		i = 0;
		n2 = First();
		while(n2 != NULL) {
			v1 = (View3DStar *) n2->Data();
//fprintf(stderr, " %d", v1->z);
			if(n_z < v1->z) {
//				Insert(v2, i);
				Insert(v2, n2);
				found = TRUE;
//fprintf(stderr, " I(%d)\n", i);
				break;
			}
			n2 = Next();
			i++;
		}

		if(!found) {
//fprintf(stderr, " A\n");
			Append(v2);
		}

//n2 = First();
//while(n2 != NULL) {
//	v1 = (View3DStar *) n2->Data();
//	fprintf(stderr, "%d ", v1->z);
//	n2 = Next();
//}
//fprintf(stderr, "\n");


		n1 = sect->Next();
	}
}

// =======================================================================
// kludge! (see mtu_dlg.cpp)
void main_frame_callback(unsigned long f, HARD_TIMES_VIEW v, 
			float s, int x, int y, int z)
{
	frame->SetFlags(f);
	frame->SetHTView(v);
	frame->SetScale(s);
	frame->SetRot(x, y, z);

	frame->Refresh();
}

// =======================================================================
// UGH!!!!  There is no way to determine the proper number of pages!!!!
//	For example:  18x12 total sectors
//	in landscape: 6x3 sectors/page are printed for 12 pages.
//	in portrait: 4x4 sectors/page are printed for 15 pages.
//  wxPrintData can change between creation time and print time
//	causing either to many pages printed, or not enough printed.
//	To top that off, calls such as GetPixelSize() return 0,0 until
//	print time......
//	
//	Should make huge estimates and make better use of HasPage() ?????
MTUPrintOut::MTUPrintOut(int s_x, int s_y, int e_x, int e_y, 
		wxPrintData *pd, bool sc) :
	wxPrintout()
{
	d3 = FALSE;
	start_x = s_x;
	start_y = s_y;
	end_x = e_x;
	end_y = e_y;
	scale = sc;
	if(scale)
		page_max = 1;
	else {
		int a_h,a_w,x,y;
		long size_x,size_y;

		// this is pretty much a SWAG.....
		if(pd->GetOrientation() == wxPORTRAIT) {
			a_h = 841 - 55;
			a_w = 595 - 35;
		} else {
			a_h = 595 - 35;
			a_w = 841 - 55;
		}

		frame->CalcPixelSizes(&size_x, &size_y, start_x, start_y, end_x, end_y);
		x = size_x / a_w;
		if(size_x % a_w) x++;
		y = size_y / a_h;
		if(size_y % a_h) y++;
		page_max = x * y;
	}
}

MTUPrintOut::MTUPrintOut() :
	wxPrintout()
{
	start_x = start_y = end_x = end_y = 0;
	scale = TRUE;
	d3 = TRUE;
	page_max = 1;
}
MTUPrintOut::~MTUPrintOut()
{
}

bool 
MTUPrintOut::OnPrintPage(int page)
{
wxDC *dc;

	dc = GetDC();
	if(dc) {
		int w,h,a_h,a_w;
		long size_x,size_y;

		GetPageSizePixels(&w, &h);

		// this seems required for my printer.....
		a_h = h;// - 55;
		a_w = w;// - 35;

		if(!d3)
			frame->CalcPixelSizes(&size_x, &size_y, 
					start_x, start_y, end_x, end_y);
		else {
			frame->GetMax3DView(&size_x, &size_y);
		}

		if(scale) {
			float scale,scale_x,scale_y;

			scale_x = (float) a_w / (float) size_x;
			scale_y = (float) a_h / (float) size_y;
			scale = wxMin(scale_x, scale_y);
			dc->SetUserScale(scale, scale);
			if(!d3) {
				frame->Draw2D(*dc, start_x, start_y, end_x, end_y);
			} else {
				frame->Draw3D(*dc);
				// ????
			}
		} else {
			int x_sect_page,y_sect_page;
			int x_page,y_page,s_x,s_y;

			frame->CalcSectorSize(&x_sect_page, &y_sect_page);
			x_page = a_w / x_sect_page;
			y_page = a_h / y_sect_page;

			s_x = s_y = 0;
			page--;
			while(page > 0) {
				s_x += x_page;
				if(s_x >= (end_x - start_x + 1)) {
					s_x = 0;
					s_y += y_page;
				}
				page--;
			}
			if(s_y >= (end_y - start_y + 1))
				return(FALSE);

			s_x += start_x;
			s_y += start_y;

			frame->Draw2D(*dc, s_x, s_y, s_x + x_page - 1, s_y + y_page - 1);
		}
		return(TRUE);
	}
	return(FALSE);
}

bool 
MTUPrintOut::HasPage(int page)
{
	if(page <= page_max)
		return(TRUE);
	return(FALSE);
}

void 
MTUPrintOut::GetPageInfo(int *min, int *max, int *sel_from, int *sel_to)
{
	*min = *sel_from = 1;
	*max = *sel_to = page_max;
}


// =======================================================================
// this is in here 'cause we need the Draw() method
MTUBitmap::MTUBitmap(int width, int height) :
		wxBitmap(width, height)
{
}

MTUBitmap::~MTUBitmap()
{
}

void
MTUBitmap::Clear(wxDC &dc)
{
int w,h;
double s_w,s_h;

	// why isn't ther a @%#$@ Clear() ????
	dc.GetSize(&w, &h);
	dc.GetUserScale(&s_w, &s_h);
	w = (int) (w / s_w) + 1;
	h = (int) (h / s_h) + 1;
	dc.SetBrush(wxBrush(*frame->GetColor(WHITE_COLOR_INDEX), wxSOLID));
	dc.SetPen(*wxWHITE_PEN);
	dc.DrawRectangle( 0, 0, w, h);
}

void
MTUBitmap::Draw(wxDC &dc, int start_x, int start_y, int end_x, int end_y)
{
	Clear(dc);
	frame->Draw2D(dc, start_x, start_y, end_x, end_y);
}

void
MTUBitmap::Draw(wxDC &dc)
{
	Clear(dc);
	frame->Draw3D(dc);
}

// =======================================================================
// =======================================================================
