/*
 * File:      uwp_dlg.cpp
 * Purpose:   uwp dialog related functions
 * Author:    Mark A. Nordstrand
 * Created:   1/23/2000
 * Updated:
 * Copyright: LGPL
Traveller is a registered trademark of Far Future Enterprises.
Portions based upon material Copyright 1977-2002 Far Future Enterprises.
 */

/* rcsid[] = "$RCSfile: uwp_dlg.cpp,v $ $Revision: 1.12 $ $Author: man $ $Date: 2003/11/02 01:55:11 $" */

#include "uwp_dlg.h"
#include "t_dlg.h"


// ===========================================================================
#define DUMMY_BUFFER_SIZE			120

// control id's
#define ID_PORT_CH		600
#define ID_SIZE_CH		601
#define ID_ATMS_CH		602
#define ID_HYDR_CH		603
#define ID_POP_CH		604
#define ID_GOVT_CH		605
#define ID_LAW_CH		606
#define ID_TECH_CH		607
#define ID_VFY_BTN		608
#define ID_OK_BTN		611
#define ID_RESET_BTN	612
#define ID_UNDO_BTN		613

#define NUM_PORT_STR        10
#define NUM_SIZE_STR        13
#define NUM_ATMOS_STR       16
#define NUM_HYDRO_STR       11
#define NUM_POP_STR         12
#define NUM_GOVT_STR        17
#define NUM_LAW_STR         22

#define MAX_UWP_STRINGS		26

// ---------------------------------------------------------------------------
// common strings used
static char general_str[MAX_UWP_STRINGS][4] = {
	" 0 ",
	" 1 ",
	" 2 ",
	" 3 ",
	" 4 ",
	" 5 ",
	" 6 ",
	" 7 ",
	" 8 ",
	" 9 ",
	" A ",
	" B ",
	" C ",
	" D ",
	" E ",
	" F ",
	" G ",
	" H ",
	" J ",
	" K ",
	" L ",
	" R ",
	" S ",
	" X ",
    " Y ",
    "   " };

static wxString dummy_str[MAX_UWP_STRINGS];

static wxString port_str[NUM_PORT_STR] = {
  general_str[10],
  general_str[11],
  general_str[12],
  general_str[13],
  general_str[14],
  general_str[15],
  general_str[16],
  general_str[17],
  general_str[23],
  general_str[24] };

static wxString size_str[NUM_SIZE_STR] = {
  general_str[0],
  general_str[1],
  general_str[2],
  general_str[3],
  general_str[4],
  general_str[5],
  general_str[6],
  general_str[7],
  general_str[8],
  general_str[9],
  general_str[10],
  general_str[22],
  general_str[21] };

static wxString atmos_str[NUM_ATMOS_STR] = {
  general_str[0],
  general_str[1],
  general_str[2],
  general_str[3],
  general_str[4],
  general_str[5],
  general_str[6],
  general_str[7],
  general_str[8],
  general_str[9],
  general_str[10],
  general_str[11],
  general_str[12],
  general_str[13],
  general_str[14],
  general_str[15] };

static wxString hydro_str[NUM_HYDRO_STR] = {
  general_str[0],
  general_str[1],
  general_str[2],
  general_str[3],
  general_str[4],
  general_str[5],
  general_str[6],
  general_str[7],
  general_str[8],
  general_str[9],
  general_str[10] };

static wxString pop_str[NUM_POP_STR] = {
  general_str[25],
  general_str[0],
  general_str[1],
  general_str[2],
  general_str[3],
  general_str[4],
  general_str[5],
  general_str[6],
  general_str[7],
  general_str[8],
  general_str[9],
  general_str[10] };

static wxString govt_str[NUM_GOVT_STR] = {
  general_str[25],
  general_str[0],
  general_str[1],
  general_str[2],
  general_str[3],
  general_str[4],
  general_str[5],
  general_str[6],
  general_str[7],
  general_str[8],
  general_str[9],
  general_str[10],
  general_str[11],
  general_str[12],
  general_str[13],
  general_str[14],
  general_str[15] };

static wxString law_str[NUM_LAW_STR] = {
  general_str[25],
  general_str[0],
  general_str[1],
  general_str[2],
  general_str[3],
  general_str[4],
  general_str[5],
  general_str[6],
  general_str[7],
  general_str[8],
  general_str[9],
  general_str[10],
  general_str[11],
  general_str[12],
  general_str[13],
  general_str[14],
  general_str[15],
  general_str[16],
  general_str[17],
  general_str[18],
  general_str[19],
  general_str[20]
};

static wxString type_str[6] = {
	" B ",
	" A ",
	" F ",
	" G ",
	" K ",
	" M " 
};

static wxString class_str[10] = {
  general_str[0],
  general_str[1],
  general_str[2],
  general_str[3],
  general_str[4],
  general_str[5],
  general_str[6],
  general_str[7],
  general_str[8],
  general_str[9]
};

static wxString ssize_str[8] = {
	" Ia  ",
	" Ib  ",
	" II  ",
	" III ",
	" IV  ",
	" V   ",
	" VI  ",
	" VII "
};

// common strings used
static char *uwp_desc[8] = {
  "Port",
  "Size",
  "Atmos",
  "Hydro",
  "Pop",
  "Govt",
  "Law",
  "Tech" };

// ---------------------------------------------------------------------------
UWPBox::UWPBox(wxWindow *p, char *title, int base_id) :
	wxStaticBox(p, -1, title)
{
int i,id;
wxFlexGridSizer *fs;
wxString *ptr;

	if(base_id < 0)
		id = ID_PORT_CH;
	else
		id = base_id;
	err_dlg = new ErrorDialog(this);
	if(NULL == title)
		SetTitle("UWP");

	msg[0] = new wxStaticText(p, -1, "Port:");
	msg[1] = new wxStaticText(p, -1, "Size:");
	msg[2] = new wxStaticText(p, -1, "Atmos:");
	msg[3] = new wxStaticText(p, -1, "Hydro:");
	msg[4] = new wxStaticText(p, -1, "Pop:");
	msg[5] = new wxStaticText(p, -1, "Govt:");
	msg[6] = new wxStaticText(p, -1, "Law:");
	msg[7] = new wxStaticText(p, -1, "Tech:");

	for(i = 0; i < MAX_UWP_CHOICES;i++) {
		int max;

		l_uwp[i] = NULL;
		// default to any possible string
		switch(i) {
			case 0:
				ptr = port_str;
				max = NUM_PORT_STR;
			    break;
			case 1:
				ptr = size_str;
				max = NUM_SIZE_STR;
			    break;
			case 2:
				ptr = atmos_str;
				max = NUM_ATMOS_STR;
			    break;
			case 3:
				ptr = hydro_str;
				max = NUM_HYDRO_STR;
			    break;
			case 4:
				ptr = pop_str;
				max = NUM_POP_STR;
			    break;
			case 5:
				ptr = govt_str;
				max = NUM_GOVT_STR;
			    break;
			case 6:
			case 7:
				ptr = law_str;
				max = NUM_LAW_STR;
			    break;
		}

		l_uwp[i] = new wxChoice(p, id + i,
			wxPoint(-1, -1),
			wxSize((int) (COL_OFFSET * 2), -1),
			max, ptr);
		l_uwp[i]->SetSelection(0);		
	}

	sizer = new wxStaticBoxSizer(this, wxHORIZONTAL);
	fs = new wxFlexGridSizer(3, 2, 0, 0);
	for(i = 0;i < MAX_UWP_CHOICES;i++) {
		fs->Add(msg[i], 0, wxALL | wxALIGN_BOTTOM, DLG_OFFSET / 2);
    	fs->Add(l_uwp[i], 0, wxALL, DLG_OFFSET / 2);
	}
	sizer->Add(fs, 0, 0, 0);
}

UWPBox::~UWPBox()
{
}

// ---------------------------------------------------------------------------
void
UWPBox::SetUWP(char *buff, UWP_IO_TYPE t)
{
	io_type = t;
	InitUwpChoices(t);
	strcpy(orig_uwp, buff);

	StrToChoice(buff);
}

void
UWPBox::StrToChoice(char *buff)
{
int i=0,j=0;

	while((buff[i] != 0) && (j < MAX_UWP_CHOICES)) {
		if(isalnum(buff[i])) {
			SetDigit(buff[i], l_uwp[j]);
			j++;
		}
		i++;
	}
}

char *
UWPBox::GetUWP(char *buff)
{
	sprintf(buff, "%c%c%c%c%c%c%c%c", GetDigit(0), GetDigit(1), GetDigit(2),
		GetDigit(3), GetDigit(4), GetDigit(5), GetDigit(6), GetDigit(7));
	return(buff);
}

bool
UWPBox::GetUWP(uwp *p)
{
	p->SetPort(GetDigit(0));
	p->SetSize(GetDigit(1));
	p->SetAtmos(GetDigit(2));
	p->SetHydro(GetDigit(3));
	p->SetPop(GetDigit(4));
	p->SetGovt(GetDigit(5));
	p->SetLaw(GetDigit(6));
	p->SetTech(GetDigit(7));

	return(TRUE);
}

bool
UWPBox::SetDigit(char c, wxChoice *ch)
{
int i;
char *ptr;

	for(i = 0;i < ch->GetCount();i++) {
		ptr = (char *)ch->GetString(i).GetData();
		if(toupper(c) == ptr[1]) {
			ch->SetSelection(i);
			return(FALSE);
		}
	}

	return(FALSE);
}

char 
UWPBox::GetDigit(int ndx)
{
int sel;
char *ptr;

	sel = l_uwp[ndx]->GetSelection();
	ptr = (char *)l_uwp[ndx]->GetString(sel).GetData();
	return(ptr[1]);
}

bool 
UWPBox::Enable(bool val)
{
bool stat;
int i;

	stat = wxStaticBox::Enable(val);
	for(i = 0;i < MAX_UWP_CHOICES;i++) {
		msg[i]->Enable(val);
		l_uwp[i]->Enable(val);
	}

	return(stat);
}

void
UWPBox::InitUwpChoices(UWP_IO_TYPE t)
{
int i,j;

	// start w/ default
	for(i = 0;i < MAX_UWP_CHOICES;i++) {
		int max,start,end,k;

		l_uwp[i]->Clear();
		switch(i) {
			case 0:
				if(UI_SECTOR == t) {
					max = 6;
					dummy_str[0] = port_str[0];
					dummy_str[1] = port_str[1];
					dummy_str[2] = port_str[2];
					dummy_str[3] = port_str[3];
					dummy_str[4] = port_str[4];
					dummy_str[5] = port_str[8];
				} else {
					max = NUM_PORT_STR;
					for(j = 0;j < NUM_PORT_STR;j++)
						dummy_str[j] = port_str[j];
				}
			    break;
			case 1:
				if(UI_SECTOR == t) {
					start = 0;
					end = 11;
				} else if(UI_SYS_PLANET == t) {
					start = 1;
					end = 12;
				} else if(UI_SYS_SAT == t) {
					start = 1;
					end = 13;
				} else if(UI_BELT == t) {
					start = 0;
					end = 1;
				} else {
					start = 0;
					end = 13;
				}
				max = 0;
				for(j = start,k = 0;j < end;j++,k++) {
					dummy_str[k] = size_str[j];
					max++;
				}
			    break;
			case 2:
				if(t != UI_BELT) {
					for(j = 0;j < NUM_ATMOS_STR;j++)
						dummy_str[j] = atmos_str[j];
					max = NUM_ATMOS_STR;
				}
			    break;
			case 3:
				if(t != UI_BELT) {
					for(j = 0;j < NUM_HYDRO_STR;j++)
						dummy_str[j] = hydro_str[j];
					max = NUM_HYDRO_STR;
				}
			    break;
			case 4:
				max = 0;
				if(t != UI_SECTOR) {
					for(j = 0;j < NUM_POP_STR;j++) {
						dummy_str[j] = pop_str[j];
						max++;
					}
				} else {
					for(j = 1;j < NUM_POP_STR;j++) {
						dummy_str[j - 1] = pop_str[j];
						max++;
					}
				}
			    break;
			case 5:
				max = 0;
				if(t != UI_SECTOR) {
					for(j = 0;j < NUM_GOVT_STR;j++) {
						dummy_str[j] = govt_str[j];
						max++;
					}
				} else {
					for(j = 1;j < NUM_GOVT_STR;j++) {
						dummy_str[j - 1] = govt_str[j];
						max++;
					}
				}
			    break;
			case 6:
			case 7:
				max = 0;
				if(t != UI_SECTOR) {
					for(j = 0;j < NUM_LAW_STR;j++) {
						dummy_str[j] = law_str[j];
						max++;
					}
				} else {
					for(j = 1;j < NUM_LAW_STR;j++) {
						dummy_str[j - 1] = law_str[j];
						max++;
					}
				}
			    break;
		}

		for(j = 0;j < max;j++)
			l_uwp[i]->Append(dummy_str[j]);
		l_uwp[i]->SetSelection(0);		

		if((UI_BELT == t) && (i > 0) && (i < 4))
			l_uwp[i]->Enable(FALSE);
		else
			l_uwp[i]->Enable(TRUE);
	}
}

// ---------------------------------------------------------------------------
// real simple error checking
// this gets a bit hokey.....
//	This retrofit of main worlds (worlds from a sector file) onto a
//	Book 6 system causes some real headaches here.  The best that
//  can be done is to first trap the type of check (UWP_IO_TYPE)
//  being done and then look at the port type and assume that if a 
//	starport, this was a main world.
unsigned short
UWPBox::ErrorCheck(char *user_uwp, char *adj_uwp)
{
unsigned short ret=0;
int uwp_vals[MAX_UWP_CHOICES],i;

	// convert all but port to ints (makes life simpler)
	//  use -1 for S and -2 for R
	uwp_vals[0] = user_uwp[0];
	if((user_uwp[1] >= '0') || (user_uwp[1] <= '9'))
		uwp_vals[1] = user_uwp[1] - '0';
	else if('A' == user_uwp[1])
		uwp_vals[1] = user_uwp[1] - 'A' + 10;
	else if('S' == user_uwp[1])
		uwp_vals[1] = -1;
	else if('R' == user_uwp[1])
		uwp_vals[1] = -2;
	else
		uwp_vals[1] = -10;				// the infamous can't happen
	for(i = 2;i < MAX_UWP_CHOICES;i++) {
		if(user_uwp[i] < 'A')
			uwp_vals[i] = user_uwp[i] - '0';
		else
			uwp_vals[i] = user_uwp[i] - 'A' + 10;
	}

	switch(io_type) {
		case UI_ANY:
			ret = CheckAny(uwp_vals);
			break;
		case UI_SECTOR:
			ret = CheckWorld(uwp_vals);
			break;
		case UI_SYS_PLANET:
		case UI_SYS_SAT:
		case UI_BELT:
			ret = CheckPlanet(uwp_vals);
			break;
//		case UI_SYS_SAT:
//			ret = CheckSat(uwp_vals);
//			break;
	}

	if(ret != 0) {
		adj_uwp[0] = (char) uwp_vals[0];
		if(uwp_vals[1] < -1) adj_uwp[1] = 'R';
		else if(uwp_vals[1] < 0) adj_uwp[1] = 'S';
		else if(uwp_vals[1] < 10) adj_uwp[1] = (char) uwp_vals[1] + '0';
		else adj_uwp[1] = (char) uwp_vals[1] + 'A' - 10;
		for(i = 2;i < MAX_UWP_CHOICES;i++) {
			if(uwp_vals[i] < 0) adj_uwp[i] = ' ';
			else if(uwp_vals[i] < 10) adj_uwp[i] = uwp_vals[i] + '0';
			else adj_uwp[i] = uwp_vals[i] + 'A' - 10;
		}
	}

	return(ret);
}

unsigned short
UWPBox::CheckAny(int *u)
{
	return(0);
}

// check an array of 'main world' uwp's
unsigned short
UWPBox::CheckWorld(int *u)
{
short ret=0;
int min,max;

	// the infamous shouldn't happen case:
	if(u[1] < -1) {
		ret = CheckRing(u);
	} else {
		// atmos
		if(0 == u[1]) {				// astroid belt
			ret |= CheckBelt(u);
		} else {
			if(-1 == u[1]) {		// size small
				min = 0;
				max = 5;
			} else {
				min = 2 - 7 + 10;
				max = 12 - 7 + 10;
			}
			if(!CheckUWP(&u[2], min, max))
				ret |= UWP_MASK_ATMOS;

			// hydro
			min = 2 - 7 + u[2];
			max = 12 - 7 + u[2];
			if((u[2] < 2) || (u[2] > 9)) {
				min -= 4;
				max -= 4;
			}
			if(!CheckUWP(&u[3], min, max))
				ret |= UWP_MASK_HYDRO;
		}

		// general pop check
		ret |= CheckNoPop(u);
		if(u[4] > -1) {
			// check govt
			if(!CheckUWP(&u[5], 2 - 7 + u[4], 12 - 7 + u[4]))
				ret |= UWP_MASK_GOVT;

			// check law
			if(!CheckUWP(&u[6], 2 - 7 + u[5], 12 - 7 + u[5]))
				ret |= UWP_MASK_LAW;

			// check tech
			min = 1;
			max = 6;
			if(u[0] == 'A') {
				min += 6;
				max += 6;
			}
			if(u[0] == 'B') {
				min += 4;
				max += 4;
			}
			if(u[0] == 'C') {
				min += 2;
				max += 2;
			}
			if(u[0] == 'X') {
				min -= 4;
				max -= 4;
			}
			if(u[1] < 2) {
				min += 2;
				max += 2;
			} else if(u[1] < 5) {
				min += 1;
				max += 1;
			}
			if((u[2] < 4) || (u[2] > 9)) {
				min += 1;
				max += 1;
			}
			if(u[3] == 9) {
				min += 1;
				max += 1;
			}
			if(u[3] == 10) {
				min += 2;
				max += 2;
			}
			if(u[4] < 6) {
				min += 1;
				max += 1;
			}
			if(u[4] == 9) {
				min += 2;
				max += 2;
			}
			if(u[4] == 10) {
				min += 4;
				max += 4;
			}
			if((u[5] == 0) || (u[5] == 5)) {
				min += 1;
				max += 1;
			}
			if(u[5] == 13) {
				min -= 2;
				max -= 2;
			}
			if(!CheckUWP(&u[7], min, max))
				ret |= UWP_MASK_TECH;
		}
	}
   
	return(ret);
}

// check an array of 'planet' uwp's
// returns 0 if nothing was changed
// else bitwise indication of anything changed.
// NOTE:
//    size 0 (asteroid belt) is never checked and
//    size R and S are temporarly 0 and -1
// NOTE2:
//    I've departed from the books a little bit
//    Aside from the use of ' ' for _no_ population
//    (10^0 = 1 => 1-9 people NOT 0 people),
//    I've tried to make satillites and worlds
//    more consistant.
unsigned short
UWPBox::CheckPlanet(int *u)
{
short min,max;
short ret = 0;

	if((UI_SYS_PLANET == io_type) && (-2 == u[1])) {
		u[1] = 0;
		ret |= UWP_MASK_SIZE;
	} else if((UI_SYS_SAT == io_type) && (0 == u[1])) {
		u[1] = -2;
		ret |= UWP_MASK_SIZE;
	}

	if(u[1] < -1)
		return(ret | CheckRing(u));

	if(0 == u[1]) {			// astroid belt
		ret = CheckBelt(u);
	} else {
		min = 0;
		max = 10;
		if(UI_SYS_SAT == io_type) {
			if(parent_size == SIZE_LARGE_GG) {
				min = -2;
				max = 8;
			} else if(parent_size == SIZE_SMALL_GG) {
				min = -4;
				max = 6;
			} else if(parent_size == SIZE_SMALL) {
				min = -1;
				max = 0;
			} else {
				min = parent_size - 6;
				max = parent_size - 1;
			}
		} else {
			if(star_type == 6)
				max -= 2;
			if(orb <= 0.5)
				max -= 5;
			else if(orb <= 1.5)
				max -= 4;
			else if(orb <= 2.5)
				max -= 2;
		}
		if(!CheckUWP(&u[1], min, max)) {
			if(u[1] < 1) u[1] = -1;
			ret |= UWP_MASK_SIZE;
		}
		// check atmos
		if(u[1] > 10)
			min = max = 0;
		else {
			min = 2 - 7 + u[1];
			max = 12 - 7 + u[1];
			if(zone < HAB_HABIT) {
				min -= 2;
				max -= 2;
			}
			if(zone > HAB_HABIT) {
				min -= 4;
				max -= 4;
			}
		}
	   // &#^$&*^$ special case!
	   if(!((zone > HAB_OUTER) && (u[2] == 0) && (u[2] == 10))) {
		   if(!CheckUWP(&u[2], min, max))
			   ret |= UWP_MASK_ATMOS;
	   }

	   // check hydro
	   if((u[1] > 10) || (zone < HAB_HABIT))
			min = max = 0;
	   else {
			min = 2 - 7 + u[1];
			max = 12 - 7 + u[1];
			if(zone > HAB_HABIT) {
				min -= 2;
				max -= 2;
			}
			if(u[2] > 9) {
				min -= 4;
				max -= 4;
			}
		}
		if(min < 0)
			min = 0;
		if(max > 0)
			max = 0;
	   
		if(!CheckUWP(&u[3], min, max))
			ret |= UWP_MASK_HYDRO;	   
	}

	ret |= CheckNoPop(u);

	if(u[4] > -1) {
		// check pop
		min = 0;
		max = 10;
		if(zone < HAB_HABIT) {
			min -= 5;
			max -= 5;
		}
		if(zone > HAB_HABIT) {
			min -= 3;
			max -= 3;
		}
		if((u[2] != 0) && (u[2] != 5) && (u[2] != 6) && (u[2] != 8)) {
			min -= 2;
			max -= 2;
		}
		if(!CheckUWP(&u[4], min, max))
			ret |= UWP_MASK_POP;

		// check govt
		if((u[5] > 6) || (u[5] == 4) || (u[5] == 5)) {
			u[5] = 6;
			ret |= UWP_MASK_GOVT;
		}
		// by Book 6, there is no real way to check law and tl
		//  (altough ridiculas values less than 0 are checked
	}

   // check port
   min = 1;
   max = 6;
   if(u[4] > 5) {
	   min += 2;
	   max += 2;
   }
   if(u[4] < 0) {
	   min -= 3;
	   max -= 3;
   }
   else if(u[4] < 2) {
	   min -= 2;
	   max -= 2;
   }
   switch(min) {
	   case 3:
		   if(u[0] > 'H') {
			   u[0] = 'H';
			   ret |= UWP_MASK_PORT;
		   }
		   break;
	   case 4:
	   case 5:
		   if(u[0] > 'G') {
			   u[0] = 'G';
			   ret |= UWP_MASK_PORT;
		   }
		   break;
	   case 6:
	   case 7:
	   case 8:
		   if(u[0] != 'F') {
			   u[0] = 'F';
			   ret |= UWP_MASK_PORT;
		   }
		   break;
	   default:
		   break;
	}
    switch(max) {
	   case -2:
	   case -1:
	   case 0:
	   case 1:
	   case 2:
		   if(u[0] != 'Y') {
				u[0] = 'Y';
				ret |= UWP_MASK_PORT;
		    }
		   break;
	   case 3:
		   if(u[0] < 'X') {
			   u[0] = 'X';
			   ret |= UWP_MASK_PORT;
		   }
		   break;
	   case 4:
	   case 5:
		   if(u[0] < 'H') {
			   u[0] = 'H';
			   ret |= UWP_MASK_PORT;
		   }
		   break;
	   default:
		   break;
	}
	 
	return(ret);
}

// check atmos and hydro are 0
unsigned short
UWPBox::CheckBelt(int *u)
{
unsigned short ret=0;

	if(u[2] != 0) {
		u[2] = 0;
		ret |= UWP_MASK_ATMOS;
	}
	if(u[3] != 0) {
		u[3] = 0;
		ret |= UWP_MASK_HYDRO;
	}

	return(ret);
}

// nice simple case: rings (for systems only)
unsigned short
UWPBox::CheckRing(int *u)
{
int ret = 0;

	if(u[2] != 0) {
		u[2] = 0;
		ret |= UWP_MASK_ATMOS;
	}
	if(u[3] != 0) {
		u[3] = 0;
		ret |= UWP_MASK_HYDRO;
	}
	if(u[4] != -1) {
		u[4] = -1;
		ret |= UWP_MASK_POP;
	}
	if(u[5] != -1) {
		u[5] = -1;
		ret |= UWP_MASK_GOVT;
	}
	if(u[6] != -1) {
		u[6] = -1;
		ret |= UWP_MASK_LAW;
	}
	if(u[7] != -1) {
		u[7] = -1;
		ret |= UWP_MASK_TECH;
	}

	return(ret);
}

// just makes sure the pop, govt, law and tech are consistantly
//  all -1 or not -1 (for systems only)
unsigned short
UWPBox::CheckNoPop(int *u)
{
short ret=0;
short mask=UWP_MASK_GOVT;
int i;

    if(u[4] < 0) {
		for(i = 5; i < 8;i++) {
	  		if(u[i] > -1) {
				u[i] = -1;
				ret |= mask;
			}
			mask *= 2;
		}
	} else {
		for(i = 5; i < 8;i++) {
	  		if(u[i] < 0) {
				u[i] = 0;
				ret |= mask;
			}
			mask *= 2;
		}
	}

	return(ret);
}

// generic check of val against min and max values
bool
UWPBox::CheckUWP(int *val, int min, int max)
{
bool ret=TRUE;

    if(*val < min) {
		*val = min;
		ret = FALSE;
	}
	if(*val > max) {
		*val = max;
		ret = FALSE;
	}

	return(ret);
}

void 
UWPBox::CheckUWP(void)
{
char user_uwp[MAX_UWP_CHOICES+1],adj_uwp[MAX_UWP_CHOICES+1];
unsigned short vfy;

	GetUWP(user_uwp);
	GetUWP(adj_uwp);

	if((vfy = ErrorCheck(user_uwp, adj_uwp)) != 0) {
		ERROR_RESP er;
		char buff[DUMMY_BUFFER_SIZE];
		unsigned short mask=UWP_MASK_PORT;
		int i;
		bool comma=FALSE;

		buff[0] = 0;
		for(i = 0; i < MAX_UWP_CHOICES;i++) {
			if(mask & vfy) {
				if(comma)
				    strcat(buff, ", ");
				strcat(buff, uwp_desc[i]);
				comma = TRUE;
			}
			mask *= 2;
		}
		err_dlg->info->SetLabel(buff);

		er = err_dlg->GetResp();
		if(ER_OK == er) {
			// set controls to adj_uwp
			StrToChoice(adj_uwp);
		} else if(ER_RESET == er) {
			// set controls to original
			StrToChoice(orig_uwp);
		} else {
			// leave controls as user_uwp ie. do nothing
		}
	} else {
		wxMessageBox("UWP has been verified.", "UWP verify.",
					 wxOK | wxCENTRE | wxICON_INFORMATION, this);
	}
}

// ===============================================================
BEGIN_EVENT_TABLE(ErrorDialog, wxDialog)
	EVT_BUTTON(ID_OK_BTN, ErrorDialog::DoOk)
	EVT_BUTTON(ID_RESET_BTN, ErrorDialog::DoReset)
	EVT_BUTTON(ID_UNDO_BTN, ErrorDialog::DoUndo)
END_EVENT_TABLE()

// ---------------------------------------------------------------
ErrorDialog::ErrorDialog(wxWindow *p) :
    wxDialog(p, -1, "UWP Error!")
{
wxStaticText *msg[2];
wxButton *btn[3];
wxBoxSizer *bs[2];

    msg[0] = new wxStaticText(this, -1, "Error(s) in UWP specification!");
	msg[1] = new wxStaticText(this, -1, "Changes have been made to:");
	info = new wxStaticText(this, -1, "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX");
    btn[0] = new wxButton(this, ID_OK_BTN, "Ok");
    btn[1] = new wxButton(this, ID_RESET_BTN, "Reset");
    btn[2] = new wxButton(this, ID_UNDO_BTN, "Undo");

	btn[0]->SetDefault();

	bs[0] = new wxBoxSizer(wxVERTICAL);
	bs[1] = new wxBoxSizer(wxHORIZONTAL);

	bs[0]->Add(msg[0], 0, wxALL, DLG_OFFSET);
	bs[0]->Add(msg[1], 0, wxALL, DLG_OFFSET);
	bs[0]->Add(info, 0, wxALL, DLG_OFFSET);
	bs[1]->Add(btn[0], 0, wxALL, DLG_OFFSET);
	bs[1]->Add(btn[1], 0, wxALL, DLG_OFFSET);
	bs[1]->Add(btn[2], 0, wxALL, DLG_OFFSET);
	bs[0]->Add(bs[1], 0, 0, 0);

	SetAutoLayout(TRUE);
	SetSizer(bs[0]);
	bs[0]->Fit(this);
	bs[0]->SetSizeHints(this);
	Layout();
}

ERROR_RESP
ErrorDialog::GetResp(void)
{
    resp = ER_OK;
    ShowModal();
	return(resp);
}

void
ErrorDialog::SetResp(ERROR_RESP er)
{
    resp = er;
	EndModal(er);
}

// ===========================================================================
ZoneChoice::ZoneChoice(wxWindow *p, bool a) :
	wxChoice(p, -1, wxPoint(-1, -1), wxSize(3 * COL_OFFSET, -1))
{
	allow_none = a;
	if(allow_none)
		Append("None");
	Append("Green");
	Append("Amber");
	Append("Red");
}

ZoneChoice::~ZoneChoice()
{
}

char
ZoneChoice::GetZone(void)
{
int sel;

	sel = GetSelection();
	if(!allow_none)
		sel++;
	
	if(sel < 1) return(' ');
	if(sel < 2) return('G');
	if(sel < 3) return('A');
	return('R');
}

void 
ZoneChoice::SetZone(char z)
{
int sel = 0;	// default to the first
	
	if('R' == z) {
		sel = 2;
		if(allow_none) 
			sel++;
	} else if('A' == z) {
		sel = 1;
		if(allow_none) 
			sel++;
	} else if('G' == z) {
		if(allow_none) 
			sel++;
	}

	SetSelection(sel);
}

// ===============================================================
EditCodeDialog::EditCodeDialog(wxWindow *p, CodeTable *ct) :
    StdDialog(p, "Edit Codes")
{
int i,j;
ListNode *n;
CodeTableEntry *cte;
char *ptr;
wxBoxSizer *bs;
wxGridSizer *gs;

    for(i = 0;i < MAX_CODES;i++)
	    codes[i] = NULL;
	j = 0;

	n = ct->First();
	while(n != NULL) {
		cte = (CodeTableEntry *) n->Data();
		if((ptr = cte->GetCode()) == NULL)
		    break;

		codes[j] = new wxCheckBox(this, -1, ptr, 
				wxPoint(-1, -1),
				wxSize((int) (1.5 * COL_OFFSET), -1));

		j++;
		n = n->Next();
	}

	bs = new wxBoxSizer(wxVERTICAL);
	gs = new wxGridSizer(4, DLG_OFFSET, DLG_OFFSET);
	for(i = 0;i < j;i++) {
		gs->Add(codes[i], i % 4, wxALIGN_LEFT | wxALIGN_RIGHT);
	}
	bs->Add(gs, 0, wxALL, DLG_OFFSET);
	DoLayout(wxALL, DLG_OFFSET, bs);
}

EditCodeDialog::~EditCodeDialog()
{
}

bool
EditCodeDialog::GetCodes(unsigned long *c)
{
unsigned long test=1l;
int i=0;

    while(codes[i] != NULL) {
		if(*c & test)
		    codes[i]->SetValue(TRUE);
		else
		    codes[i]->SetValue(FALSE);
		i++;
		test *= 2;
	}

	if(wxID_CANCEL == ShowModal())
	    return(FALSE);

	i = 0;
	test = 1l;
	*c = 0l;
	while(codes[i] != NULL) {
	  if(codes[i]->GetValue())
		  *c &= test;
	  i++;
      test *= 1;
	}

	return(TRUE);
}

// ===============================================================
CodeList::CodeList(wxWindow *p, bool a) :
	wxListBox(p, -1, wxPoint(-1, -1), wxSize(4 * COL_OFFSET, (8 * ROW_OFFSET)))
{
	allow_none = a;
}

CodeList::~CodeList()
{
}

char *
CodeList::GetCode(void)
{
	return((char *)GetStringSelection().GetData());
}

void
CodeList::SetCode(char *c, SimpleCodeTable *ct)
{
int i=0;
char buffer[120];

	Clear();
	if(allow_none) {
		Append("   None");
		i++;
	}
	buffer[0] = ' ';
	while(buffer[0] != 0) {
		buffer[0] = 0;
		ct->GetFormated(buffer);
		if(buffer[0] != 0) {
			Append(buffer);
			if(strncmp(c, buffer, strlen(c)) == 0)
				SetSelection(i);
		}
		i++;
	}
	if(strlen(c) == 0)
		SetSelection(0);
}

// ===============================================================
StarBox::StarBox(wxWindow *p, char *title) :
	wxStaticBox(p, -1, title)
{
wxFlexGridSizer *fs;

	if(NULL == title)
		SetTitle("Star");
	
	msg[0] = new wxStaticText(p, -1, "Type:");
	msg[1] = new wxStaticText(p, -1, "Class:");
	msg[2] = new wxStaticText(p, -1, "Size:");

	star_type = new wxChoice(p, -1,
			wxPoint(-1, -1),
			wxSize((int) (COL_OFFSET * 2), -1),
			6, type_str);
	star_class = new wxChoice(p, -1,
			wxPoint(-1, -1),
			wxSize((int) (COL_OFFSET * 2), -1),
			10, class_str);
	star_size = new wxChoice(p, -1,
			wxPoint(-1, -1),
			wxSize((int) (COL_OFFSET * 2), -1),
			8, ssize_str);

	sizer = new wxStaticBoxSizer(this, wxHORIZONTAL);
	fs = new wxFlexGridSizer(3, 2, 0, 0);
	fs->Add(msg[0], 0, wxALL | wxALIGN_BOTTOM, DLG_OFFSET/2);
    fs->Add(star_type, 0, wxALL, DLG_OFFSET/2);
    fs->Add(msg[1], 0, wxALL | wxALIGN_BOTTOM, DLG_OFFSET/2);
    fs->Add(star_class, 0, wxALL, DLG_OFFSET/2);
    fs->Add(msg[2], 0, wxALL | wxALIGN_BOTTOM, DLG_OFFSET/2);
    fs->Add(star_size, 0, wxALL, DLG_OFFSET/2);
	sizer->Add(fs, 0, 0, 0);
}

StarBox::~StarBox()
{
}

// ---------------------------------------------------------------------------
bool 
StarBox::Enable(bool val)
{
bool stat;
int i;

	stat = wxStaticBox::Enable(val);
	star_type->Enable(val);
	star_class->Enable(val);
	star_size->Enable(val);
	for(i = 0;i < 3;i++) {
		msg[i]->Enable(val);
	}

	return(stat);
}

// ---------------------------------------------------------------------------
void 
StarBox::SetStar(int *ty, int *cl, int *sz)
{
	star_type->SetSelection(*ty);
	star_class->SetSelection(*cl);
	star_size->SetSelection(*sz);
}

bool 
StarBox::GetStar(int *ty, int *cl, int *sz)
{
	*ty = star_type->GetSelection();
	*cl = star_class->GetSelection();
	*sz = star_size->GetSelection();
	return(TRUE);
}

void
StarBox::SetStar(char *st)
{
int i;
char *ptr;

	star_type->SetSelection(0);
	star_class->SetSelection(0);
	star_size->SetSelection(0);

//fprintf(stderr, "--->st:%s\n", st);
	for(i = 0;i < star_type->GetCount();i++) {
		ptr = (char *)star_type->GetString(i).GetData();
		if(toupper(*st) == ptr[1]) {
			star_type->SetSelection(i);
			break;
		}
	}
	star_class->SetSelection(st[1] - '0');
	for(i = 0;i < star_size->GetCount();i++) {
		ptr = (char *)star_size->GetString(i).GetData();
//fprintf(stderr, "    s:%s p:%s\n", &st[2], &ptr[1]);
		if(strncmp(&st[2], &ptr[1], strlen(&st[2])) == 0) {
			star_size->SetSelection(i);
			break;
		}
	}
}

bool
StarBox::GetStar(char *st)
{
int i;
char *ptr;

	i = star_type->GetSelection();
	ptr = (char *)star_type->GetString(i).GetData();
	st[0] = ptr[1];
	st[1] = star_class->GetSelection() + '0';
	st[2] = 0;
	i = star_size->GetSelection();
	ptr = (char *)star_size->GetString(i).GetData();
	strcat(st, &ptr[1]);

	return(TRUE);
}

