/*
 * File:      sg_build.cpp
 * Purpose:	  generate Traveller systems (1-3)
 * Author:	
 * Created:	
 * Updated:	
 * Copyright: LGPL.
 *            Traveller is a registered trademark of Far Future Enterprises.	
 */

/* rcsid[] = "$RCSfile: sg_build.cpp,v $ $Revision: 1.1 $ $Author: man $ $Date: 2002/04/14 17:03:48 $" */

#include "sg_build.h"

static char *greek_names[] = {
    "Alpha",
	"Beta",
	"Gamma",
	"Delta",
	"Epsilon",
	"Zeta",
	"Eta",
	"Theta",
	"Iota",
	"Kappa",
	"Lambda",
	"Mu",
	"Nu",
	"Xi",
	"Omicron",
	"Pi",
	"Rho",
	"Sigma",
	"Tau",
	"Upsilon",
	"Phi",
	"Chi",
	"Psi",
	"Omega" };

static char *anglic_names[] = {
    "Ay",
	"Bee",
	"See",
	"Dee",
	"Ee",
	"Eff",
	"Gee",
	"Aitch",
	"Eye",
	"Jay",
	"Kay",
	"Ell",
	"Em",
	"En",
	"Oh",
	"Pee",
	"Cue",
	"Are",
	"Ess",
	"Tee",
	"You",
	"Vee",
	"Double-You",
	"Eks",
	"Wye",
	"Zee" };

// ===============================================================
BuildOrbit::BuildOrbit(int o, bool s) :
    ListData()
{
    orbit = o;
	if(o < 1)
	    orb_fraction = dice.Roll(6, 1) - 1;
	else
	    orb_fraction = dice.Roll(10, 1) - 5;

	sat = s;
	flags = 0;
	
	data[0] = data[1] = data[2] = data[3] = data[4] =
	  data[5] = data[6] = data[7] = data[8] = ' ';
	data[9] = 0;

	name = NULL;
	sat_orbits = NULL;
}

BuildOrbit::~BuildOrbit()
{
    if(sat_orbits != NULL)
	    delete sat_orbits;
	if(name != NULL)
	    delete name;
}

void
BuildOrbit::SetName(char *n)
{
    if(name != NULL)
	    delete name;
	name = new char[strlen(n) + 1];
	strcpy(name, n);
}

void
BuildOrbit::SetData(char *uwp)
{
int i;

    for(i = 0;i < 9;i++)
	    data[i] = uwp[i];
}

bool
BuildOrbit::AddSat(char *uwp)
{
BuildOrbit *bo;
int orb,i;

  // NOTE:  UWP_STAR types should not have data added here
  //        (shouldn't be satillites anyhow)
    if((data[1] == UWP_UNDEFINED) || (data[1] == UWP_BELT) ||
	   (data[1] == UWP_RING) || (data[1] == UWP_SAT) ||
	   (data[1] == UWP_STAR) || (data[1] == '0'))
	    return(FALSE);

	if(sat_orbits == NULL)
	    sat_orbits = new OrbitList();

	if(uwp[1] == 'R') {
	    i = dice.Roll(6,1);
		if(i < 4) orb = 1;
		else if(i < 6) orb = 2;
		else orb = 3;
	} else {
		i = dice.Roll(6, 2);
		if(i < 8)
		    orb = dice.Roll(6,2) + 1;
		else if(i < 12)
		    orb = ((dice.Roll(6,2) - 1) * 5) + 10;
		else
		    orb = ((dice.Roll(6,2) - 1) * 25) + 50;
	}

	bo = new BuildOrbit(orb, TRUE);
	bo->SetData(uwp);
	sat_orbits->AppendOrbit(bo);
	return(TRUE);
}

// ---------------------------------------------------------------
char *
BuildOrbit::FormatOrbit(char *buff)
{
int temp;

    temp = orbit;
    if(sat)
	    sprintf(buff, "%d  ", orbit);
	else {
		if(orb_fraction < 0) {
			temp = orbit - 1;
		    sprintf(buff, "%d.%d  ", orbit - 1, 10 + orb_fraction);
		} else
		    sprintf(buff, "%d.%d  ", orbit, orb_fraction);
	}

//fprintf(stderr, "f_orb: %d [%s]", temp, buff);
	if(temp < 10)
	    strrot(buff, 1, TRUE);
	if(temp < 100)
	    strrot(buff, 1, TRUE);
	strstrip(buff, TRUE);
//fprintf(stderr, "[%s]\n", buff);
	return(buff);
}

// ---------------------------------------------------------------
BuildOrbit *
BuildOrbit::GetSat(int ndx)
{
    if(sat_orbits == NULL)
	    return(NULL);
	return(sat_orbits->GetOrbit(ndx));
}

// ---------------------------------------------------------------
void
BuildOrbit::GenFac2(void)
{
	if((data[5] == '6') && (data[4] > '4'))
	    flags |= FAC_COL;
}

void
BuildOrbit::GenFac(bool hab, char *mw_uwp, char mw_bases)
{
int dm=0;

    flags = 0;
	
    if((hab) && (data[2] > '3') && (data[2] <= '9'))
	    flags |= FAC_FARM;
	if((data[4] > '4') && (mw_uwp[4] > '8') &&
	   (((mw_uwp[2] > '1') && (mw_uwp[2] < '5')) ||
		(mw_uwp[2] == '7') || (mw_uwp[2] == '9')))
	    flags |= FAC_MINE;
#if 0
	if((data[5] == '6') && (data[4] > '4'))
	    flags |= FAC_COL;
#endif
	if(mw_uwp[7] > '7') {
		if(mw_uwp[7] > '9')
		    dm = 2;
		if(dice.Roll(6, 2, dm) > 10)
		    flags |= FAC_LAB;
	}
	if(!((data[4] < '1') || ((mw_uwp[2] > '1') && (mw_uwp[2] < '6') &&
						   mw_uwp[3] < '4'))) {
		dm = 0;
		if(mw_uwp[4] > '7')
		    dm++;
		if(mw_uwp[2] == data[2])
		    dm += 2;
		if((mw_bases != ' '))
		    dm++;
		if(dice.Roll(6, 2, dm) > 11)
		    flags |= FAC_MILT;
	}
}

char *
BuildOrbit::GetFac(char *buff)
{
    buff[0] = 0;
	if(flags & FAC_FARM)
	    strcat(buff, "Fa ");
	if(flags & FAC_MINE)
	    strcat(buff, "Mm ");
	if(flags & FAC_COL)
	    strcat(buff, "Co ");
	if(flags & FAC_LAB)
	    strcat(buff, "La ");
	if(flags & FAC_MILT)
	    strcat(buff, "Mi ");

	return(buff);
}

// ===============================================================
OrbitList::OrbitList() :
    LinkedList()
{
}

OrbitList::~OrbitList()
{
BuildOrbit *bo;
ListNode *n;

    n = First();
	while(n != NULL) {
		bo = (BuildOrbit *)n->Data();
		delete bo;
		DeleteNode(n);
//		Remove();
		n = First();
	}
}

void
OrbitList::AppendOrbit(BuildOrbit *bo)
{
int i=0;
BuildOrbit *o;
ListNode *n;

    n = First();
	while(n != NULL) { 
		o = (BuildOrbit *)n->Data();
		if((bo->GetOrbit() > o->GetOrbit()) ||
		   ((bo->GetOrbit() == o->GetOrbit()) &&
			(bo->GetFraction() > o->GetFraction()))) {
			n = n->Next();
			i++;
			continue;
		} else {
			Insert(bo, i);
			return;
		}
	}

	// fell through
	Append(bo);
}

BuildOrbit *
OrbitList::GetOrbit(int ndx)
{
ListNode *n;
int count=0;

    n = First();
	while(n != NULL) {
		if(count == ndx)
		    return((BuildOrbit *)n->Data());
		count++;
		n = n->Next();
	}

	return(NULL);
}

BuildOrbit *
OrbitList::LastOrbit(void)
{
ListNode *n;

    n = Last();
	if(n != NULL)
	    return((BuildOrbit *)n->Data());

	return(NULL);
}

// ===============================================================
BuildStar::BuildStar(char *n, char *d, bool e, UWPGen *gt) :
    DetailStar(n, d)
{
    Init(e, gt);
}

BuildStar::BuildStar(char *n, int t, int c, int s, bool e, UWPGen *gt) :
    DetailStar(n, t, c, s)
{
    Init(e, gt);
}

BuildStar::~BuildStar()
{
    if(name != NULL)
	    delete name;
	delete sub_orbits;
}

void
BuildStar::Init(bool e, UWPGen *gt)
{
int i;

    edit = e;
	gen_uwp = gt;
	
    orbit = ORBIT_UNDEFINED;
	num_orbits = 0;
	for(i = 0;i < 3;i++)
	    mt_orb[i] = capt_orb[i] = -1;
	sub_orbits = new OrbitList();
	for(i = 0;i < MAX_GG;i++)
	    gg_orbs[i] = ORBIT_UNDEFINED;
	for(i = 0;i < MAX_BELTS;i++)
	    belt_orbs[i] = ORBIT_UNDEFINED;
	mw_orb = ORBIT_UNDEFINED;
	name = NULL;
}

void
BuildStar::SetName(char *n)
{
    if(name != NULL)
	    delete name;
	name = new char[strlen(n) + 1];
	strcpy(name, n);
}

void
BuildStar::AddEmpty(int o)
{
    if(mt_orb[0] < 0)
		mt_orb[0] = o;
	else if(mt_orb[1] < 0)
	    mt_orb[1] = o;
	else if(mt_orb[2] < 0)
	    mt_orb[2] = o;
}

void
BuildStar::AddCapt(int o)
{
    if(capt_orb[0] < 0)
		capt_orb[0] = o;
	else if(capt_orb[1] < 0)
	    capt_orb[1] = o;
	else if(capt_orb[2] < 0)
	    capt_orb[2] = o;
}

int
BuildStar::RandomOrbit(void)
{
int i;

    i = num_orbits - GetFirstOrbit() + 1;
	if(i > 0)
	    return(GetFirstOrbit() + dice.Roll(i, 1) - 1);
	else
	    return(-1);
}

int
BuildStar::ValidateOrbit(int o)
{
    if(o < 0)
	    return(o);
	
    if(o < GetFirstOrbit())
	    o = GetFirstOrbit();
	if(o > num_orbits)
	    o = num_orbits;
	
    return(o);
}

void
BuildStar::ValidateAllOrbits(void)
{
int i;
bool ret=FALSE;

    if((orbit == ORBIT_UNDEFINED) || (orbit == ORBIT_CLOSE)) {
		num_orbits = ORBIT_UNDEFINED;
		for(i = 0;i < 3;i++) {
			mt_orb[i] = ORBIT_UNDEFINED;
			capt_orb[i] = ORBIT_UNDEFINED;
		}
	} else if(orbit != ORBIT_PRIMARY) {
		if(num_orbits > (orbit / 2)) {
		    num_orbits = orbit / 2;
			ret = TRUE;
		}
		for(i = 0;i < 3;i++) {
			mt_orb[i] = ValidateOrbit(mt_orb[i]);
			capt_orb[i] = ValidateOrbit(capt_orb[i]);
		}
	}
}

// ---------------------------------------------------------------
BuildOrbit *
BuildStar::GetWorldOrbit(int ndx)
{
    if(sub_orbits == NULL)
	    return(NULL);
	return(sub_orbits->GetOrbit(ndx));
}
// ----------------------------------------------------------------------
int
BuildStar::GetInnerGG(void)
{
int i,ret=9999;

    for(i = 0;i < MAX_GG;i++) {
		if((gg_orbs[i] != ORBIT_UNDEFINED) && (gg_orbs[i] < ret))
		    ret = gg_orbs[i];
	}

	if(ret == 9999)
	    return(ORBIT_UNDEFINED);

	return(ret);
}

bool
BuildStar::SetOrbFlag(int orb, int typ, bool sat_ok)
{
int orb_count=1,i;

    orb_count = GetOrbCount(orb);

	if(orb_count < 1) {
		// special case for sat_ok look for a gas giant
		if((sat_ok) && (typ == UWP_PLANET)) {
			for(i = 0;i < MAX_GG;i++) {
				if(gg_orbs[i] == orb) {
					mw_orb = orb;
					return(TRUE);
				}
			}
		}
		return(FALSE);
	}

	// fell thru, stuff the value
	if(typ == UWP_GG) {
		for(i = 0;i < MAX_GG;i++) {
			if(gg_orbs[i] == ORBIT_UNDEFINED) {
				gg_orbs[i] = orb;
				return(TRUE);
			}
		}
	} else if(typ == UWP_BELT) {
		for(i = 0;i < MAX_BELTS;i++) {
			if(belt_orbs[i] == ORBIT_UNDEFINED) {
				belt_orbs[i] = orb;
				return(TRUE);
			}
		}
	} else if(typ == UWP_PLANET) {
		if(mw_orb == ORBIT_UNDEFINED) {
			mw_orb = orb;
			return(TRUE);
		}
	}

	return(FALSE);
}

bool
BuildStar::GetOrbCount(int orb)
{
int orb_count,i;

    if((orb > num_orbits) || (orb < GetFirstOrbit()))
	    return(0);

	orb_count = 1;
    // determine if orb is avail (walk thru capt and mt)
    for(i = 0;i < 3;i++) {
		if(mt_orb[i] == orb)
		    orb_count--;
		if(capt_orb[i] == orb)
		    orb_count++;
	}
	
    // is orb already used?
	for(i = 0;i < MAX_GG;i++) {
		if(gg_orbs[i] == orb)
		    orb_count--;
	}
	for(i = 0;i < MAX_BELTS;i++) {
		if(belt_orbs[i] == orb)
		    orb_count--;
	}
	if(mw_orb == orb)
	    orb_count--;

	return(orb_count);
}

int
BuildStar::GetTotalOrbits(void)
{
    if(sub_orbits == NULL)
	    return(0);

	return(sub_orbits->Count());
}

void
BuildStar::ClearGG(void)
{
int i;

    for(i = 0;i < MAX_GG;i++)
	    gg_orbs[i] = ORBIT_UNDEFINED;
}

void
BuildStar::ClearBelts(void)
{
int i;

    for(i = 0;i < MAX_BELTS;i++)
	    belt_orbs[i] = ORBIT_UNDEFINED;
}

void
BuildStar::SetGGOrbit(int orb)
{
int i;

    for(i = 0;i < MAX_GG;i++) {
		if(gg_orbs[i] == ORBIT_UNDEFINED) {
			gg_orbs[i] = orb;
			return;
		}
	}
}

void
BuildStar::SetBeltOrbit(int orb)
{
int i;

    for(i = 0;i < MAX_BELTS;i++) {
		if(belt_orbs[i] == ORBIT_UNDEFINED) {
			belt_orbs[i] = orb;
			return;
		}
	}
}

void
BuildStar::DeleteGG(int orb)
{
int i;

    for(i = 0;i < MAX_GG;i++) {
		if(gg_orbs[i] == orb) {
			gg_orbs[i] = ORBIT_UNDEFINED;
			return;
		}
	}
}

void
BuildStar::DeleteBelt(int orb)
{
int i;

    for(i = 0;i < MAX_GG;i++) {
		if(belt_orbs[i] == orb) {
			belt_orbs[i] = ORBIT_UNDEFINED;
			return;
		}
	}
}

// -----------------------------------------------------------------
#define PRIORITY_BAD         -1
#define PRIORITY_HAB         799
#define PRIORITY_POP         ((PRIORITY_HAB + 1) / 2)
#define PRIORITY_UWP         ((PRIORITY_HAB + 1) / 4)
#define PRIORITY_UWP1        (PRIORITY_UWP / 2)
#define PRIORITY_UWP2        (PRIORITY_UWP / 4)
#define PRIORITY_UWP3        (PRIORITY_UWP / 8)

long
BuildStar::GetPriority(bool no_main, char *uwp, char hi_pop, BuildOrbit *bo)
{
char *data;
long ret=0l;

    data = bo->GetData();
#ifdef DEBUG_PRIORITY
	fprintf(stderr, "=======================================\n");
	fprintf(stderr, "GetPriority() data:<%s>\n", data);
#endif
	if(no_main && ((data[1] == 'm') || (data[1] == 'a') || (data[1] == 'R'))) {
#ifdef DEBUG_PRIORITY
	fprintf(stderr, "returning PRIORITY_BAD\n");
#endif
	    return(PRIORITY_BAD);
	}
#if 0
	if(data[4] == ' ')
	    return(PRIORITY_BAD);
#endif
	if(bo->GetOrbit() == GetHabZone()) {
	    ret += PRIORITY_HAB;
#ifdef DEBUG_PRIORITY
		fprintf(stderr, "In HabZone (%ld)\n", ret);
#endif
	}

	if(data[4] >= hi_pop) {
	    ret += PRIORITY_POP;
#ifdef DEBUG_PRIORITY
		fprintf(stderr, "Hi Pop (%ld)\n", ret);
#endif
	}

	if(!no_main) {
		int i;

		for(i = 1;i < 4;i++) {
			int val,uval;

			if(data[i] < '0')
			    val = -1;
			else if(data[i] > '9')
			    val = data[i] - 'A' + 10;
			else
			    val = data[i] - '9';
			if(uwp[i] > '9')
			    uval = uwp[i] - 'A' + 10;
			else
			    uval = uwp[i] - '9';

			if(val == uval)
			    ret += PRIORITY_UWP;
			else if((val == (uval + 1)) || (val == (uval -1)))
			    ret += PRIORITY_UWP1;
#ifdef DEBUG_PRIORITY
			fprintf(stderr, "Checking uwp(%d) (%ld)\n", i, ret);
#endif
		}
	} else {
		if(data[3] > '8')
			ret += PRIORITY_UWP2;
		else if(data[3] > '4')
			ret += PRIORITY_UWP1;
		if((data[2] < 'A') && (data[2] > '3')) {
			if(('8' == data[2]) || ('5' == data[2]) || ('6' == data[2]))
				ret += PRIORITY_UWP1;
			else
				ret += PRIORITY_UWP2;
		}
		if('S' == data[1])			// cannot be a mainworld
			ret -= PRIORITY_UWP;
	}

#ifdef DEBUG_PRIORITY
	fprintf(stderr, "--->returning(%ld)\n", ret);
#endif
	//fprintf(stderr, "%c %s %ld\n", hi_pop, data, ret);
	return(ret);
}

// -----------------------------------------------------------------
void
BuildStar::FillInWorlds(int st1, int st2, bool force)
{
int i,j,count;

    for(i = this->GetFirstOrbit();i < num_orbits;i++) {
		if((st1 > -1) && (i > (st1 / 2)) && (i < (st1 + 2)))
		    continue;
		if((st2 > -1) && (i > (st2 / 2)) && (i < (st2 + 2)))
		    continue;

		count = 1;
		for(j = 0;j < 3;j++) {
			if(mt_orb[j] == i)
			    count--;
			if(capt_orb[j] == i)
			    count++;
		}
		if(force)
			count = 1;

		GenWorlds(i, count);
	}
}

void
BuildStar::GenWorlds(int orb, int num)
{
bool gg_placed=FALSE,type_m_star=FALSE;
int i,z,count;
int temp_gg[MAX_GG],temp_belt[MAX_BELTS],temp_mw;
BuildOrbit *bo;

    temp_mw = mw_orb;
	for(i = 0;i < MAX_GG;i++)
	    temp_gg[i] = gg_orbs[i];
	for(i = 0;i < MAX_BELTS;i++)
	    temp_belt[i] = belt_orbs[i];

	if(GetStarType() == 5)
		type_m_star = TRUE;
	while(num > 0) {
		bool loop;
		char uwp[10],s_uwp[10];

		loop = TRUE;
		bo = new BuildOrbit(orb);

		// place ggs
		for(i = 0;i < MAX_GG;i++) {
			if(orb == temp_gg[i]) {
				temp_gg[i] = ORBIT_UNDEFINED;
				gg_placed = TRUE;
				loop = FALSE;

				if(dice.Roll(6, 1) > 3)
				    bo->SetData("Large GG");
				else
				    bo->SetData("Small GG");
				num--;
				break;
			}
		}

		if(orb < GetHabZone())
			z = -1;
		else if(orb > GetHabZone() + 2)
			z = 2;
		else if(orb > GetHabZone())
			z = 1;

		// place belt
		if(loop) {
			for(i = 0;i < MAX_BELTS;i++) {
				if(orb == temp_belt[i]) {
					temp_belt[i] = ORBIT_UNDEFINED;
					loop = FALSE;
					sprintf(uwp, " 000    ");
					bo->SetData(uwp);
					bo->SetPop(gen_uwp->GenPop(z, uwp));
					num--;
					break;
				}
			}
		}

		// place mw
		if(loop) {
			if(gg_placed) {
			} else {
			}
		}

		// regular world
		if(loop) {
			gen_uwp->GenSystem1(orb, z, NULL, uwp, type_m_star);
			bo->SetData(uwp);
			num--;
		}

		// add it
		sub_orbits->AppendOrbit(bo);

		// gen sats
    	s_uwp[0] = ' ';
		if('m' == uwp[1]) count = dice.Roll(6, 2) - 4;
		else if('a' == uwp[1]) count = dice.Roll(6, 2);
		else count = dice.Roll(6, 1) - 3;

		for(i = 0;i < count;i++) {
			gen_uwp->GenSystem1(orb, z, uwp, s_uwp, type_m_star);
			bo->AddSat(s_uwp);
		}
	}
}

// -----------------------------------------------------------------
void
BuildStar::FinishWorlds(char *uwp, char base)
{
int i=0;
BuildOrbit *bo;

    while((bo = GetWorldOrbit(i)) != NULL) {
		int j=0;
		BuildOrbit *sat;
		char *data;

		data = bo->GetData();
		if((data[1] != 'm') && (data[1] != 'a'))
			FinishWorlds(bo, uwp, base);

		while((sat = bo->GetSat(j)) != NULL) {
			FinishWorlds(sat, uwp, base);
			j++;
		}
		
		i++;
	}
}

void
BuildStar::FinishWorlds(BuildOrbit *bo, char *uwp, char base)
{
char *data,buffer[12];

    data = bo->GetData();
	if(' ' == data[UWPI_PORT]) {
		strcpy(buffer, data);
		bo->GenFac(bo->GetOrbit() == GetHabZone(), uwp, base);
		gen_uwp->GenSystem3(uwp, buffer, data, 
				bo->LabPresent() || bo->MiltPresent());
		bo->GenFac2();
	}
}

// -----------------------------------------------------------------
bool
BuildStar::OrbitAvail(int o, bool sat_ok)
{
int j;

    if(o > num_orbits)
	    return(FALSE);
	
	if(GetOrbCount(o) < 1)
	    {
		if(sat_ok)
		    {
			for(j = 0;j < MAX_GG;j++)
			    if(o == gg_orbs[MAX_GG])
				    return(TRUE);
			}
		return(FALSE);
		}

    return(TRUE);
}

// -----------------------------------------------------------------
void
BuildStar::ClearOrbits(void)
{
    if(sub_orbits != NULL)
	    delete sub_orbits;

	sub_orbits = new OrbitList();
}

void
BuildStar::ClearKnown(void)
{
int i;

//    for(i = 0;i < 3;i++)
//	capt_orb[i] = mt_orb[i] = ORB_UNDEFINED;
    for(i = 0;i < MAX_GG;i++)
	    gg_orbs[i] = ORBIT_UNDEFINED;
	for(i = 0;i < MAX_BELTS;i++)
	    belt_orbs[i] = ORBIT_UNDEFINED;
	mw_orb = ORBIT_UNDEFINED;
}

// ===============================================================
SystemGenerate::SystemGenerate(char *dir_name, TECH_GEN tg, int n, int x)
{
int i;
char file_name[256];

    state = BS_INIT;
	for(i = 0;i < MAX_STARS;i++)
	    stars[i] = NULL;

	mw_st = mw_orb = mw_sat = -1;
	
	sprintf(file_name, "%s/%s", dir_name, STARGEN_DATA_FILE);
	gen_uwp = new UWPGen(tg, n, x);
	gen_star = new GenStar(file_name);
}

SystemGenerate::~SystemGenerate()
{
int i;

    for(i = 0;i < MAX_STARS;i++)
	    if(stars[i] != NULL)
		    delete stars[i];
	delete gen_uwp;
	delete gen_star;
}

bool
SystemGenerate::NextState(struct sys_conf *cfg)
{
    switch(state) {
		case BS_INIT:
		  // determine stars and features
		    BuildStars(cfg);
		    state = BS_STAR;
		    break;
		case BS_STAR:
		  // edit known stars
		    PlaceKnown(cfg);
		    state = BS_KNOWN;
		    break;
		case BS_KNOWN:
		  // place known components (belts, gg, mw?)
		    FillInWorlds(cfg);
		    state = BS_MW_SELECT;
		    break;
		case BS_MW_SELECT:
		  // select a main world
		    FinishWorlds(cfg);
		    state = BS_FINISH;
		    break;
		case BS_FINISH:
		    return(FALSE);
		    break;
	}
	
	return(TRUE);
}

bool
SystemGenerate::PrevState(struct sys_conf *cfg)
{
    switch(state) {
		case BS_INIT:
		    return(FALSE);
		    break;
		case BS_STAR:
		    RemoveStars(cfg);
		    state = BS_INIT;
		    break;
		case BS_KNOWN:
		    RemoveKnown(cfg);
		    state = BS_STAR;
		    break;
		case BS_MW_SELECT:
		    UnFillIn(cfg);
		    state = BS_KNOWN;
		    break;
		case BS_FINISH:
		    UnFinish(cfg);
		    state = BS_MW_SELECT;
		    break;
	}
	
	return(TRUE);
}

bool
SystemGenerate::Cancel(sys_conf *cfg)
{
    switch(state) {
		case BS_FINISH:
		    UnFinish(cfg);
		case BS_MW_SELECT:
		    UnFillIn(cfg);
		case BS_KNOWN:
		    RemoveKnown(cfg);
		case BS_STAR:
		    RemoveStars(cfg);
		case BS_INIT:
		    break;
	}
  
    state = BS_INIT;
	return(TRUE);
}

bool
SystemGenerate::End(sys_conf *cfg)
{
// wrap it up, everything falls through

    switch(state) {
		case BS_INIT:
		    BuildStars(cfg);
		case BS_STAR:
		    PlaceKnown(cfg);
		case BS_KNOWN:
		    FillInWorlds(cfg);
		case BS_MW_SELECT:
		    FinishWorlds(cfg);
		case BS_FINISH:
		    break;
	}

	state = BS_FINISH;
	return(TRUE);
}

// ---------------------------------------------------------------
void
SystemGenerate::BuildStars(struct sys_conf *cfg)
{
int i,j,k,type_roll,size_roll;
char lstars[3][8];
bool edit=TRUE;

// update status bar here
    // clear these out since we may have come 'back' to here
    for(i = 0;i < MAX_STARS;i++) {
		if(stars[i] != NULL) {
			delete stars[i];
			stars[i] = NULL;
		}
	}
	
    // get local copies
    for(i = 0;i < 3;i++) {
		strcpy(lstars[i], cfg->stars[i]);
		strstrip(lstars[i], FALSE);
	}
	if((lstars[0][0] != 'R') && (lstars[1][0] != 'R') && (lstars[2][0] != 'R'))
	    edit = FALSE;
	// determine solo, binary, trinary nature of system
	if((lstars[0][0] == 'R') && (lstars[1][0] == 'R') && 
			(lstars[2][0] == 'R')) {
		j = gen_star->GenSystemNature();
		if(j < 2) {
			lstars[1][0] = 'N';
			lstars[2][0] = 'N';
		} else if(j < 3) {
		    lstars[1][0] = 'R';
		    lstars[2][0] = 'N';
		} else {
		    lstars[1][0] = 'R';
		    lstars[2][0] = 'R';
		}
	}

	type_roll = size_roll = 0;
	// walk thru the 'known' stars
    for(i = 0;i < 3;i++) {
// update status bar here
		// random?
		if(lstars[i][0] == 'R') {
			gen_star->GenerateStar(i, &type_roll, &size_roll, 
							cfg->uwp, lstars[i]);
			stars[i] = new BuildStar(NULL, lstars[i], edit, gen_uwp);
		// present (and not random)?
		} else if(lstars[i][0] != 'N')
			stars[i] = new BuildStar(NULL, lstars[i], edit, gen_uwp);
	}

	// deterimine each star's orbit
	for(i = 0;i < 3;i++)
// update status bar here
		SetStarOrbit(i, 0);
	// adjust position of each star
// warning here
	VerifyStarOrbits();
// ---------------------------------------
	// determine additional stars (and their orbit)
// update status bar here
	for(i = 1;i < 3;i++)
	    if((stars[i] != NULL) && (stars[i]->GetOrbit() > ORBIT_FAR) &&
		   (stars[i+2] != NULL) && (dice.Roll(6,2) > 8)) {
			gen_star->GenerateStar(i+2, &type_roll, &size_roll, 
							cfg->uwp, lstars[i+2]);
			stars[i] = new BuildStar(NULL, lstars[i+2], edit, gen_uwp);
			SetStarOrbit(i+2, i);
		}
	// determine number of orbits (and mt and capt orbits)
	for(i = 0;i < MAX_STARS;i++) {
		if(stars[i] == NULL)
		    continue;
		j = dice.Roll(6, 2);
		if(stars[i]->GetStarSize() < 3)
		    j += 8;
		if(stars[i]->GetStarSize() == 3)
		    j += 4;
		if(stars[i]->GetStarType() > 4)
		    j -= 4;
		if(stars[i]->GetStarType() == 4)
		    j -= 2;
// ---------------------------------------
		if(i != 0) {
			if(j >(stars[i]->GetOrbit() / 2))
			    j = stars[i]->GetOrbit() / 2;
		}
		stars[i]->SetNumOrbits(j);
		if(j < 0)
		    continue;
		
		// capt orbits
		j = dice.Roll(6, 1);
		if(stars[i]->GetStarType() < 3)
		    j++;
		if(j > 4) {
			j = dice.Roll(6, 1);
			if(stars[i]->GetStarType() < 3)
			    j++;
			for(k = 0;k < (j/2);k++)
			    stars[i]->AddCapt(stars[i]->RandomOrbit());
		}
		// mt orbits
		j = dice.Roll(6, 1);
		if(stars[i]->GetStarType() < 3)
		    j++;
		if(j > 4) {
			j = dice.Roll(6, 1);
			if(stars[i]->GetStarType() < 3)
			    j++;
			if(j == 4)
			    j = 3;
			else
			    j /= 2;
			for(k = 0;k < j;k++)
			    stars[i]->AddEmpty(stars[i]->RandomOrbit());
		}
	}
	// done
}

void
SystemGenerate::SetStarOrbit(int ndx, int p_ndx)
{
int j,k;

    if(stars[ndx] != NULL) {
		if(ndx == 0)
		    stars[ndx]->SetOrbit(ORBIT_PRIMARY);
		else {
			j = dice.Roll(6, 2);
			if(ndx == 2)    // second comp?
			    j += 4;
			else if(ndx > 2)
			    j -= 4;

			if(j < 3) {
				if((stars[p_ndx]->GetStarType() < 5) ||
				   ((stars[p_ndx]->GetStarType() == 5) &&
					(stars[p_ndx]->GetStarSize() < 5)))
				    k = ORBIT_CLOSE;
				else
				    k = 0;
			} else if(j < 7)
			    k = j - 3;
			else if(j < 12)
			    k = (j - 4) + dice.Roll(6, 1);
			else {
				j = dice.Roll(6, 2);
				if((j < 3) || (j > 11))
				    k = 25 + dice.Roll(6, 1);
				else if((j < 6) || (j > 8))
				    k = 19 + dice.Roll(6, 1);
				else
				    k = 13 + dice.Roll(6, 1);
			}
			if(ndx > 2) {
				if(k > (stars[p_ndx]->GetOrbit() / 2))
				    k = stars[p_ndx]->GetOrbit() / 2;
			}
			stars[ndx]->SetOrbit(k);			
		}
	}
}

bool
SystemGenerate::VerifyStarOrbits(void)
{
bool ret=FALSE;
int i;

	if((stars[1] == NULL) && (stars[2] != NULL))
	    {
		stars[1] = stars[2];
		stars[2] = NULL;
		if(stars[4] != NULL)
		    {
			stars[3] = stars[4];
			stars[4] = NULL;
			}
		ret = TRUE;
		}
	else if((stars[1] != NULL) && (stars[2] != NULL))
	    {
		// make star 2 orbit greater than star 1
		if(stars[1]->GetOrbit() > stars[2]->GetOrbit())
		    {
			BuildStar *bs;

			bs = stars[2];
			stars[2] = stars[1];
			stars[1] = bs;
			ret = TRUE;
			}
		// make sure star 1 is in an allowed orbit
		if((stars[1]->GetOrbit() > ORBIT_CLOSE) &&
		   (stars[2]->GetOrbit() > ORBIT_CLOSE))
		    {
			while(stars[1]->GetOrbit() > (stars[2]->GetOrbit() / 2))
			    {
				if(stars[1]->GetOrbit() > 0)
				    stars[1]->SetOrbit(stars[1]->GetOrbit() - 1);
				stars[2]->SetOrbit(stars[2]->GetOrbit() + 1);
				ret = TRUE;
				}
			}
		}

	for(i = 1;i < 2;i++)
	    {
		if(stars[i] != NULL)
		    {
			if((stars[i]->GetOrbit() < ORBIT_FAR) && (stars[i+2] != NULL))
			    {
				DeleteStar(i+2);
				ret = TRUE;
				}
			if(stars[i+2] != NULL)
			    {
				if(stars[i+2]->GetOrbit() > (stars[i]->GetOrbit() / 2))
				    {
					stars[i+2]->SetOrbit(stars[i]->GetOrbit() / 2);
					ret = TRUE;
					}
				}
			}
		else if(stars[i+2] != NULL)
		    {
			DeleteStar(i+2);
			ret = TRUE;
			}
		}

	return(ret);
}

bool
SystemGenerate::OrbitAvail(int ndx, int o, bool sat_ok)
{
int i,j;

    if(stars[ndx] == NULL)
	    return(FALSE);

    if(ndx == 0)
	    {
		for(i = 1;i < 3;i++)
		    {
			if(stars[i] == NULL)
			    continue;
			j = stars[i]->GetOrbit();
			if((j >= 0) && (o > (j / 2)) && (o < (j + 2)))
			    return(FALSE);
			}
		}
	else if(ndx < 3)
	    {
		if(stars[ndx + 2] != NULL)
		    {
			j = stars[ndx + 2]->GetOrbit();
			if((j >= 0) && (o > (j / 2)) && (o < (j + 2)))
			    return(FALSE);
			}
		}

	if(!stars[ndx]->OrbitAvail(o, sat_ok))
	    return(FALSE);

	return(TRUE);
}

// ----------------------------------------------------------------------
void
SystemGenerate::DeleteStar(int ndx)
{
    if(stars[ndx] != NULL) {
		delete stars[ndx];
		stars[ndx] = NULL;
	}
}

// ----------------------------------------------------------------------
void
SystemGenerate::PlaceKnown(sys_conf *cfg)
{
int gg,belts,mw=0;
int i,j,meth;

	// check for gg's
    if(cfg->gg < 0) {
		if(dice.Roll(6, 2) < 4)
		    gg = 0;
		else {
			i = dice.Roll(6, 2);
			if(i < 4)
			    gg = 1;
			else if(i < 6)
			    gg = 2;
			else if(i < 8)
			    gg = 3;
			else if(i < 11)
			    gg = 4;
			else
			    gg = 5;
		}
	} else {
		gg = cfg->gg;
		if(gg > MAX_GG)
		    gg = MAX_GG;
	}

	// check for belts
	if(cfg->belts < 0) {
		if(dice.Roll(6, 2) < 7)
		  belts = 0;
		else {
			i = dice.Roll(6,2);
			if(i < 8)
			    belts = 1;
			else if(i < 12)
			    belts = 2;
			else
			    belts = 3;
		}
	} else {
		belts = cfg->belts;
		if(belts > MAX_BELTS)
		    belts = MAX_BELTS;
	}
	// check for mw
	if(!cfg->no_main)
		mw = 1;

	// assign gg's
	meth = GP_HAB_PLUS_2;
	while(gg > 0) {
		int temp_gg;

		temp_gg = gg;
		for(i = 0;i < temp_gg;i++) {
			for(j = 0;j < MAX_STARS;j++) {
				if(stars[j] == NULL)
				    continue;
				if((j != 0) && (stars[j]->GetOrbit() < 0))
				    continue;
				gg -= PlaceGG(j, (GG_PLACE) meth);
				if(gg == 0)
				    break;
			}
		}
		meth++;
#ifdef SG_DEBUG
		if((meth > GP_EXTEND) && (gg > 0)) {
			fprintf(stderr, "AAAARGH!!!!!  PLACE GG FAILED!!!!!!\n");
			fprintf(stderr, "gg: %d  belts: %d  mw: %d\n", gg, belts, mw);
			DumpSG(cfg);
		}
#endif
	}

	// assign belts
	meth = BP_GG;
	while(belts > 0) {
		int temp_belts;

		temp_belts = belts;
		for(i = 0;i < temp_belts;i++) {
			for(j = 0;j < MAX_STARS;j++) {
				if(stars[j] == NULL)
				    continue;
				if((j != 0) && (stars[j]->GetOrbit() < 0))
				    continue;
				belts -= PlaceBelt(j, (BELT_PLACE) meth);
				if(belts == 0)
				    break;
			}
		}
		meth++;
#ifdef SG_DEBUG
		if((meth > BP_EXTEND) && (belts > 0)) {
			fprintf(stderr, "AAAARGH!!!!!  PLACE BELTS FAILED!!!!!!\n");
			fprintf(stderr, "gg: %d  belts: %d  mw: %d\n", gg, belts, mw);
			DumpSG(cfg);
		}
#endif
	}

	// assign mw
	meth = MP_HAB;
	while(mw > 0) {
		for(j = 0;j < MAX_STARS;j++) {
			if(stars[j] == NULL)
			    continue;
			if((j != 0) && (stars[j]->GetOrbit() < 0))
			    continue;
			if(PlaceMW(j, (MW_PLACE) meth) != 0) {
				mw = 0;
			    break;
			}
		}
		meth++;
#ifdef SG_DEBUG
		if((meth > MP_EXTEND) && (mw > 0)) {
			fprintf(stderr, "AAAARGH!!!!!  PLACE MW FAILED!!!!!!\n");
			fprintf(stderr, "gg: %d  belts: %d  mw: %d\n", gg, belts, mw);
			DumpSG(cfg);
		}
#endif
	}
}

// ----------------------------------------------------------------------
int
SystemGenerate::PlaceGG(int ndx, GG_PLACE meth)
{
    switch(meth) {
		case GP_HAB_PLUS_2:
		    if(GGHab2(ndx))
			    return(1);
			break;
		case GP_INNER:
		    if(AddInner(ndx, UWP_GG))
			    return(1);
			break;
		case GP_ANY:
		    if(AddAnywhere(ndx, UWP_GG))
			    return(1);
			break;
		case GP_EXTEND:
		    if(AddExtend(ndx, UWP_GG))
			    return(1);
			break;
	}
	return(0);
}

int
SystemGenerate::PlaceBelt(int ndx, BELT_PLACE meth)
{
    switch(meth)
	    {
		case BP_GG:
		    if(BeltGG(ndx))
			    return(1);
			break;
		case BP_INNER:
		    if(AddInner(ndx, UWP_BELT))
			    return(1);
			break;
		case BP_ANY:
		    if(AddAnywhere(ndx, UWP_BELT))
			    return(1);
			break;
		case BP_EXTEND:
		    if(AddExtend(ndx, UWP_BELT))
			    return(1);
			break;
		}

	return(0);
}

int
SystemGenerate::PlaceMW(int ndx, MW_PLACE meth)
{
    switch(meth)
	    {
		case MP_HAB:
		    if(AddToHab(ndx, FALSE))
				    return(1);
			break;
		case MP_HAB_GG:
		    if(AddToHab(ndx, TRUE))
				    return(1);
		    break;
		case MP_ANY:
		    if(AddAnywhere(ndx, UWP_PLANET))
				    return(1);
		    break;
		case MP_EXTEND:
		    if(AddExtend(ndx, UWP_PLANET))
				    return(1);
		    break;
		}

	return(0);
}

bool 
SystemGenerate::GGHab2(int ndx)
{
int orb,i;

    if((orb = stars[ndx]->GetHabZone()) < 0)
	    return(FALSE);
	orb += 2;

	for(i = orb;i <= stars[ndx]->GetNumOrbits();i++) {
		if(OrbitAvail(ndx, i, FALSE) && (stars[ndx]->SetOrbFlag(i, UWP_GG)))
		    return(TRUE);
	}

	return(FALSE);
}

bool 
SystemGenerate::BeltGG(int ndx)
{
int orb,i;

   if((orb = stars[ndx]->GetInnerGG()) < 0)
	   orb = stars[ndx]->GetNumOrbits();
   else
	   orb--;

   if(orb < 0)
	   return(FALSE);
   
   for(i = orb;i >= stars[ndx]->GetFirstOrbit();i--) {
	   if(stars[ndx]->GetHabZone() == i)
		   continue;
	   if(OrbitAvail(ndx, i, FALSE) && (stars[ndx]->SetOrbFlag(i, UWP_BELT)))
		   return(TRUE);
	}

   return(FALSE);
}

bool 
SystemGenerate::AddToHab(int ndx, bool sat_ok)
{
int orb;

    if((orb = stars[ndx]->GetHabZone()) < 0)
	    return(FALSE);

	if(OrbitAvail(ndx, orb, sat_ok) &&
	   (stars[ndx]->SetOrbFlag(orb, UWP_PLANET, sat_ok)))
	    return(TRUE);

	return(FALSE);
}

bool 
SystemGenerate::AddInner(int ndx, int typ)
{
int orb,i,j;
bool flag;

    if((orb = stars[ndx]->GetFirstOrbit()) < 0)
	    return(FALSE);

	if((j = stars[ndx]->GetHabZone()) < 0)
	    return(FALSE);

	if(UWP_PLANET == typ)
	    flag = TRUE;
	else
	    flag = FALSE;
	for(i = orb;i < j;i++) {
		if(OrbitAvail(ndx, i, flag) && (stars[ndx]->SetOrbFlag(i, typ)))
		   return(TRUE);
	}

	return(FALSE);
}

bool 
SystemGenerate::AddAnywhere(int ndx, int typ)
{
int orb,i;
bool flag;

    if((orb = stars[ndx]->GetFirstOrbit()) < 0)
	    return(FALSE);

	if(UWP_PLANET == typ)
	    flag = TRUE;
	else
	    flag = FALSE;
	for(i = orb;i <= stars[ndx]->GetNumOrbits();i++) {
		if(OrbitAvail(ndx, i, flag) && (stars[ndx]->SetOrbFlag(i, typ)))
		   return(TRUE);
	}

	return(FALSE);
}

bool 
SystemGenerate::AddExtend(int ndx, int typ)
{
int orb;

    if((orb = stars[ndx]->GetNumOrbits()) < 0) {
		if(stars[ndx]->GetOrbit() == ORBIT_CLOSE)
		    return(FALSE);
	}

	while(TRUE) {
		orb++;
#ifdef SG_DEBUG
		if(orb > 100) {
			fprintf(stderr, "AAAARGH!!!!!  EXTEND ORBITS FAILED!!!!!!\n");
			fprintf(stderr, "ndx: %d  typ: %d\n", ndx, typ);
			DumpSG(NULL);
		}
#endif

		if(ndx == 0) {
			int i;
			bool flag;

			flag=FALSE;
			for(i = 1;i < 3;i++) {
				if(stars[i] != NULL) {
					int j;

					j = stars[i]->GetOrbit();
					if((j >= 0) && (orb > (j / 2)) && (orb < (j + 2)))
					    flag = TRUE;
				}
			}
			if(flag)
			    continue;
		} else if(ndx < 2) {
			if(orb > (stars[ndx]->GetOrbit() / 2))
			    return(0);
			if((ndx < 3) && (stars[ndx+2] != NULL)) {
				int j;

				j = stars[ndx+2]->GetOrbit();
				if((j >= 0) && ((orb > (j / 2)) && (orb < (j + 2))))
				    continue;
			}
		}

		stars[ndx]->SetNumOrbits(orb);
		if(stars[ndx]->SetOrbFlag(orb, typ))
		    return(TRUE);
	}

	return(FALSE);
}

// ----------------------------------------------------------------------
void
SystemGenerate::FillInWorlds(sys_conf *cfg)
{
int i,st1,st2;

    for(i = 0;i < MAX_STARS;i++) {
		st1 = st2 = ORBIT_UNDEFINED;
		if(i == 0) {
			if(stars[1] != NULL) {
			    st1 = stars[1]->GetOrbit();
			}
			if(stars[2] != NULL) {
			    st2 = stars[2]->GetOrbit();
			}
		} else if(i < 3) {
			if(stars[i + 2] != NULL) {
			    st1 = stars[i + 2]->GetOrbit();
			}
		}

		if(stars[i] != NULL) {
		    stars[i]->FillInWorlds(st1, st2);
//			FillInNames(cfg->name, cfg->lang, i);
			FillInNames(NULL, cfg->lang, i);
		}
	}

	SetMW(cfg);
}

void
SystemGenerate::SetMW(struct sys_conf *cfg)
{
char old_pop = ' ',*data;
int i,j,k;
long priority=-1,temp;
BuildOrbit *bo,*sat,*wo=NULL;

	mw_st = mw_orb = mw_sat = -1;
    for(i = 0;i < MAX_STARS;i++) {
		if(stars[i] != NULL) {
			j = 0;
			while((bo = stars[i]->GetWorldOrbit(j)) != NULL) {
				if((temp = stars[i]->GetPriority(cfg->no_main, cfg->uwp,
												 old_pop, bo)) > priority) {
					  mw_st = i;
					  mw_orb = j;
					  mw_sat = -1;
					  wo = bo;
					  priority = temp;
				}
				data = bo->GetData();
				if((data[1] != 'm') && (data[1] != 'a') && (data[4] > old_pop))
				  old_pop = data[4];
				k = 0;
				while((sat = bo->GetSat(k)) != NULL) {
					if((temp =
						stars[i]->GetPriority(cfg->no_main, cfg->uwp,
											  old_pop, sat)) > priority) {
						  mw_st = i;
						  mw_orb = j;
						  mw_sat = k;
						  wo = sat;
						  priority = temp;
					}
					data = sat->GetData();
					if(data[4] > old_pop)
					    old_pop = data[4];					
					k++;
				}
				j++;
			}
		}
	}

	// XXX
	if((mw_st < 0) || (mw_orb < 0)) {
		// Oh-oh, looks like there isn't any orbits
		//  brute force it at the end
		int st1=-1,st2=-1;

		mw_st = 0;
		mw_orb = 0;
		temp = stars[0]->GetNumOrbits();
		temp++;
		if(stars[1] != NULL) {
			st1 = stars[1]->GetOrbit();
			temp = stars[1]->GetOrbit() + 2;
		}
		if(stars[2] != NULL) {
			st2 = stars[2]->GetOrbit();
			temp = stars[2]->GetOrbit() + 2;
		}
		stars[0]->SetNumOrbits(temp);
		stars[0]->FillInWorlds(st1, st2, TRUE);
		FillInNames(NULL, cfg->lang, 0);
		wo = stars[0]->GetWorldOrbit(0);
	}

	if(cfg->no_main) {
#ifdef DEBUG_PRIORITY
		fprintf(stderr, "Setting mw: %d:%d:%d %c\n", mw_st, mw_orb, mw_sat, cfg->port);
#endif
	    GenMW(mw_st, mw_orb, mw_sat, cfg->port);
	} else {
	    wo->SetData(cfg->uwp);
		wo->SetName(cfg->name);
#if 0
		wo->SetBase(cfg->bases);
		wo->SetAlleg(cfg->alleg);
		wo->SetZone(cfg->alleg);
#endif
	}
}

void
SystemGenerate::FillInNames(char *n, WordGen *lang, int ndx)
{
int i,j,k,count=0,sat_count;
int st1,st2;
int ndx1,ndx2;
BuildOrbit *bo,*sat;
char nbuff[120],bbuff[120];

    if(ndx != 0)
	    sprintf(bbuff, stars[ndx]->GetName());
	else if(n != NULL)
	    sprintf(bbuff, n);
	else
	    bbuff[0] = 0;

	st1 = st2 = ORBIT_EXTREME;
	if(0 == ndx)
	    {
	    stars[ndx]->SetName("Primary");
		if(stars[1] != NULL)
		    {
		    st1 = stars[1]->GetOrbit();
			ndx1 = 1;
			}
		if(stars[2] != NULL)
		    {
			st2 = stars[2]->GetOrbit();
			ndx2 = 2;
			}
		}
	else
	    {
		if(stars[ndx]->GetOrbit() == ORBIT_CLOSE)
		    stars[ndx]->SetName("Companion");
		if(i < 3)
		    {
			if(stars[ndx + 2] != NULL)
			    {
			    st1 = stars[ndx + 2]->GetOrbit();
				ndx1 = 1;
				}
			}
		}
	
	j = 0;
	while((bo = stars[ndx]->GetWorldOrbit(j)) != NULL)
	    {
		if(bo->GetOrbit() > st1)
		    {
			if(lang != NULL)
			    lang->GenerateWord(nbuff);
			else
			    sprintf(nbuff, "%s%s", bbuff, greek_names[count++]);
			stars[ndx1]->SetName(nbuff);
			st1 = ORBIT_EXTREME;
			}
		if(bo->GetOrbit() > st2)
		    {
			if(lang != NULL)
			    lang->GenerateWord(nbuff);
			else
			    sprintf(nbuff, "%s%s", bbuff, greek_names[count++]);
			stars[ndx2]->SetName(nbuff);
			st2 = ORBIT_EXTREME;
			}
		if(lang != NULL)
		    lang->GenerateWord(nbuff);
		else
		    sprintf(nbuff, "%s%s", bbuff, greek_names[count]);
		bo->SetName(nbuff);

		k = 0;
		sat_count = 0;
		while((sat = bo->GetSat(k)) != NULL)
		    {
			if(lang != NULL)
			    lang->GenerateWord(nbuff);
			else
			    sprintf(nbuff, "%s%s%s", bbuff, greek_names[count],
					  anglic_names[sat_count++]);
			sat->SetName(nbuff);
			k++;
			}
		count++;
		j++;
		}
	
	if(st1 < ORBIT_EXTREME)
	    {
		if(lang != NULL)
			lang->GenerateWord(nbuff);
		else
		    sprintf(nbuff, "%s%s", bbuff, greek_names[count++]);
		stars[ndx1]->SetName(nbuff);
		}
	if(st2 < ORBIT_EXTREME)
	    {
	    if(lang != NULL)
		    lang->GenerateWord(nbuff);
		else
		    sprintf(nbuff, "%s%s", bbuff, greek_names[count++]);
		stars[ndx2]->SetName(nbuff);
		}
}

// ----------------------------------------------------------------------
void
SystemGenerate::ClearMW(void)
{
    if(mw_st >= 0)
	    {
		BuildOrbit *bo;
		char *ptr;

		bo = stars[mw_st]->GetWorldOrbit(mw_orb);
		if(mw_sat >= 0)
		    bo = bo->GetSat(mw_sat);

		ptr = bo->GetData();
		ptr[0] = ptr[5] = ptr[6] = ptr[7] = ' ';

		mw_st = mw_orb = mw_sat = -1;
		}
}

void
SystemGenerate::GenMW(int st, int o, int s, char p)
{
BuildOrbit *bo;
char *ptr,buffer[12];

	mw_st = st;
	mw_orb = o;
	mw_sat = s;

	bo = stars[mw_st]->GetWorldOrbit(mw_orb);
#ifdef SG_DEBUG
	if(NULL == bo) {
		fprintf(stderr, "---->GenMW(%d, %d, %d, %c) %d %d %d\n", 
			st, o, s, p, mw_st, mw_orb, mw_sat);
		DumpSG(NULL);
	}
#endif
	if(mw_sat >= 0)
	    bo = bo->GetSat(mw_sat);
	ptr = bo->GetData();

	if(' ' == ptr[UWPI_POP])
	    ptr[UWPI_POP] = '0';

	strcpy(buffer, ptr);
	gen_uwp->GenSystem2(p, buffer, ptr);
}

// ----------------------------------------------------------------------
void
SystemGenerate::FinishWorlds(sys_conf *cfg)
{
int i;
char *ptr;

    if(cfg->no_main) {
		BuildOrbit *bo;

		bo = stars[mw_st]->GetWorldOrbit(mw_orb);
		if(mw_sat > -1)
		    bo = bo->GetSat(mw_sat);
		ptr = bo->GetData();
	} else
	    ptr = cfg->uwp;
	
    for(i = 0;i < MAX_STARS;i++)
		if(stars[i] != NULL)
		    stars[i]->FinishWorlds(ptr, cfg->bases[0]);
}

// ----------------------------------------------------------------------
int
SystemGenerate::Save(char *filename, sys_conf *cfg)
{
int count=0,name_len;
FILE *fp;

    if((fp = fopen(filename, "w+")) == NULL)
	    return(-1);

	name_len = LongestName();
	if(name_len < 16) name_len = 16;
	
	if(cfg->name != NULL) {
		fprintf(fp, "    %s", cfg->name);
		if(cfg->loc != NULL)
		    fprintf(fp, " %s", cfg->loc);
		fprintf(fp, "\n");
		fprintf(fp, "System: %s\n", cfg->name);
	} else
	    fprintf(fp, "System: %s\n", "Unknown");

	if(cfg->sect != NULL)
	    fprintf(fp, "Sector: %s\n", cfg->sect);
	else
	    fprintf(fp, "Sector: %s\n", "Unknown");

	if(cfg->ssect != NULL)
	    fprintf(fp, "SubSector: %s\n", cfg->ssect);
	else
	    fprintf(fp, "SubSector: %s\n", "Unknown");

	if(cfg->loc != NULL)
	    fprintf(fp, "Location: %s\n\n", cfg->loc);
	else
	    fprintf(fp, "Location: %s\n\n", "Unknown");

	fprintf(fp,
		"The data in the system text files is laid out in column format:\n");
	fprintf(fp, " 1-14: Orbit(s)\n");
	fprintf(fp, "15-%d: Name\n", name_len + 14);
	fprintf(fp, "%d-%d: UWP\n", name_len + 16, name_len + 23);
	fprintf(fp, "   %d: Bases\n", name_len + 25);
	fprintf(fp, "%d-%d: Codes & Comments\n", name_len + 27, name_len + 49);
	fprintf(fp, "   %d: Travel Zone (optional)\n", name_len + 51);
	fprintf(fp, "%d-%d: Allegiance (optional)\n\n", name_len + 53,
			name_len + 54);

	fprintf(fp,
	"The orbits are formated by the last position of it's number as follows:\n"
			" 4: Primary Orbit\n"
			" 5: Primary Satillite Orbit\n"
			" 7: Primary Companion Orbit\n"
			" 8: Primary Companion Satillite Orbit\n"
			"10: Secondary Companion Orbit\n"
			"11: Secondary Companion Satillite Orbit\n\n");

	fprintf(fp, "....+....1....+....2....+....3....+....4"
			    "....+....5....+....6....+....7....+....8\n"
			    "1234567890123456789012345678901234567890"
			    "1234567890123456789012345678901234567890\n");

	SaveStar(fp, 0, name_len);

    fclose(fp);
	return(count);
}

void
SystemGenerate::SaveStar(FILE *fp, int ndx, int name_len)
{
char buff[120];
int i,st1,st2,ndx1=-1,ndx2=-1,indent;
BuildOrbit *bo;

	st1 = st2 = ORBIT_EXTREME;
	if(0 == ndx) {
		indent = -3;
	    sprintf(buff, "Primary");
		if(stars[1] != NULL) {
		    st1 = stars[1]->GetOrbit();
			ndx1 = 1;
		}
		if(stars[2] != NULL) {
			st2 = stars[2]->GetOrbit();
			ndx2 = 2;
		}
	} else {
		if(stars[ndx]->GetOrbit() == ORBIT_CLOSE)
		    sprintf(buff, "Close");
		else
		    sprintf(buff, "%d.0", stars[ndx]->GetOrbit());
		if(ndx < 3) {
			indent = 0;
			if(stars[ndx + 2] != NULL) {
			    st1 = stars[ndx + 2]->GetOrbit();
				ndx1 = 1;
			}
		} else
		    indent = 3;
	}

	strfixsize(buff, 14);
	if(indent > 0)
	    strrot(buff, indent, TRUE);
	if((stars[ndx]->GetOrbit() < 10) && (stars[ndx]->GetOrbit() >= 0))
	    strrot(buff, 1, TRUE);
	fprintf(fp, buff);

	sprintf(buff, "%s", stars[ndx]->GetName());
	strfixsize(buff, name_len + 1);
	fprintf(fp, buff);

	stars[ndx]->GetStarStr(buff);
	fprintf(fp, "%s\n", buff);

	i = 0;
	while((bo = stars[ndx]->GetWorldOrbit(i)) != NULL) {
		if(ndx1 > -1) {
			if(st1 < bo->GetOrbit()) {
				SaveStar(fp, ndx1, name_len);
				ndx1 = -1;
			}
		}
		if(ndx2 > -1) {
			if(st2 < bo->GetOrbit()) {
				SaveStar(fp, ndx2, name_len);
				ndx2 = -1;
			}
		}
		SaveOrbit(fp, bo, indent + 3, name_len);
		i++;
	}
	if(ndx1 > -1)
		SaveStar(fp, ndx1, name_len);
	if(ndx2 > -1)
	    SaveStar(fp, ndx2, name_len);
}


void
SystemGenerate::SaveOrbit(FILE *fp, BuildOrbit *bo, int indent, int name_len)
{
char buff[120],buff2[120];
int i=0;
BuildOrbit *sat;

    bo->FormatOrbit(buff);
	strfixsize(buff, 14);
//fprintf(stderr, "orb: %d <%s>", indent, buff);
	if(indent > 0)
	    strrot(buff, indent, TRUE);
	strrot(buff, 1, FALSE);
//fprintf(stderr, "<%s>\n", buff);
	fprintf(fp, buff);

	sprintf(buff, "%s", bo->GetName());
	strfixsize(buff, name_len + 1);
	fprintf(fp, buff);

	sprintf(buff, "%s", bo->GetData());
	strfixsize(buff, 9);
	fprintf(fp, buff);
	
	// FIX need bases
	sprintf(buff, "%c %s", ' ',  bo->GetFac(buff2));
	strfixsize(buff, 35);
	fprintf(fp, buff);

	// FIX need zone and alleg
	sprintf(buff, "   \n");
	fprintf(fp, "%s", buff);

	while((sat = bo->GetSat(i)) != NULL) {
		SaveOrbit(fp, sat, indent + 3, name_len);
		i++;
	}
}

int
SystemGenerate::LongestName(void)
{
int ret=0,val;
int i,j,k;
char *ptr;
BuildOrbit *bo,*sat;

    for(i = 0;i < MAX_STARS;i++)
	    {
		if(stars[i] != NULL)
		    {
			ptr = stars[i]->GetName();
			if(NULL == ptr)
				val = 0;
			else
				val = strlen(ptr);
			if(val > ret)
			    ret = val;
			j = 0;
			while((bo = stars[i]->GetWorldOrbit(j)) != NULL)
			    {
				ptr = bo->GetName();
				if(NULL == ptr)
					val = 0;
				else
					val = strlen(ptr);
				if(val > ret)
				    ret = val;
				k = 0;
				while((sat = bo->GetSat(k)) != NULL)
				    {
					ptr = sat->GetName();
					if(NULL == ptr)
						val = 0;
					else
						val = strlen(ptr);
					if(val > ret)
					    ret = val;
					k++;
					}
				j++;
				}
			}
		}

	return(ret);
}

// ----------------------------------------------------------------------
BuildOrbit *
SystemGenerate::GetMWOrbit(void)
{
int st,o,s;
BuildStar *bs;
BuildOrbit *bo;

	GetMWndx(&st, &o, &s);
	bs = stars[st];
	bo = bs->GetWorldOrbit(o);
	if(s >= 0)
		bo = bo->GetSat(s);

	return(bo);
}

int
SystemGenerate::GetNumGG(void)
{
	return(GetItem(TRUE));
}

int
SystemGenerate::GetNumBelts(void)
{
	return(GetItem(FALSE));
}

char *
SystemGenerate::GetStarType(int ndx, char *buff)
{
BuildStar *bs;

	if((bs = stars[ndx]) != NULL) {
		return(bs->GetStarStr(buff));
	}
	return(NULL);
}

int
SystemGenerate::GetItem(bool gg)
{
int i,j,tot=0;
char *ptr;
BuildStar *bs;
BuildOrbit *bo;

	for(i = 0;i < MAX_STARS;i++) {
		if((bs = stars[i]) != NULL) {
			j = 0;
			while((bo = bs->GetWorldOrbit(j)) != NULL) {
				ptr = bo->GetData();
				if(gg) {
					if(strstr(ptr, "GG") != NULL)
						tot++;
				} else {
					if('0' == ptr[1])
						tot++;
				}
				j++;
			}
		}
	}

	return(tot);
}

// ----------------------------------------------------------------------
void
SystemGenerate::RemoveStars(sys_conf *cfg)
{
int i;

    for(i =0;i < MAX_STARS;i++)
	    {
		if(stars[i] != NULL)
		    {
			delete stars[i];
			stars[i] = NULL;
			}
		}
}

void
SystemGenerate::RemoveKnown(sys_conf *cfg)
{
int i;

    for(i =0;i < MAX_STARS;i++)
	    {
		if(stars[i] != NULL)
		    stars[i]->ClearKnown();
		}
}

void 
SystemGenerate::UnFillIn(sys_conf *cfg)
{
int i;

    for(i =0;i < MAX_STARS;i++)
	    {
		if(stars[i] != NULL)
		    stars[i]->ClearOrbits();
		}
}

void 
SystemGenerate::UnFinish(sys_conf *cfg)
{
int i,j,k;
BuildOrbit *bo,*sat;

    for(i = 0;i < MAX_STARS;i++)
	    {
		if(stars[i] != NULL)
		    {
			j = 0;
			while((bo = stars[i]->GetWorldOrbit(j)) != NULL)
			    {
				char *data;

				data = bo->GetData();
				if((data[1] != 'm') && (data[1] != 'a') &&
				   (!((i == mw_st) && (j == mw_orb) && (mw_sat == -1))))
				    {
				    data[0] = data[5] = data[6] = data[7] = ' ';
					data[8] = 0;
					}
				bo->ClearFac();
				
				k = 0;
				while((sat = bo->GetSat(k)) != NULL)
				    {
					data = sat->GetData();
					if(!((i == mw_st) && (j == mw_orb) && (k == mw_sat)))
					    {
						data[0] = data[5] = data[6] = data[7] = ' ';
						data[8] = 0;
						}
					sat->ClearFac();
					k++;
					}
				j++;
				}
			}
	  }
}

// ----------------------------------------------------------------------
#ifdef SG_DEBUG
void
SystemGenerate::DumpSG(sys_conf *cfg)
{
int i;

    fprintf(stderr, "========================================\n");
    if(cfg == NULL)
	    fprintf(stderr, "No sys_conf!!\n");
	else
	    {
		fprintf(stderr, "Name: ");
		if(cfg->name == NULL)  fprintf(stderr, "None");
		else  fprintf(stderr, cfg->name);
		fprintf(stderr, " Sect: ");
		if(cfg->sect == NULL)  fprintf(stderr, "None");
		else  fprintf(stderr, cfg->sect);
		fprintf(stderr, "\nSSect: ");
		if(cfg->ssect == NULL)  fprintf(stderr, "None");
		else  fprintf(stderr, cfg->ssect);
		fprintf(stderr, " Loc: ");
		if(cfg->loc == NULL)  fprintf(stderr, "None");
		else  fprintf(stderr, cfg->loc);
		fprintf(stderr, "\nBelts: %d  GG: %d  UWP:<%s>\n", cfg->belts,
				cfg->gg, cfg->uwp);
		for(i = 0;i < 3;i++)
		    fprintf(stderr, "%s ", cfg->stars[i]);
		fprintf(stderr, "out:%s\n", cfg->out_file);
		fprintf(stderr, "\nBases: <%s>  Alleg: <%s>  Zone: <%s>\n",
				cfg->bases, cfg->alleg, cfg->zone);
		fprintf(stderr, "Main: %d   Rand: %d  Quiet:%d\n\n",
				cfg->no_main, cfg->full_rand, cfg->quiet_mode);
		}

	for(i = 0;i < MAX_STARS;i++)
	    {
		fprintf(stderr, "========================================\n");
		fprintf(stderr, "Star %d\n", i + 1);
		if(stars[i] != NULL)
		    stars[i]->DumpStar();
		}

	exit(-1);
}

void
BuildStar::DumpStar(void)
{
int i;

    fprintf(stderr, "Orb: %d  1st: %d  last:%d  hab: %d\n", orbit,
			GetFirstOrbit(), num_orbits, GetHabZone());

	fprintf(stderr, "Empty: ");
	for(i = 0;i < 3;i++)
	    fprintf(stderr, "%d ", mt_orb[i]);
	fprintf(stderr, " Capt: ");
	for(i = 0;i < 3;i++)
	    fprintf(stderr, "%d ", capt_orb[i]);
	fprintf(stderr, "\nGG: ");
	for(i = 0;i < MAX_GG;i++)
	    fprintf(stderr, "%d ", gg_orbs[i]);
	fprintf(stderr, " Belts: ");
	for(i = 0;i < MAX_BELTS;i++)
	    fprintf(stderr, "%d ", belt_orbs[i]);
	fprintf(stderr, "MW: %d\n\n", mw_orb);
}

#endif

// ----------------------------------------------------------------------
