/*
 * File:      mtu_dir.cpp
 * Purpose:   really crudy way of handling dirs and extensions for mtu
 * Author:    Mark A. Nordstrand
 * Created:   9/1/2000
 * Updated:
 * Copyright: LGPL
Traveller is a registered trademark of Far Future Enterprises.
Portions based upon material Copyright 1977-2002 Far Future Enterprises.
 */

/* rcsid[] = "$RCSfile: mtu_dir.cpp,v $ $Revision: 1.10 $ $Author: man $ $Date: 2002/06/05 04:51:07 $" */

/*
typical layout (and stuff we'll need to find):

<mut dir>
	|
	+- mtu.dat 					(main mtu file)
	+- <universe>
		 +- <unverse>.dat		(main universe file)
		 +- <sector>
		 	  +- <sector>.sec	(main sector file)
			  +- <sector>.dat	(aux sector file)
			  +- desc
			  	  +- sector.txt	(sector desc)
			  	  +- <ssec>.txt	(subsector desc)
			  	  +- <loc>.txt	(location desc)
			  +- enc
			  	  +- <loc>.enc	(location enc table)
			  +- sys
			  	  +- <loc>.sys	(location system)
			  +- world
			  	  +- <loc>.wld	(location world detail)
	+- data
		 +- (data files)

*/

#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include "mtu_dir.h"

// XXX fairly unix specific values
#define DIR_DELIM		'/'
#define EXT_DELIM		'.'
#define NOTES_DIR		"notes"
#define NOTES_EXT		"txt"
#define ENC_DIR			"enc"
#define ENC_EXT			"enc"
#define SYS_DIR			"sys"
#define SYS_EXT			"sys"
#define WORLD_DIR		"world"
#define WORLD_EXT		"wld"
#define DATA_DIR		"data"
#define MTU_DIR			"mtu"

#define DEFAULT_PERM	0755

#define MAX_FILE_LENGTH		256

// ===========================================================================
MTUDir::MTUDir(char *mtu_dir)
{
	main_dir = NULL;
	if((mtu_dir != NULL) || (mtu_dir[0] != 0)) {
		main_dir = new char[strlen(mtu_dir) + 1];
		strcpy(main_dir, mtu_dir);
	}
	univ_dir = NULL;
	sect_dir = NULL;
	for_3d = FALSE;
}

MTUDir::~MTUDir()
{
	if(main_dir != NULL)
		delete(main_dir);
	ClearNames();
}

void
MTUDir::SetUniv(char *u)
{
	if(univ_dir != NULL)
		delete univ_dir;
	univ_dir = new char[strlen(u) + 1];
	strcpy(univ_dir, u);
}

void
MTUDir::SetSect(char *s)
{
	if(sect_dir != NULL)
		delete sect_dir;
	sect_dir = new char[strlen(s) + 1];
	strcpy(sect_dir, s);
	// implied
	for_3d = FALSE;
}

void
MTUDir::ClearNames(void)
{
	if(univ_dir != NULL)
		delete(univ_dir);
	univ_dir = NULL;
	ClearSect();
}

void 
MTUDir::ClearSect(void)
{
	if(sect_dir != NULL)
		delete(sect_dir);
	sect_dir = NULL;
}

// given a filename (sect or sys) crack out the univ and sect portion
//  and set univ_dir and sect_dir
void
MTUDir::CrackNames(char *f_name)
{
char *ptr;

// ../../data/mtu/classic/spinmar/world
//                ^univ^^ ^sect^^
	ClearNames();
	if((main_dir != NULL) && (strlen(f_name) > strlen(main_dir))) {
		ptr = &f_name[strlen(main_dir)];
		univ_dir = ExtractDir(ptr);
		if(ptr != NULL) ptr++;
		ptr = strchr(ptr, DIR_DELIM);
		sect_dir = ExtractDir(ptr);
		// is this appropriate?
		if(sect_dir != NULL)
			for_3d = FALSE;
		else
			for_3d = TRUE;
	}
}

// given a string create a new string up to DIR_DELIM in the old string
char *
MTUDir::ExtractDir(char *buff)
{
char *ptr,*new_ptr=NULL;
int i=0;

	if(buff != NULL) {
		// go past first DIR_DELIM
		while((DIR_DELIM == *buff) && (0 != *buff))
			buff++;
		// find next DIR_DELIM
		ptr = strchr(buff, DIR_DELIM);
		// create a new string and copy stuff in
		if(ptr != NULL) {
			new_ptr = new char[ptr - buff + 1];
			while(buff != ptr) {
				new_ptr[i] = *buff;
				buff++;
				i++;
			}
			new_ptr[i] = 0;
		}
	}

	return(new_ptr);
}

// ===========================================================================
// misc dir building (in general, default to cwd)
char *
MTUDir::GetUnivDir(char *buff)
{
	buff[0] = 0;
	if(main_dir != NULL) {
		if(univ_dir != NULL) 
			sprintf(buff, "%s%c%s", main_dir, DIR_DELIM, univ_dir);
	}

	return(buff);
}

char *
MTUDir::GetSectDir(char *buff)
{
	buff[0] = 0;
	if(main_dir != NULL) {
		if(univ_dir != NULL) {
			if(sect_dir != NULL)
				sprintf(buff, "%s%c%s%c%s", main_dir, DIR_DELIM, univ_dir,
					DIR_DELIM, sect_dir);
			else if(for_3d)
				sprintf(buff, "%s%c%s", main_dir, DIR_DELIM, univ_dir);
		}
	}

	return(buff);
}

char *
MTUDir::GetDataDir(char *buff)
{
	return(BuildDir(buff, DATA_DIR, FALSE));
}

char *
MTUDir::GetNotesDir(char *buff)
{
	return(BuildDir(buff, NOTES_DIR));
}

char *
MTUDir::GetSysDir(char *buff)
{
	return(BuildDir(buff, SYS_DIR));
}

char *
MTUDir::GetEncDir(char *buff)
{
	return(BuildDir(buff, ENC_DIR));
}

char *
MTUDir::GetWorldDir(char *buff)
{
	return(BuildDir(buff, WORLD_DIR));
}

// ===========================================================================
char *
MTUDir::BuildDir(char *buff, char *str, bool extra)
{
	buff[0] = 0;
	if(main_dir != NULL) {
		if(extra) {
			if(univ_dir != NULL) {
				if(sect_dir != NULL) 
					sprintf(buff, "%s%c%s%c%s%c%s%c%s%c", main_dir, DIR_DELIM, 
							MTU_DIR, DIR_DELIM, univ_dir, DIR_DELIM, sect_dir, 
							DIR_DELIM, str, DIR_DELIM);
				else if(for_3d)
					sprintf(buff, "%s%c%s%c%s%c%s%c", main_dir, DIR_DELIM, 
							MTU_DIR, DIR_DELIM, univ_dir, DIR_DELIM, str, 
							DIR_DELIM);
			}
		} else {
			sprintf(buff, "%s%c%s%c%s%c", main_dir, DIR_DELIM, MTU_DIR, 
						DIR_DELIM, str, DIR_DELIM);
		}
	}

	return(buff);
}

// ===========================================================================
//  blindly set up required dirs
void
MTUDir::VerifyDirs(unsigned int which)
{
char buff[MAX_FILE_LENGTH];

	if(which & MTUD_NOTES) {
		GetNotesDir(buff);
		mkdir(buff, DEFAULT_PERM);
	}

	if(which & MTUD_SYS) {
		GetSysDir(buff);
		mkdir(buff, DEFAULT_PERM);
	}

	if(which & MTUD_ENC) {
		GetEncDir(buff);
		mkdir(buff, DEFAULT_PERM);
	}

	if(which & MTUD_WORLD) {
		GetWorldDir(buff);
		mkdir(buff, DEFAULT_PERM);
	}
}

