/*
 * File:    	syst.h
 * Purpose:		define a system class
 * Author:	
 * Created:	
 * Updated:	
 * Copyright: LGPL.
 *            Traveller is a registered trademark of Far Future Enterprises.
 */

/* rcsid[] = "$RCSfile: syst.h,v $ $Revision: 1.5 $ $Author: man $ $Date: 2001/12/02 18:11:58 $" */

#ifndef SYST_H
#define SYST_H

#ifdef SYS_DEBUG
#include <stdio.h>
#endif

#include "planet.h"
#include "d_star.h"
//#include "codetbl.h"
//#include "basetbl.h"
//#include "t_code.h"
#include "d_sect.h"

// this is where we _set_ max stars
#define MAX_STARS		5

// this should be more private:
#define MAX_AU_TABLE    20

// define indexing for stars
typedef enum STAR_INDEX {
    SI_UNKNOWN=0,
	SI_PRIMARY,
	SI_1ST_COMP,
	SI_1ST_TERT,
	SI_2ND_COMP,
	SI_2ND_TERT
	};

// orbit type defines (poorly implemented):
typedef enum ORBIT_TYPE {
    OT_UNKNOWN=0,
    OT_STAR,
	OT_GG,
	OT_BELT,
	OT_WORLD,
	OT_SATILLITE,
	OT_RING
    };
  
// parsing enums
typedef enum SYS_PARSE_PARAMS {
	SYS_PARSE_ORBIT=0,
	SYS_PARSE_NAME,
	SYS_PARSE_UWP,
	SYS_PARSE_BASE,
	SYS_PARSE_CODE,
	SYS_PARSE_ZONE,
	SYS_PARSE_ALLEG
	};

typedef enum ORB_PARSE_PARAMS {
	ORB_PARSE_PRIMARY=0,
	ORB_PARSE_PRIMARY_SAT,
	ORB_PARSE_SECONDARY,
	ORB_PARSE_SECONDARY_SAT,
	ORB_PARSE_TERTIARY,
	ORB_PARSE_TERTIARY_SAT,
	};

// ===========================================================================
// forward references
class OrbitTable;
class BaseOrbit;
class TertOrbit;
class CompOrbit;
class PrimOrbit;


// ===========================================================================
// ===========================================================================
// orbit section
// bitmasks to help determine what each orbit contains
// low 4 bits determine what the orbit _is_
// next 4 are for what this orbit orbits
// NOTE that the primary is a special case
#define ORBIT_IS_STAR          0x0001
#define ORBIT_IS_WORLD         0x0002
#define ORBIT_IS_SAT           0x0004
#define ORBIT_IS_PRIMARY       0x0008
#define ORBIT_ORBITS_PRIMARY   0x0010
#define ORBIT_ORBITS_COMP      0x0020
#define ORBIT_ORBITS_TERT      0x0040
#define ORBIT_SECOND_STAR      0x0080

// some bitmasks
#define ORBIT_MASK_FLAGS       0x000f
#define ORBIT_ORBIT_FLAGS      0x00f0
// reserve some (for the viewer?)
#define ORBIT_RESERVED         0xf000

// ---------------------------------------------------------------------------
// simplest forms of orbits
class BaseOrbit : public ListData
{
friend class SatOrbit;
friend class SimpleOrbit;
friend class StarOrbit;
friend class WorldOrbit;
friend class TertOrbit;
friend class CompOrbit;
friend class PrimOrbit;
public:
	BaseOrbit(BaseOrbit *p, unsigned short mask_val, float o);
	~BaseOrbit();

	virtual char *GetDesc(char *)=0;
//	virtual char *GetOrbStr(char *, int name_w, int uwp_w, 
//			int base_w, int code_w)=0;


	virtual char *FormatName(char *buffer, int width)=0;
	virtual char *FormatUWP(char *buffer, int width)=0;
	virtual char *FormatBase(char *buffer, int width)=0;
	virtual void GetCodes(unsigned long *co, unsigned long *cx,
						  unsigned long *cs)=0;
	virtual char *FormatZone(char *buffer, int width)=0;
	virtual char *FormatAlleg(char *buffer, int width)=0;

	virtual BaseOrbit *GetNthOrbit(int ndx)=0;

	virtual bool IsStar(void)=0;
	virtual bool IsLargeGG(void)=0;
	virtual bool IsSmallGG(void)=0;
	virtual bool IsBelt(void)=0;
	virtual bool IsRing(void)=0;
	
//	unsigned short GetAllowed(void) { return(orb_mask & T_ORBIT_ALLOW_FLAGS);};
	unsigned short GetOrbType(void) { return(orb_mask & ORBIT_MASK_FLAGS); };
	unsigned short GetOrbOrbitType(void)
	  { return(orb_mask & ORBIT_ORBIT_FLAGS); };
	unsigned short GetReserved(void )
	  { return(orb_mask & ORBIT_RESERVED); };
	unsigned short GetOrbMask(void) { return(orb_mask); };
	float GetOrbit(void) { return(orbit); };
	
	char *FormatOrbit(char *buffer, short params[], int width);
	char *FormatOrbit(char *buffer);
	char *GetFormatedOrbit(char *buffer, bool prim_flag=FALSE);

	int GetHabitible(void);

	BaseOrbit *GetParent(void) { return(parent); };

	STAR_INDEX GetStarIndex(void);
	virtual ORBIT_TYPE GetEnumType(void)=0;

	void GetOrbitArray(int *orbs);
// really uncomfortable w/ these (but needed to move an orbit):
    void SetOrbit(float o) { orbit = o; };
    void SetMask(unsigned short m) { orb_mask = m; };

// helpers for the viewer
	long GetID(void) { return(v_id); };
	void SetID(long id) { v_id = id; };

// save/get the picture index
	int GetImageID(void) { return(i_id); };
	void SetImageID(int i) { i_id = i; };
#ifdef SYS_DEBUG
	void dump(FILE *fp);
#endif
private:
	float				orbit;					// actual orbit
	unsigned short		orb_mask;				// details of this orbit

	long				v_id;					// id for the viewer
	int					i_id;					// id for pretty picture

// keep reference to the 'parent' orbit
	BaseOrbit			*parent;				// what this orbit orbits
};

// the orbit table
class OrbitTable : public LinkedList
{
public:
	OrbitTable(unsigned short allow_mask);
	~OrbitTable();

	bool AppendOrbit(BaseOrbit *orbit);

	short GetAllowType(void) { return(type); };

#ifdef SYS_DEBUG
	void dump(FILE *fp);
#endif
private:
	unsigned short		type;
	

};

// ---------------------------------------------------------------------------
// Satillite orbit
//  note that it does not have any sub-orbits
class SatOrbit : public BaseOrbit
{
public:
	SatOrbit(BaseOrbit *p, unsigned short mask_val, float o, planet *s);
	~SatOrbit();

	planet *GetSat(void) { return(sat); };
	
// take care of pure virtuals:
	char *GetDesc(char *);
//	char *GetOrbStr(char *, int name_w, int uwp_w, 
//			int base_w, int code_w);
	char *FormatName(char *buffer, int width);
	char *FormatUWP(char *buffer, int width);
	char *FormatBase(char *buffer, int width);
	void GetCodes(unsigned long *c1, unsigned long *c2, unsigned long *c3);
	char *FormatZone(char *buffer, int width);
	char *FormatAlleg(char *buffer, int width);

	BaseOrbit *GetNthOrbit(int ndx) { return(NULL); };

	bool IsStar(void) { return(FALSE); };
	bool IsLargeGG(void) { return(FALSE); };
	bool IsSmallGG(void) { return(FALSE); };
	bool IsBelt(void) { return(FALSE); };
	bool IsRing(void);

	ORBIT_TYPE GetEnumType(void);

#ifdef SYS_DEBUG
	void dump(FILE *fp);
#endif
private:
	planet		*sat;
};

// ---------------------------------------------------------------------------
// orbit that has sub-orbits
enum ADD_ORBIT_RESULT {
	ADD_ORBIT_BAD=0,
	ADD_ORBIT_NO,
	ADD_ORBIT_OK
};

class SimpleOrbit : public BaseOrbit
{
friend class StarOrbit;
friend class WorldOrbit;
friend class TertOrbit;
friend class CompOrbit;
friend class PrimOrbit;
public:
	SimpleOrbit(BaseOrbit *p, unsigned short mask_val, float o);
	~SimpleOrbit();

// the pure virtuals should still be at this level.....
//  but, we need wrappers for the orbit tables
    BaseOrbit *AppendToOrbitTable(unsigned short star_mask, float o,
								  planet *p);
    BaseOrbit *AppendToOrbitTable(unsigned short star_mask, float o,
								  DetailStar *r);

	bool AddOrbit(BaseOrbit *bo, float o);
	
	// utils to access the OrbitTable
	BaseOrbit *GetFirstOrbit(void);
	BaseOrbit *GetNthOrbit(int ndx);

	bool DeleteOrbit(BaseOrbit *bo);

	int GetIntOrbit(void) { return((int) (orbit + 0.5)); };
	
#ifdef SYS_DEBUG
	void dump(FILE *fp);
#endif
private:
	OrbitTable			*orbits;				// orbits of this orbit

//  add a couple more pure virtuals....
// these will do the front end work for adding to the orbit table
	virtual BaseOrbit *ValidateOrbit(ADD_ORBIT_RESULT *stat, 
		unsigned short mask, float o, planet *p) = 0;
	virtual BaseOrbit *ValidateOrbit(ADD_ORBIT_RESULT *stat, 
		unsigned short mask, float o, DetailStar *r) = 0;

// and a routine which does the actual adding to the orbit table
	bool CheckOrbitStat(ADD_ORBIT_RESULT stat, BaseOrbit *bo);
};

class WorldOrbit : public SimpleOrbit
{
public:
	WorldOrbit(BaseOrbit *p, unsigned short mask_val, float o, planet *w);
	~WorldOrbit();


	planet *GetWorld(void) { return(world); };
	SatOrbit *GetSat(short index);

// take care of pure virtuals:
	char *GetDesc(char *);
//	char *GetOrbStr(char *, int name_w, int uwp_w, 
//			int base_w, int code_w);
	char *FormatName(char *buffer, int width);
	char *FormatUWP(char *buffer, int width);
	char *FormatBase(char *buffer, int width);
	void GetCodes(unsigned long *c1, unsigned long *c2, unsigned long *c3);
	char *FormatZone(char *buffer, int width);
	char *FormatAlleg(char *buffer, int width);

	bool IsStar(void) { return(FALSE); };
	bool IsLargeGG(void);
	bool IsSmallGG(void);
	bool IsBelt(void);
	bool IsRing(void) { return(FALSE); };

	float GetKmOrbit(float);

	ORBIT_TYPE GetEnumType(void);

#ifdef SYS_DEBUG
	void dump(FILE *fp);
#endif
private:
	planet			*world;

	BaseOrbit *ValidateOrbit(ADD_ORBIT_RESULT *stat, 
		unsigned short mask, float orb, planet *p);
	BaseOrbit *ValidateOrbit(ADD_ORBIT_RESULT *stat, 
		unsigned short mask, float orb, DetailStar *r);
};

class StarOrbit : public SimpleOrbit
{
friend class TertOrbit;
friend class CompOrbit;
friend class PrimOrbit;
public:
	StarOrbit(BaseOrbit *p, unsigned short mask_val, float o, DetailStar *r);
	~StarOrbit();

	DetailStar *GetStar(void) { return(star); };
	SimpleOrbit *GetOrbit(short index);

// take care of these pure virtuals:
	char *GetDesc(char *);
//	char *GetOrbStr(char *, int name_w, int uwp_w, 
//			int base_w, int code_w);
	char *FormatName(char *buffer, int width);
	char *FormatUWP(char *buffer, int width);
	char *FormatBase(char *buffer, int width);
	void GetCodes(unsigned long *c1, unsigned long *c2, unsigned long *c3);
	char *FormatZone(char *buffer, int width);
	char *FormatAlleg(char *buffer, int width);

	bool IsStar(void) { return(TRUE); };
	bool IsLargeGG(void) { return(FALSE); };
	bool IsSmallGG(void) { return(FALSE); };
	bool IsBelt(void) { return(FALSE); };
	bool IsRing(void) { return(FALSE); };

//
	bool AppendOrbit(unsigned short star_mask, float o, planet *p);
	bool AppendOrbit(unsigned short star_mask, float o, DetailStar *r);

//
    float GetAUOrbit(float);

	ORBIT_TYPE GetEnumType(void) { return(OT_STAR); };
	
#ifdef SYS_DEBUG
	void dump(FILE *fp);
#endif
private:
	DetailStar		*star;
// parsing helper (last world entered)
	WorldOrbit		*world;

// make the decision about adding to either the internal orbit table or
//	the world helper''s orbit table
	WorldOrbit *CheckWorldOrbit(ADD_ORBIT_RESULT *stat, 
		unsigned short mask, float orb, planet *p);

    static float AUOrbit[MAX_AU_TABLE];
};

class TertOrbit : public StarOrbit 
{
public:
	TertOrbit(BaseOrbit *p, unsigned short mask_val, float o, DetailStar *r);
	~TertOrbit();

#ifdef SYS_DEBUG
	void dump(FILE *fp);
#endif

private:

	BaseOrbit *ValidateOrbit(ADD_ORBIT_RESULT *stat, 
		unsigned short mask, float orb, planet *p);
	BaseOrbit *ValidateOrbit(ADD_ORBIT_RESULT *stat, 
		unsigned short mask, float orb, DetailStar *r);
};

class CompOrbit : public StarOrbit 
{
public:
	CompOrbit(BaseOrbit *p, unsigned short mask_val, float o, DetailStar *r);
	~CompOrbit();

	TertOrbit *GetTert(void) { return(tert); };

#ifdef SYS_DEBUG
	void dump(FILE *fp);
#endif
private:
	TertOrbit		*tert;

	BaseOrbit *ValidateOrbit(ADD_ORBIT_RESULT *stat, 
		unsigned short mask, float orb, planet *p);
	BaseOrbit *ValidateOrbit(ADD_ORBIT_RESULT *stat, 
		unsigned short mask, float orb, DetailStar *r);
};

class PrimOrbit : public  StarOrbit
{
public:
	PrimOrbit(unsigned short mask_val, DetailStar *r);
	~PrimOrbit();

	CompOrbit *GetComp(int index);
	TertOrbit *GetTert(int index);

#ifdef SYS_DEBUG
	void dump(FILE *fp);
#endif
private:
	CompOrbit		*comp[2];

	BaseOrbit *ValidateOrbit(ADD_ORBIT_RESULT *stat, 
		unsigned short mask, float orb, planet *p);
	BaseOrbit *ValidateOrbit(ADD_ORBIT_RESULT *stat, 
		unsigned short mask, float orb, DetailStar *r);

};






// ==========================================================================
// orbit check enum
typedef enum ORBIT_CHECK_RESP {
  OC_OK=0,
  OC_TOO_MANY_STARS,           // too many stars orbiting _this_ star
  OC_NO_ORB,                   // orbit is inside star
  OC_CONFLICTS,                // conflicts w/ existing orbit(s)
  OC_STAR_CONFLICTS,           // conflicts w/ existing star orbit
  OC_NO_SAT,                   // this orbit cannot have satillites
  OC_NOT_A_WORLD,              // cannot place this here
//  OC_STAR_ORB,
// these are warnings
//  OC_LOOSE_ORB,
  OC_LOOSE_SAT                 // will loose satillites
};

class System
{
public:
//	System();
	System(TCodes *tc);
	~System();

	void Clear(void);

	int LoadFile(char *);
	bool SaveSystem(char *);

	char *GetName(void) { return(name); };
	char *GetLoc(void) { return(loc); };
	char *GetSectName(void) { return(sect_name); };
	char *GetSSName(void) { return(ss_name); };

	void SetName(char *buff);
	void SetLoc(char *buff);
	void SetSectName(char *buff);
	void SetSSName(char *buff);
	
	PrimOrbit *GetPrimOrbit(void) { return(prim_orbit); };
	BaseOrbit *GetCurrentOrbit(void) { return(cur_orbit); };
	void SetCurrentOrbit(int index);
	void SetCurrentOrbitV(long index);

	BaseOrbit *GetIndexedOrbit(int index);
//XXXXXXXX
	BaseOrbit *GetOrbit(STAR_INDEX st, short pl, short sat);
	char *GetListing(STAR_INDEX st, short pl, short sat, char *buff);
	char *GetListing(int ndx, char *buff);

	DetailSector		*sect;

	// misc formating helpers
	char *FormatBase(char code, char *buff)
	    { return(sect->FormatBase(code, buff)); };
	char *FormatSystem(unsigned long code, char *buff) 
		{ return(sect->FormatSystem(code, buff)); };
	char *FormatTrade(unsigned long code, char *buff) 
		{ return(sect->FormatTrade(code, buff)); };
	char *FormatOther(unsigned long code, char *buff) 
		{ return(sect->FormatOther(code, buff)); };
	char *FormatZone(char code, char *buff)
	    { return(sect->FormatZone(code, buff)); };
	char *FormatAlleg(char *code, char *buff)
	    { return(sect->FormatAlleg(code, buff)); };

//	CodeTable *GetCodeTable(void)
//	    { return(sect->GetCodeTable()); };
//	BaseTable *GetBaseTable(void)
//	    { return(sect->GetBaseTable()); };
//	CodeTable *GetAllegTable(void)
//	    { return(sect->GetAllegTable()); };

	bool MoveCurrentOrbit(STAR_INDEX st, float val);
	ORBIT_CHECK_RESP CheckOrbit(STAR_INDEX st, float orb, ORBIT_TYPE ot,
								bool ig=TRUE);

	bool DeleteOrbit(BaseOrbit *bo);

	int GetOrbitIndex(BaseOrbit *bo);
#ifdef SYS_DEBUG
	void dump(FILE *fp);
#endif

	char *GetFileName(void) { return(file_name); };

	int TranslateRelIndex(int ndx, int pndx);
// helpers for viewer
	BaseOrbit *GetVIndexedOrbit(long index);
	int TranslateVIndex(long v);
	int TranslateParentVIndex(long v);

	BaseOrbit *Find(int orb1, int orb2, int orb3, int orb4);

private:
	void InitSystem(void);

	bool ValidOrbit(char *buff, unsigned short *sndx, float *orb);
	bool ParseAndAddOrbit(char *buff);
	void ParseCodes(char *src, unsigned long *co, unsigned long *cx,
		unsigned long *cs);
	char *ParseTwo(char *, int *, int *);
	char *ParseOne(char *, int *);
	char *ParseString(char *);

	DetailStar *GetPrimary(void);

	BaseOrbit *GetIndexedOrbit(BaseOrbit *bo, int index, int *c);

	char *GetListing(BaseOrbit *bo, char *buff);
	
// (global level) look up tables
//	TCode				*codes;

//	OrbitTable			*orbs;
	short				start_parse[SYS_PARSE_ALLEG + 1];
	short				end_parse[SYS_PARSE_ALLEG + 1];
	short				orb_parse[6];

//	StarWOrbits			*stars[MAX_STARS];
//	PrimStar			*primary;
	PrimOrbit			*prim_orbit;
	BaseOrbit			*cur_orbit;

	char				*name;
	char				*sect_name;
	char				*ss_name;
	char				*loc;

	char				*file_name;
};


#endif

