/*
 * File:      	core2mtu
 * Purpose:   	program to convert data from CORE to mtu
 * Author:		Mark A. Nordstrand
 * Created:	
 * Updated:	
 * Copyright:	LGPL
Traveller is a registered trademark of Far Future Enterprises.
Portions based upon material Copyright 1977-2002 Far Future Enterprises.
 */

/* rcsid[] = "$RCSfile: core2mtu.c,v $ $Revision: 1.5 $ $Author: man $ $Date: 2002/06/05 04:51:07 $" */


#include "conv.h"
#include <errno.h>
#include <libgen.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/types.h>
#include <fcntl.h>
#include <time.h>
#include "str_util.h"

#define HEADER_LINE "################################################################################\n"

#define X_OFFSET		9
#define Y_OFFSET		5

/* yuchy global */
static int verb=0;

/*
 * the only notion I can deterimine for coordinate in the CORE data is 
 * the actual filename.  so, the method choosen is to cycle through 
 * [0-9,a-z,A-Z][0-9,a-z,A-Z].GNI looking for anything.  Once a file
 * is found, it is parsed for useful info.  
 *
 * notes on "what's useful":
 * 	Sector: moved to the universe and sector .dat files
 * 	Alt: moved to sector .dat file (at the time of this writing, unused and
 * 	     given a 'M' tag for future use)
 * 	DOS: moved to universe .dat file and used to create the subdir
 * 	Coord: already determined and thrown out
 * 	[A-P]: moved to sector .dat file (subsector names)
 * 	header and data all moved to sector .sec file
 * 	finally, a blurb about this conversion is put in the sector .txt file
 *
 * notes on problems encountered:
 *  at the time of this writing, the header info from 'Bases' on is 
 *  wrong.  the sector program really depends on these, so they are
 *  fixed.  Since they all look to have fixed 'record' placement, this
 *  is fairly simple........  also, the funny column line is added 
 *  (I can't remember if the sector program needs this)
 *  I really dislike not having core sector not be at (0 0) (yes, it is
 *  a Imperial centric view, thankyouverymuch), so this adjustment is
 *  also made.
 *
 * and yup, I play fast-n-loose with open files
 *
 */
static char x_conv_char[] =
	"345ABCDEFGHIJKLMNOPQRSTUVXYZ";
/*	"0123456789ABCDEFGHIJKLMNOPQRSTUVXYZ";*/
static char y_conv_char[] =
	"abcdefghijklmnopqrstuvwxyz";
/*	"0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVXYZ";*/

/* wrapper to strip off \n" */
int 
get_core(char *buff, size_t size, FILE *fp)
{
	if(fgets(buff, BUFFER_SIZE, fp) == NULL) {
		fprintf(stderr, "Problem reading core file.\n");
		return(5);
	}

	while(('\n' == buff[strlen(buff)-1]) || ('\r' == buff[strlen(buff)-1]))
		buff[strlen(buff)-1] = 0;

	return(0);
}

int
convert_file(int x, int y, FILE *core, FILE *dat, char *mtu_dir)
{
char buffer1[BUFFER_SIZE],buffer2[BUFFER_SIZE];
char s_name[BUFFER_SIZE],a_name[BUFFER_SIZE],f_name[BUFFER_SIZE];
int ret,i;
FILE *s_fp,*d_fp,*t_fp;
time_t t;
struct tm *l_tm;

	/* probably could use better error handling...... */
	if((ret = get_core(buffer1, BUFFER_SIZE, core)) != 0)
		return(6);
	sprintf(s_name, "%s", &buffer1[8]);

	if((ret = get_core(buffer1, BUFFER_SIZE, core)) != 0)
		return(6);
	sprintf(a_name, "%s", &buffer1[8]);

	if((ret = get_core(buffer1, BUFFER_SIZE, core)) != 0)
		return(6);
	sprintf(f_name, "%s", &buffer1[8]);

	/* throw out next 2 lines */
	if((ret = get_core(buffer1, BUFFER_SIZE, core)) != 0)
		return(6);
	if((ret = get_core(buffer1, BUFFER_SIZE, core)) != 0)
		return(6);

	/* we know enough to get the mtu files */
	sprintf(buffer1, "%s/%s", mtu_dir, f_name);
	if(mkdir(buffer1, 0775) < 0) {
		if(EEXIST != errno) {
			fprintf(stderr, "Cannot create %s (%d).\n", buffer1, errno);
			return(6);
		}
	} else if(verb > 1)
		fprintf(stderr, "Created %s.\n", buffer1);

	sprintf(buffer1, "%s/%s/%s.dat", mtu_dir, f_name, f_name);
	if((d_fp = fopen(buffer1, "a+")) == NULL) {
		fprintf(stderr, "Cannot open %s (%d).\n", buffer1, errno);
		return(4);
	}

	sprintf(buffer1, "%s/%s/%s.sec", mtu_dir, f_name, f_name);
	if((s_fp = fopen(buffer1, "a+")) == NULL) {
		fprintf(stderr, "Cannot open %s (%d).\n", buffer1, errno);
		return(4);
	}

	fprintf(d_fp, "N:%s\n\n", s_name);
	/* next 16 will be subsector names */
	for(i = 0;i < 16;i++) {
		if((ret = get_core(buffer1, BUFFER_SIZE, core)) != 0)
			return(6);
		fprintf(d_fp, "S:%s\n", buffer1);
	}
	fprintf(d_fp, "\n");


	/* next 11 are the header info (which is wrong--see notes)*/
	for(i = 0;i < 11;i++) {
		if((ret = get_core(buffer1, BUFFER_SIZE, core)) != 0)
			return(6);
	}

	fprintf(s_fp, "\n");
	fprintf(s_fp, "\n");
	fprintf(s_fp, "  1-18: Name\n");
	fprintf(s_fp, " 20-23: HexNbr\n");
	fprintf(s_fp, " 25-33: UWP\n");
	fprintf(s_fp, "    35: Bases\n");
	fprintf(s_fp, " 37-51: Codes & Comments\n");
	fprintf(s_fp, "    53: Zone\n");
	fprintf(s_fp, " 55-57: PBG\n");
	fprintf(s_fp, " 59-60: Allegiance\n");
	fprintf(s_fp, " 62-80: Stellar Data\n");
	fprintf(s_fp, "\n");
	fprintf(s_fp, "....+....1....+....2....+....3....+....4....+....5....+....6....+....7....+....8\n");

	/* now we have the data, just move it across files */
	while(get_core(buffer1, BUFFER_SIZE, core) == 0) {
		fprintf(s_fp, "%s\n", buffer1);
	}

	fclose(s_fp);
	fclose(d_fp);

	/* update the text file */
	sprintf(buffer1, "%s/%s/desc", mtu_dir, f_name);
	if(mkdir(buffer1, 0775) < 0) {
		if(EEXIST != errno) {
			fprintf(stderr, "Cannot create %s (%d).\n", buffer1, errno);
			return(6);
		}
	} else if(verb > 1)
		fprintf(stderr, "Created %s.\n", buffer1);

	sprintf(buffer1, "%s/%s/desc/sector.txt", mtu_dir, f_name);
	if((t_fp = fopen(buffer1, "a+")) == NULL) {
		fprintf(stderr, "Cannot open %s (%d).\n", buffer1, errno);
		return(4);
	}
	fprintf(t_fp, "\nThe sector data originally from the core site.\n");
	fprintf(t_fp, "  http://www.solstation.com/core/index_en.htm\n");
	t = time(&t);
	l_tm = localtime(&t);
	fprintf(t_fp, "\nConverted on %s\n", asctime(l_tm));
	fclose(t_fp);

	/* update the main .dat file */
	fprintf(dat, "%s:%s:%d:%d:0\n", f_name, s_name, x-X_OFFSET, y-Y_OFFSET);

	return(0);
}

int
conv_gal(char *core_dir, char *mtu_dir, char *mtu_name)
{
FILE *core_fp,*dat_fp,*fp;
int x,y,ret;
char buffer1[BUFFER_SIZE];

	/* make sure we have the destination dir */
	if(mkdir(mtu_dir, 0775) < 0) {
		if(EEXIST != errno) {
			fprintf(stderr, "Cannot create %s (%d).\n", mtu_dir, errno);
			return(5);
		}
	} else if(verb > 1)
		fprintf(stderr, "Created %s.\n", mtu_dir);

	/* create the install file */
	sprintf(buffer1, "%s/mtu.install", mtu_dir);
	if((fp = fopen(buffer1, "a+")) == NULL) {
		fprintf(stderr, "Cannot open %s (%d).\n", buffer1, errno);
		return(4);
	}
	fprintf(fp, "# install file\n");
	fprintf(fp, "DIR=%s\n", basename(mtu_dir));
	fprintf(fp, "NAME=%s\n", mtu_name);
	fclose(fp);

	/* open up the .dat file, shove header stuff in */
	sprintf(buffer1, "%s/%s.dat", mtu_dir, basename(mtu_dir));
	if((dat_fp = fopen(buffer1, "a+")) == NULL) {
		fprintf(stderr, "Cannot open %s (%d).\n", buffer1, errno);
		return(4);
	}
	fprintf(dat_fp, "%s", HEADER_LINE);
	fprintf(dat_fp, "## MTU Sectors for the %s univers\n", mtu_name);

	/* now, we're ready to do the conversion */
	for(x = 0;x < strlen(x_conv_char);x++) {
		for(y = 0;y < strlen(y_conv_char);y++) {
			sprintf(buffer1, "%s/%c%c.GNI", core_dir, 
					y_conv_char[y], x_conv_char[x]);
			if((core_fp = fopen(buffer1, "r")) == NULL) {
				if(verb > 2)
					fprintf(stderr, "Cannot open %s\n", buffer1);
			} else {
				if(ret = convert_file(x, y, core_fp, dat_fp, mtu_dir))
					return(ret);
			}
		}
	}

	/* done */
	fclose(dat_fp);
	return(0);
}

main(int argc, char *argv[])
{
char buffer[256],*n;
char *core_dir=NULL,*mtu_dir=NULL,*mtu_name=NULL;
struct stat stat_struct;
int err=0,i,opt,count=1;

	// process args
	do {
		opt = getopt (argc, argv, "n:i:o:hv");
		switch (opt) {
			case '?':
			case 'h':
				err = 1;
				break;
			case 'i':
				if(NULL == core_dir) {
					core_dir = malloc(strlen(optarg) + 1);
					strcpy(core_dir, optarg);
				} else {
					fprintf(stderr, "Cannot use multiple input dirs.\n");
					err = 2;
				}
				break;
			case 'o':
				if(NULL == mtu_dir) {
					mtu_dir = malloc(strlen(optarg) + 1);
					strcpy(mtu_dir, optarg);
				} else {
					fprintf(stderr, "Cannot use multiple output dirs.\n");
					err = 2;
				}
				break;
			case 'v':
				verb++;
				break;
			case 'n':
				if(NULL == mtu_name) {
					mtu_name = malloc(strlen(optarg) + 1);
					strcpy(mtu_name, optarg);
				} else {
					fprintf(stderr, "Cannot use multiple names.\n");
					err = 2;
				}
				break;
			default:
//				fprintf(stderr, "Unknown command line option: %c\n", opt);
//				err = 3;
				break;
		}
	} while (opt > 0);

	if(0 == err) {
		if((NULL == core_dir) || (NULL == mtu_dir)) {
			fprintf(stderr, "Need both an input and output dir.\n");
			err = 4;
		} else {
			if(NULL == mtu_name)
				mtu_name = "CORE";
			err = conv_gal(core_dir, mtu_dir, mtu_name);
		}
	}

	if(err) {
		fprintf(stderr, "%s: Usage:\n\t%s [-v] -i<core dir> -o<mtu dir>\n",
			argv[0], argv[0]);
		fprintf(stderr, "where:\n");
		fprintf(stderr, "  core dir is the dir containing the GNI files.\n");
		fprintf(stderr, "  mtu dir is the MTU dir.\n");
		exit(err);
	}
}

