/************************************************************************
*  cyber.c  A character generation for CyberPunk 2.0.2.0. (TM)
*           Written in ANSI C by Robert Deck during July-August 1992.
*           The data tables contained within this program are Copyright
*           R. Talsorian Games.  All other code is original to the author
*           and may not be used commercially without his express written
*           permission.
************************************************************************/

/************************************************************************
*   Programmer's note:  I set my tabs to be 3 spaces wide.  8 is too big.
*   I programmed this originally on an Amiga 500, with Lattice C 5.1.  I
*   am now working on an Amiga 2000, with Lattice C 6.3.  CyberGeneration
*   will be included in Cyber 4.X.  See cyber3?.doc for history, etc.
***********************************************************************/

/******************************************************************************
* Joel : Added Sorted order option to lists.
******************************************************************************/

/* Version 3.3 */

#include <ctype.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include "cybrdata.h"

/************************************************************************
*   Definitions
************************************************************************/

#define TRUE 1
#define FALSE 0

/************************************************************************
*   Uniformly-distributed random number generator.
*   Generates a number between 1 and n, with all results having
*   EQUAL probability.  The expressions rand()%n or srand48()%n
*   have uniformly-distributed results ONLY if n is a power of 2
*   ...which is why this routine is needed.
************************************************************************/

/************************************************************************
*   RAND_MAX is not guaranteed to be 32767 on all machines, so we limit
*   the result of rand() to this value to prevent overflows.
*   the formula is:
*   {rand() from 0 to 32767} * (desired range) / (max range) + 1
************************************************************************/

#define unirand(n) ((long)(rand()&0x00007fff)*(n)/0x00008000+1L)


/************************************************************************
*   Global structure declarations
************************************************************************/

struct SIBLING {
    int trait[3];
    };

struct PERSON {
    int style[3],race,mot[5];
    };

struct YEAR {
    int num,pflag;
    char    *event[5];
    struct  PERSON  P;
    };

struct SKILL {
    int name,level,psy;
    };

/************************************************************************
*
*  Global structure Punk, of type CP
*
*   total       the total number of character points for the Punk
*   stat[17]    an array containing values for INT, REF, etc.  See cybrdata.h
*   role        role, chosen in get_role(), from 0 to NUMROLES
*   eb      starting EuroBucks
*   age     starting age
*   sib     number of siblings
*   wage        starting wage, based on Special Ability
*   skills  number of skills
*   puskills    number of points (INT+REF) to spend on pickup skills
*   vamp        boolean, vamp or non-vamp
*   VY          Vampire or WereWolf years character has
*   feed        days allowed between feedings
*   heal        amount healed per day
*   psy     boolean, psyber punk or non-psyber punk
*   wolf        0 = non-wolf, 1 = lycanthrope, 2 = were wolf
*   quirks  number of quirks, either vamp or wolf
*   
*   family[4]array of strings describing punk's family
*   psycat  psy skills catagories allowed
*   QU[25]  array of strings of chosen quirks, either vamp or wolf
*   
*   YE[12]  Years in punk's lifepath
*   
*   SI[7]       Description of Punk's siblings and their relations
*   
*   SK[50]  Skills punk has, including name, level, etc.
*   
************************************************************************/

struct CP {
    int     total,stat[17],role,eb,age,sib,wage,skills,puskills,
            vamp,VY,feed,heal,psy,wolf,quirks;
    char    *family[4],*psycat,*QU[25];
    struct  YEAR        YE[12]; /* Maximum number of years = 12 */
    struct  SIBLING SI[7];  /* Maximum number of siblings = 7 (!) */
    struct  SKILL       SK[50]; /* Maximum number of skills = 50, I think... */
    } Dude;


/************************************************************************
*   Prototyping of functions
************************************************************************/

int  die_roll       ( int num, int size );
char checkchar      ( char * mask );
int  checknum       ( int min, int max );
char *stripspc      ( char *string );
void sort_skills    ( void );
void roll_stats ( struct CP *Punk );
void buy_stats      ( struct CP *Punk );
void move_stats ( struct CP *Punk );
void get_role       ( struct CP *Punk );
void dist_points    ( struct CP *Punk );
void calc_punk      ( struct CP *Punk );
void vampire_punk   ( struct CP *Punk );
void psyber_punk    ( struct CP *Punk );
void wolven_punk    ( struct CP *Punk );
void psy_skill      ( struct CP *Punk );
void pickup_skills( struct CP *Punk );
void vamp_quirks    ( struct CP *Punk );
void wolf_quirks    ( struct CP *Punk );
void make_person    ( struct PERSON *Person );
void life_path      ( struct CP *Punk );
void output         ( struct CP *Punk );


/************************************************************************
*   Die roll - Function returning integer value of a specified die roll.
*
*   die_roll gets two arguments, num and size.  num is the number of dice
*   to be rolled, and size is the size of the die.  die_roll(9,10) = 9d10.
************************************************************************/

int die_roll( int num, int size )

{
    int cnt, roll = 0;
    for (cnt = 0; cnt < num; cnt++) roll += unirand(size);
    return roll;
}


/************************************************************************
*   checkchar - Function returning single, user chosen, char from a string.
*   
*   checkchar gets one argument, mask, which is a string of char, and
*   forces the user to safely input one of the chars that is in mask.
*   NOTE: The first character of mask is the default return value.
************************************************************************/

char checkchar( char * mask )

{
    char getit;
    fflush(stdin);
    printf("[%c]: ",mask[0]);
    while (isspace(getit = getchar())) if ( getit == '\n' ) return ( mask[0] );
    while ( strrchr(mask,getit) == NULL ) {
        printf("Invalid input -- not one of %s.  [%c]: ",mask, mask[0]);
        fflush(stdin);
        while (isspace(getit = getchar())) if ( getit == '\n' ) return ( mask[0] );
    }
    return getit;
}


/************************************************************************
*   checknum - Function returning single, user chosen, int from a range.
*   
*   checknum gets two arguments, min and max.  The user is forced to
*   safely enter an integer value between (and including) min and max.
*   There are no provisions for a default value in this function, yet.
************************************************************************/

int checknum( int min, int max )

{
    int getit;
    fflush(stdin);
    scanf("%d",&getit);
    while ( getit < min || getit > max ) {
        printf("Number out of range.  Min = %d, Max = %d.  : ",min,max);
        fflush(stdin);
        scanf("%d",&getit);
    }
    return getit;
}


/************************************************************************
*   StripSpc - Function returning a string with all white space stripped.
*
*   stripspc gets one argument, the string you want stripped of all white
*   space.  This is used to generate legal filenames in Output().
*************************************************************************/

char *stripspc(char *string)
{
    short i, l = strlen(string), p = 0;
    for (i = 0; i < l; i++) if (!isspace(string[i])) string[p++] = string[i];
    string[p] = '\0';
    return string;
}

/************************************************************************
* Sorts the skill list alphabetically into an index.
************************************************************************/
#define skillcmp(n1, n2) strcmp(short_index(n1)+7,short_index(n2)+7)
void sort_skills( void )
{
int i, j, temp,
     found = 0,
     hi_sorted = 0;
     char ch;
    /* Init the skill array */
    for (i = 0; i < NUMREGSKILLS; i++)
        skill_index[i] = i;

    while (hi_sorted < NUMREGSKILLS)
    {
        found = hi_sorted;
        /* Find the lowest unsorted skill */
        for (j = hi_sorted; j <= NUMREGSKILLS; j++)
         if (skillcmp(j, found) < 0 )
                     found = j;

        /* swap the found one */
        temp = skill_index[found];
        skill_index[found] = skill_index[hi_sorted];
        skill_index[hi_sorted] = temp;
        hi_sorted++;
    }
}
/************************************************************************
*   Roll stats - Procedure generating the random part of the character.
*
*   roll_stats gets the global struct Punk, type CP, and either rolls up
*   the stats/total, or allows the user to enter the stats/total.  If the
*   total character points method is used, buy_stats() is called to allow
*   the user to spend the character points legally.
************************************************************************/

void roll_stats( struct CP *Punk )

{
    int cnt, doit = TRUE;

    printf("\nRoll Stats:\n");
    printf("Enter or Roll stats/totals? (e/r) ");

    if ( checkchar("re") == 'e') {
        printf("Stats or Total? (s/t) ");
        if ( checkchar("st") == 's' ) {
            while ( doit ){
                Punk->total = 0;
                for (cnt = 0; cnt < 9; cnt++){
                    printf("%d %s = ",cnt,statistic[cnt]);
                    Punk->stat[cnt] = checknum(2,10);
                    Punk->total = Punk->total + Punk->stat[cnt];
                }
                printf("Are these acceptable?  (y/n) ");
                if ( checkchar("ny") == 'y') doit = FALSE;
            }
        }
        else {
            while ( doit ) {
                printf("Please enter character points between 20 and 90 ");
                Punk->total = checknum(20,90);
                printf("Character points = %d\n",Punk->total);
                printf("Is this acceptable?  (y/n) ");
                if ( checkchar("ny") == 'y') doit = FALSE;
            }
            buy_stats( Punk );
        }
    }
    else {
        printf("Stats or Total? (s/t) ");
        if ( checkchar ("st") == 's' ) {
            while ( doit ){
                Punk->total = 0;
                for (cnt = 0; cnt < 9; cnt++){
                    Punk->stat[cnt] = die_roll(2,6);
                    if (Punk->stat[cnt] > 10) --cnt;
                    else {
                        printf("%d %s = %2d\n",cnt,statistic[cnt],Punk->stat[cnt]);
                        Punk->total = Punk->total + Punk->stat[cnt];
                    }
                }
                printf("Are these acceptable?  (y/n) ");
                if ( checkchar("ny") == 'y') doit = FALSE;
            }
        }
        else {
            while ( doit ) {
                Punk->total = die_roll(9,10);
                printf("Character points = %d\n",Punk->total);
                printf("Is this acceptable?  (y/n) ");
                if ( checkchar("ny") == 'y') doit = FALSE;
            }
            buy_stats( Punk );
        }
    }
}


/************************************************************************
*   Buy_stats - Procedure lets user spend points legally on stats.
*
*   buy_stats gets the global struct Punk, of type CP.  It is called from
*   roll_stats if the user decides to enter/roll total character points.
************************************************************************/

void buy_stats ( struct CP *Punk )

{
    int cnt,num,doit;
    doit = TRUE;

    printf("\nBuy Stats:\n");

    while ( doit ){
        num = Punk->total; cnt = 0;
        while ( cnt < 9 ){
            printf("%2d character points, %2d stats left->  %s = ",num,9 - cnt,statistic[cnt]);
            Punk->stat[cnt] = checknum(2,10);
            num = num - Punk->stat[cnt];
            if ( num < ( 16 - (cnt*2) ) ){
                printf("Insufficient points for remaining stats.  Restarting.\n");
                num = Punk->total; cnt = 0;
            }
            else if ( cnt == 8 && num != 0 ){
                printf("%2d remaining character points.  Restarting.\n",num);
                num = Punk->total; cnt = 0;
            }
            else
                ++cnt;
        }
        printf("Are these acceptable?  (y/n) ");
        if ( checkchar("ny") == 'y') doit = FALSE;
    }
}


/************************************************************************
*   Move_stats - Procedure allowing user to place stats where desired.
*
*   move_stats gets the global struct Punk, of type CP.  This is called
*   after roll_stats to allow the user to rearrange their stats as desired.
*   Note that even if total character points method was used, this proc is
*   still called, to allow for maximum flexibility.
************************************************************************/

void move_stats( struct CP *Punk )

{
    int cnt,num0,num1,doit=TRUE;
    num0 = num1 = 0;

    printf("\nMove Stats:\n");
    printf("Do you want to rearrange your stats? (y/n) ");
    if ( checkchar("ny") == 'n') doit = FALSE;

    for (cnt = 0; cnt < 9; cnt++)
        printf("%d %s %3d\n",cnt,statistic[cnt],Punk->stat[cnt]);

    while ( doit ) {
        printf("Enter the two statistics' numbers you wish to switch\n");
        printf("First statistic  => "); num0 = checknum(0,8);
        printf("Second statistic => "); num1 = checknum(0,8);

        cnt = Punk->stat[num0];
        Punk->stat[num0] = Punk->stat[num1];
        Punk->stat[num1] = cnt;

        for (cnt = 0; cnt < 9; cnt++)
            printf("%d %s %3d\n",cnt,statistic[cnt],Punk->stat[cnt]);

        printf("Do you want to switch any others? (y/n) ");
        if ( checkchar("ny") == 'n') doit = FALSE;
    }
}


/************************************************************************
*   Get Role - Procedure to let user choose role, and calculate accordingly.
*
*   get_role gets the global struct Punk of type CP.  This was the LONG one
*   to type in.  All classes are defined here in terms of base wage,
*   skill package, and class name.  The fbc loop (forward, backward, choose)
*   is used here for the first time to display/choose the character role.
************************************************************************/

void get_role( struct CP *Punk )
{
    int num,cnt,tmp,jnk,doit,shflag;
    char junk;

    doit = TRUE;
    num  = 0;

    printf("\nGet Role:\n");

    while ( doit ) {

        for ( cnt = num; cnt < num + 15; cnt++ ) printf("%2d = %s\n",cnt,roles[cnt]);

        printf("Pick the role this character will play.\n");
        printf("Forward, Backward, Choose (f/b/c) ");
        junk = checkchar( "fbc" );

        if ( junk == 'f' ){
            num = num + 15;
            if ( num > NUMROLES - 15 ) num = NUMROLES - 15;
        }

        if ( junk == 'b' ) {
            num = num - 15;
            if ( num < 0 ) num = 0;
        }

        if ( junk == 'c' ) {
            printf("Role number :  ");
            Punk->role = checknum(0,NUMROLES - 1);
            doit = FALSE;
        }
    }

switch (Punk -> role ) {

case 0: Punk->wage = 1000;  /* Rocker */
    Punk->SK[0].name = 110; Punk->SK[1].name = 23;
    Punk->SK[2].name = 18;  Punk->SK[3].name = 1;
    Punk->SK[4].name = 28;  Punk->SK[5].name = 56;
    Punk->SK[6].name = 105; Punk->SK[7].name = 10;
    Punk->SK[8].name = 17;  Punk->SK[9].name = 15;
    break;

case 1: Punk->wage = 2000;  /* Cult Leader */
    Punk->SK[0].name = 110; Punk->SK[1].name = 23;
    Punk->SK[2].name = 32;  Punk->SK[3].name = 12;
    Punk->SK[4].name = 6;   Punk->SK[5].name = 37;
    Punk->SK[6].name = 38;  Punk->SK[7].name = 8;
    Punk->SK[8].name = 18;  Punk->SK[9].name = 17;
    break;

case 2: Punk->wage = 1500;  /* Stripper */
    Punk->SK[0].name = 111; Punk->SK[1].name = 15;
    Punk->SK[2].name = 17;  Punk->SK[3].name = 59;
    Punk->SK[4].name = 10;  Punk->SK[5].name = 0;
    Punk->SK[6].name = 1;   Punk->SK[7].name = 13;
    Punk->SK[8].name = 23;  Punk->SK[9].name = 56;
    break;

case 3: Punk->wage = 1700;  /* Spy */
    Punk->SK[0].name = 112; Punk->SK[1].name = 12;
    Punk->SK[2].name = 9;   Punk->SK[3].name = 65;
    Punk->SK[4].name = 67;  Punk->SK[5].name = 30;
    Punk->SK[6].name = 49;  Punk->SK[7].name = 46;
    Punk->SK[8].name = 94;  Punk->SK[9].name = 5;
    break;

case 4: Punk->wage = 2000;  /* Solo */
    Punk->SK[0].name = 113; Punk->SK[1].name = 23;
    Punk->SK[2].name = 65;  Punk->SK[3].name = 56;
    Punk->SK[4].name = 68;  Punk->SK[5].name = 67;
    Punk->SK[6].name = 78;  Punk->SK[7].name = 55;
    Punk->SK[8].name = 82;  Punk->SK[9].name = 81;
    break;

case 5: Punk->wage = 2000;  /* Corporate Operative */
    Punk->SK[0].name = 113; Punk->SK[1].name = 5;
    Punk->SK[2].name = 12;  Punk->SK[3].name = 32;
    Punk->SK[4].name = 38;  Punk->SK[5].name = 96;
    Punk->SK[6].name = 103; Punk->SK[7].name = 65;
    Punk->SK[8].name = 67;  Punk->SK[9].name = 9;
    break;

case 6: Punk->wage = 2000;  /* Ex-cybersoldier */
    Punk->SK[0].name = 113; Punk->SK[1].name = 61;
    Punk->SK[2].name = 87;  Punk->SK[3].name = 2;
    Punk->SK[4].name = 107; Punk->SK[5].name = 70;
    Punk->SK[6].name = 66;  Punk->SK[7].name = 67;
    Punk->SK[8].name = 78;  Punk->SK[9].name = 91;
    break;

case 7: Punk->wage = 2000;  /* Bodyguard */
    Punk->SK[0].name = 113; Punk->SK[1].name = 65;
    Punk->SK[2].name = 16;  Punk->SK[3].name = 12;
    Punk->SK[4].name = 6;   Punk->SK[5].name = 96;
    Punk->SK[6].name = 1;   Punk->SK[7].name = 61;
    Punk->SK[8].name = 67;  Punk->SK[9].name = 68;
    break;

case 8: Punk->wage = 2000;  /* Bounty Hunter */
    Punk->SK[0].name = 113; Punk->SK[1].name = 32;
    Punk->SK[2].name = 38;  Punk->SK[3].name = 46;
    Punk->SK[4].name = 6;   Punk->SK[5].name = 10;
    Punk->SK[6].name = 52;  Punk->SK[7].name = 61;
    Punk->SK[8].name = 67;  Punk->SK[9].name = 68;
    break;

case 9: Punk->wage = 2000;  /* Street Samurai */
    Punk->SK[0].name = 113; Punk->SK[1].name = 10;
    Punk->SK[2].name = 46;  Punk->SK[3].name = 17;
    Punk->SK[4].name = 6;   Punk->SK[5].name = 1;
    Punk->SK[6].name = 65;  Punk->SK[7].name = 61;
    Punk->SK[8].name = 67;  Punk->SK[9].name = 68;
    break;

case 10:    Punk->wage = 2000;  /* Space Marine */
    Punk->SK[0].name = 114; Punk->SK[1].name = 47;
    Punk->SK[2].name = 62;  Punk->SK[3].name = 83;
    Punk->SK[4].name = 84;  Punk->SK[5].name = 23;
    Punk->SK[6].name = 71;  Punk->SK[7].name = 61;
    Punk->SK[8].name = 65;  Punk->SK[9].name = 78;
    break;

case 11:    Punk->wage = 2000;  /* PA Trooper */
    Punk->SK[0].name = 115; Punk->SK[1].name = 23;
    Punk->SK[2].name = 66;  Punk->SK[3].name = 78;
    Punk->SK[4].name = 68;  Punk->SK[5].name = 67;
    Punk->SK[6].name = 55;  Punk->SK[7].name = 107;
    Punk->SK[8].name = 106; Punk->SK[9].name = 6;
    break;

case 12:    Punk->wage = 1000;  /* Cop */
    Punk->SK[0].name = 116; Punk->SK[1].name = 23;
    Punk->SK[2].name = 65;  Punk->SK[3].name = 12;
    Punk->SK[4].name = 55;  Punk->SK[5].name = 30;
    Punk->SK[6].name = 56;  Punk->SK[7].name = 68;
    Punk->SK[8].name = 5;   Punk->SK[9].name = 10;
    break;

case 13:    Punk->wage = 1500;  /* Private Investigator */
    Punk->SK[0].name = 117; Punk->SK[1].name = 23;
    Punk->SK[2].name = 10;  Punk->SK[3].name = 65;
    Punk->SK[4].name = 35;  Punk->SK[5].name = 38;
    Punk->SK[6].name = 46;  Punk->SK[7].name = 12;
    Punk->SK[8].name = 56;  Punk->SK[9].name = 17;
    break;

case 14:    Punk->wage = 1500;  /* Corporate */
    Punk->SK[0].name = 118; Punk->SK[1].name = 23;
    Punk->SK[2].name = 12;  Punk->SK[3].name = 30;
    Punk->SK[4].name = 38;  Punk->SK[5].name = 16;
    Punk->SK[6].name = 17;  Punk->SK[7].name = 48;
    Punk->SK[8].name = 1;   Punk->SK[9].name = 0;
    break;

case 15:    Punk->wage = 1700;  /* Government Official */
    Punk->SK[0].name = 118; Punk->SK[1].name = 23;
    Punk->SK[2].name = 12;  Punk->SK[3].name = 30;
    Punk->SK[4].name = 38;  Punk->SK[5].name = 16;
    Punk->SK[6].name = 17;  Punk->SK[7].name = 6;
    Punk->SK[8].name = 32;  Punk->SK[9].name = 19;
    break;

case 16:    Punk->wage = 2300;  /* Lawyer */
    Punk->SK[0].name = 119; Punk->SK[1].name = 23;
    Punk->SK[2].name = 30;  Punk->SK[3].name = 32;
    Punk->SK[4].name = 12;  Punk->SK[5].name = 5;
    Punk->SK[6].name = 38;  Punk->SK[7].name = 8;
    Punk->SK[8].name = 1;   Punk->SK[9].name = 17;
    break;

case 17:    Punk->wage = 1000;  /* Media */
    Punk->SK[0].name = 119; Punk->SK[1].name = 23;
    Punk->SK[2].name = 28;  Punk->SK[3].name = 30;
    Punk->SK[4].name = 17;  Punk->SK[5].name = 12;
    Punk->SK[6].name = 16;  Punk->SK[7].name = 10;
    Punk->SK[8].name = 101; Punk->SK[9].name = 13;
    break;

case 18:    Punk->wage = 1700;  /* Journalist */
    Punk->SK[0].name = 119; Punk->SK[1].name = 23;
    Punk->SK[2].name = 28;  Punk->SK[3].name = 30;
    Punk->SK[4].name = 17;  Punk->SK[5].name = 12;
    Punk->SK[6].name = 16;  Punk->SK[7].name = 101;
    Punk->SK[8].name = 10;  Punk->SK[9].name = 13;
    break;

case 19:    Punk->wage = 1700;  /* Investigative Reporter */
    Punk->SK[0].name = 119; Punk->SK[1].name = 12;
    Punk->SK[2].name = 10;  Punk->SK[3].name = 101;
    Punk->SK[4].name = 65;  Punk->SK[5].name = 30;
    Punk->SK[6].name = 17;  Punk->SK[7].name = 8;
    Punk->SK[8].name = 23;  Punk->SK[9].name = 46;
    break;

case 20:    Punk->wage = 1700;  /* War Correspondent */
    Punk->SK[0].name = 119; Punk->SK[1].name = 65;
    Punk->SK[2].name = 37;  Punk->SK[3].name = 101;
    Punk->SK[4].name = 2;   Punk->SK[5].name = 30;
    Punk->SK[6].name = 17;  Punk->SK[7].name = 13;
    Punk->SK[8].name = 23;  Punk->SK[9].name = 9;
    break;

case 21:    Punk->wage = 1700;  /* Sensationalist */
    Punk->SK[0].name = 119; Punk->SK[1].name = 23;
    Punk->SK[2].name = 40;  Punk->SK[3].name = 72;
    Punk->SK[4].name = 17;  Punk->SK[5].name = 46;
    Punk->SK[6].name = 61;  Punk->SK[7].name = 101;
    Punk->SK[8].name = 10;  Punk->SK[9].name = 13;
    break;

case 22:    Punk->wage = 1400;  /* Tabloid Reporter */
    Punk->SK[0].name = 120; Punk->SK[1].name = 23;
    Punk->SK[2].name = 1;   Punk->SK[3].name = 10;
    Punk->SK[4].name = 17;  Punk->SK[5].name = 12;
    Punk->SK[6].name = 13;  Punk->SK[7].name = 101;
    Punk->SK[8].name = 46;  Punk->SK[9].name = 32;
    break;

case 23:    Punk->wage = 1500;  /* Fixer */
    Punk->SK[0].name = 121; Punk->SK[1].name = 23;
    Punk->SK[2].name = 98;  Punk->SK[3].name = 65;
    Punk->SK[4].name = 56;  Punk->SK[5].name = 68;
    Punk->SK[6].name = 103; Punk->SK[7].name = 104;
    Punk->SK[8].name = 5;   Punk->SK[9].name = 10;
    break;

case 24:    Punk->wage = 1500;  /* Scavanger */
    Punk->SK[0].name = 121; Punk->SK[1].name = 23;
    Punk->SK[2].name = 87;  Punk->SK[3].name = 97;
    Punk->SK[4].name = 12;  Punk->SK[5].name = 65;
    Punk->SK[6].name = 81;  Punk->SK[7].name = 3;
    Punk->SK[8].name = 2;   Punk->SK[9].name = 32;
    break;

case 25:    Punk->wage = 1500;  /* Pimp/Pusher */
    Punk->SK[0].name = 121; Punk->SK[1].name = 23;
    Punk->SK[2].name = 6;   Punk->SK[3].name = 17;
    Punk->SK[4].name = 1;   Punk->SK[5].name = 65;
    Punk->SK[6].name = 82;  Punk->SK[7].name = 15;
    Punk->SK[8].name = 61;  Punk->SK[9].name = 27;
    break;

case 26:    Punk->wage = 1500;  /* Broker */
    Punk->SK[0].name = 121; Punk->SK[1].name = 23;
    Punk->SK[2].name = 19;  Punk->SK[3].name = 12;
    Punk->SK[4].name = 30;  Punk->SK[5].name = 48;
    Punk->SK[6].name = 38;  Punk->SK[7].name = 60;
    Punk->SK[8].name = 32;  Punk->SK[9].name = 65;
    break;

case 27:    Punk->wage = 1500;  /* Racketeer */
    Punk->SK[0].name = 121; Punk->SK[1].name = 23;
    Punk->SK[2].name = 56;  Punk->SK[3].name = 37;
    Punk->SK[4].name = 3;   Punk->SK[5].name = 5;
    Punk->SK[6].name = 82;  Punk->SK[7].name = 94;
    Punk->SK[8].name = 96;  Punk->SK[9].name = 6;
    break;

case 28:    Punk->wage = 1500;  /* Black Marketeer */
    Punk->SK[0].name = 121; Punk->SK[1].name = 19;
    Punk->SK[2].name = 37;  Punk->SK[3].name = 32;
    Punk->SK[4].name = 17;  Punk->SK[5].name = 16;
    Punk->SK[6].name = 61;  Punk->SK[7].name = 82;
    Punk->SK[8].name = 9;   Punk->SK[9].name = 98;
    break;

case 29:    Punk->wage = 1000;  /* Techie */
    Punk->SK[0].name = 122; Punk->SK[1].name = 23;
    Punk->SK[2].name = 87;  Punk->SK[3].name = 91;
    Punk->SK[4].name = 50;  Punk->SK[5].name = 30;
    Punk->SK[6].name = 95;

/************************************************************************
*   Choose the remaining 3 technical skills for the techie.
*   NOTE: The TECH skills are from 85 to 108.  Change these values for
*   Skill list changes in cyrbdata.h
************************************************************************/

    tmp = 7; num = 85; shflag = TRUE;

    while ( tmp < 10 ) {

        if ( shflag )
            for ( cnt = num; cnt < num + 15; cnt++ )
                printf("%d %s\n",cnt,short_skills[cnt]);
        else 
            for ( cnt = num; cnt < num + 15; cnt++ )
                printf("%d %s\n",cnt,long_skills[cnt]);

        printf("Choose one to complete your skills.  You have %d skill(s) left.\n",10 - tmp);
        printf("Forward, Backward, Choose, Long/Short names, Display (f/b/c/l/s/d) ");
        junk = checkchar( "fbclsd" );

        if ( junk == 'f' ){
            num = num + 15;
            if ( num > 93 ) num = 93;
        }

        if ( junk == 'b' ) {
            num = num - 15;
            if ( num < 85 ) num = 85;
        }

        if ( junk == 'c' ) {
            printf("Skill number :  "); 
            jnk = checknum(85,(NUMREGSKILLS-2));
            doit = TRUE;

            for ( cnt = 0; cnt < 10; cnt++ )
                if ( Punk->SK[cnt].name == jnk ) doit = FALSE;

            if ( doit ) {
                Punk->SK[tmp].name = jnk;
                tmp += 1;
            }
            else {
                printf("Invalid skill choice.  Choose another.\n");
            }
        }
        else if ( junk == 'l' ) shflag = FALSE;
        else if ( junk == 's' ) shflag = TRUE;
        else if ( junk == 'd' ) {
            for ( cnt = 0; cnt < Punk->skills; cnt++ )
                if ( shflag )   printf("%d %s\n",cnt,short_skills[Punk->SK[cnt].name]);
                else                printf("%d %s\n",cnt,long_skills[Punk->SK[cnt].name]);
            printf("/nEnter y to continue :");
            checkchar("y");
        }

    }
    break;

case 30:    Punk->wage = 1200;  /* Salver */
    Punk->SK[0].name = 123; Punk->SK[1].name = 47;
    Punk->SK[2].name = 62;  Punk->SK[3].name = 83;
    Punk->SK[4].name = 22;  Punk->SK[5].name = 70;
    Punk->SK[6].name = 71;  Punk->SK[7].name = 87;
    Punk->SK[8].name = 23;  Punk->SK[9].name = 68;
    break;

case 31:    Punk->wage = 1600;  /* Medic */
    Punk->SK[0].name = 124; Punk->SK[1].name = 23;
    Punk->SK[2].name = 87;  Punk->SK[3].name = 29;
    Punk->SK[4].name = 30;  Punk->SK[5].name = 89;
    Punk->SK[6].name = 38;  Punk->SK[7].name = 102;
    Punk->SK[8].name = 53;  Punk->SK[9].name = 12;
    break;

case 32:    Punk->wage = 1800;  /* Jockey */
    Punk->SK[0].name = 125; Punk->SK[1].name = 23;
    Punk->SK[2].name = 56;  Punk->SK[3].name = 69;
    Punk->SK[4].name = 41;  Punk->SK[5].name = 66;
    Punk->SK[6].name = 104;

/************************************************************************
*   Choose the remaining 3 piloting skills for the jockey.
*   The Pilot skills are from 71 to 77 in the skill list.
************************************************************************/

    for ( tmp = 7; tmp < 10; tmp++ ) {
        for ( cnt = 71; cnt < 78; cnt++ ) printf("%d %s\n",cnt,short_skills[cnt]);
        printf("Choose one to complete your skills.  You have %d skill(s) left.\n",10 - tmp);

        jnk = checknum(71,77);

        doit = TRUE;
        for ( cnt = 0; cnt < 10; cnt++ )
            if ( Punk->SK[cnt].name == jnk ) doit = FALSE;

        if ( doit ) Punk->SK[tmp].name = jnk;
        else {
            printf("You already have that skill.  Choose another.\n");
            tmp--;
        }
    }
    break;

case 33:    Punk->wage = 1500;  /* Highrider */
    Punk->SK[0].name = 125; Punk->SK[1].name = 47;
    Punk->SK[2].name = 62;  Punk->SK[3].name = 22;
    Punk->SK[4].name = 71;  Punk->SK[5].name = 77;
    Punk->SK[6].name = 87;  Punk->SK[7].name = 43;
    Punk->SK[8].name = 39;  Punk->SK[9].name = 42;
    break;

case 34:    Punk->wage = 1000;  /* Workganger */
    Punk->SK[0].name = 126; Punk->SK[1].name = 47;
    Punk->SK[2].name = 62;  Punk->SK[3].name = 83;
    Punk->SK[4].name = 70;  Punk->SK[5].name = 71;
    Punk->SK[6].name = 87;  Punk->SK[7].name = 95;
    Punk->SK[8].name = 32;  Punk->SK[9].name = 3;
    break;

case 35:    Punk->wage = 1000;  /* Netrunner */
    Punk->SK[0].name = 127; Punk->SK[1].name = 23;
    Punk->SK[2].name = 87;  Punk->SK[3].name = 30;
    Punk->SK[4].name = 49;  Punk->SK[5].name = 91;
    Punk->SK[6].name = 90;  Punk->SK[7].name = 28;
    Punk->SK[8].name = 95;  Punk->SK[9].name = 43;
    break;

case 36:    Punk->wage = 1000;  /* Rogue Hunter */
    Punk->SK[0].name = 127; Punk->SK[1].name = 23;
    Punk->SK[2].name = 87;  Punk->SK[3].name = 30;
    Punk->SK[4].name = 49;  Punk->SK[5].name = 91;
    Punk->SK[6].name = 43;  Punk->SK[7].name = 95;
    Punk->SK[8].name = 90;  Punk->SK[9].name = 32;
    break;

case 37:    Punk->wage = 1000;  /* Nomad */
    Punk->SK[0].name = 128; Punk->SK[1].name = 23;
    Punk->SK[2].name = 2;   Punk->SK[3].name = 68;
    Punk->SK[4].name = 78;  Punk->SK[5].name = 61;
    Punk->SK[6].name = 87;  Punk->SK[7].name = 52;
    Punk->SK[8].name = 56;  Punk->SK[9].name = 55;
    break;

case 38:    Punk->wage = 1000;  /* Pirate */
    Punk->SK[0].name = 128; Punk->SK[1].name = 23;
    Punk->SK[2].name = 4;   Punk->SK[3].name = 61;
    Punk->SK[4].name = 45;  Punk->SK[5].name = 56;
    Punk->SK[6].name = 68;  Punk->SK[7].name = 41;
    Punk->SK[8].name = 2;   Punk->SK[9].name = 78;
    break;

case 39:    Punk->wage = 1200;  /* Poser Streetpunk */
    Punk->SK[0].name = 129; Punk->SK[1].name = 23;
    Punk->SK[2].name = 10;  Punk->SK[3].name = 56;
    Punk->SK[4].name = 68;  Punk->SK[5].name = 1;
    Punk->SK[6].name = 17;  Punk->SK[7].name = 0;
    Punk->SK[8].name = 32;  Punk->SK[9].name = 94;
    break;

case 40:    Punk->wage = 1000;  /* Chromer Streetpunk */
    Punk->SK[0].name = 129; Punk->SK[1].name = 23;
    Punk->SK[2].name = 10;  Punk->SK[3].name = 56;
    Punk->SK[4].name = 68;  Punk->SK[5].name = 1;
    Punk->SK[6].name = 17;  Punk->SK[7].name = 105;
    Punk->SK[8].name = 59;  Punk->SK[9].name = 32;
    break;

case 41:    Punk->wage = 1100;  /* Booster Streetpunk */
    Punk->SK[0].name = 129; Punk->SK[1].name = 23;
    Punk->SK[2].name = 10;  Punk->SK[3].name = 56;
    Punk->SK[4].name = 68;  Punk->SK[5].name = 55;
    Punk->SK[6].name = 2;   Punk->SK[7].name = 6;
    Punk->SK[8].name = 91;  Punk->SK[9].name = 32;
    break;

case 42:    Punk->wage = 1000;  /* 'Dorpher Streetpunk */
    Punk->SK[0].name = 129; Punk->SK[1].name = 23;
    Punk->SK[2].name = 10;  Punk->SK[3].name = 56;
    Punk->SK[4].name = 68;  Punk->SK[5].name = 103;
    Punk->SK[6].name = 104; Punk->SK[7].name = 17;
    Punk->SK[8].name = 27;  Punk->SK[9].name = 32;
    break;

case 43:    Punk->wage = 1000;  /* Zoomer Streetpunk */
    Punk->SK[0].name = 129; Punk->SK[1].name = 23;
    Punk->SK[2].name = 10;  Punk->SK[3].name = 56;
    Punk->SK[4].name = 68;  Punk->SK[5].name = 87;
    Punk->SK[6].name = 55;  Punk->SK[7].name = 6;
    Punk->SK[8].name = 69;  Punk->SK[9].name = 32;
    break;

case 44:    Punk->wage = 900;   /* Inquisitor Predator */
    Punk->SK[0].name = 129; Punk->SK[1].name = 23;
    Punk->SK[2].name = 67;  Punk->SK[3].name = 55;
    Punk->SK[4].name = 60;  Punk->SK[5].name = 68;
    Punk->SK[6].name = 81;  Punk->SK[7].name = 51;
    Punk->SK[8].name = 3;   Punk->SK[9].name = 2;
    break;

}
    Punk->skills = 10;
}


/************************************************************************
*   Distribute Points - Procedure to distribute points to skills.
*   
*   dist_points gets the global struct Punk of type CP.  This allows the
*   user to spend their initial 40 skill points on the basic skill package
*   for their role.  Long/Short skill names are supported, and the fbc loop
*   was modified to allow this.  Legal point distribution is forced.
************************************************************************/

void dist_points( struct CP *Punk )

{
    int cnt,num,doit,shflag;
    doit = TRUE;
    shflag = TRUE;

    printf("\nDistribute Skill Points:\n");
    printf("Do you want short or long skill names? (s/l) ");
    if (checkchar("sl") == 'l' ) shflag = FALSE;

    for ( cnt = 0; cnt < 10; cnt++ )
        if ( shflag )   printf("%s\n",short_skills[Punk->SK[cnt].name]);
        else                printf("%s\n",long_skills[Punk->SK[cnt].name]);

    printf("\n");

    while ( doit ) {
        num = 40; cnt = 0;
        while ( cnt < 10 ){

            if ( shflag ) {
                printf("%2d skill points, %2d skills left->  %s = ",num,10 - cnt,short_skills[Punk->SK[cnt].name]);
                Punk->SK[cnt].level = checknum(1,10);
            }
            else {
                printf("%s\n",long_skills[Punk->SK[cnt].name]);
                printf("%2d skill points, %2d skills left->  %s = ",num,10 - cnt,short_skills[Punk->SK[cnt].name]);
                Punk->SK[cnt].level = checknum(1,10);
            }
            num = num - Punk->SK[cnt].level;

            if ( num < (8 - cnt) ){
                printf("Insufficient points for remaining skills.  Restarting.\n");
                num = 40; cnt = 0;
            }
            else if ( cnt == 9 && num != 0 ) {
                printf("%2d remaining skill points.  Restarting.\n",num);
                num = 40; cnt = 0;
            }
            else
                ++cnt;
        }
        printf("Are these acceptable?  (y/n) ");
        if ( checkchar("ny") == 'y') doit = FALSE;
    }
}


/************************************************************************
*   Calculate Punk - Procedure to calc the rest of the character
*
*   calc_punk gets the global struct Punk, of type CP.  This procedure is
*   primarily the hard-coding of the charts/starting calculated stats.
************************************************************************/

void calc_punk( struct CP *Punk )

{
    int cnt;

    printf("\nCalculating:\n");
    Punk->vamp = FALSE;                                         /* Nights Edge */
    Punk->VY = 0;
    Punk->feed = 0;
    Punk->heal = 0;
    Punk->wolf = 0;
    Punk->psy = FALSE;

    Punk->puskills = Punk->stat[0] + Punk->stat[1];     /* Pickup Skills */

    if ( Punk->SK[0].level > 5 )                                /* Starting eb */
        Punk->wage = Punk->wage * ( Punk->SK[0].level - 5 );
    Punk->eb = Punk->wage * die_roll(1,3);

    Punk->stat[9]  = 0;                                         /* Psyche      */
    Punk->stat[10] = 1;                                         /* Reputation  */
    Punk->stat[11] = Punk->stat[6] * 3;                     /* Run         */
    Punk->stat[12] = Punk->stat[11] / 4;                    /* Leap        */
    Punk->stat[13] = Punk->stat[7] * 10;                    /* Humanity    */

    if ( Punk->stat[8] == 2 )                                   /* BT Modifier */
        Punk->stat[14] = 0;
    if ( Punk->stat[8] == 3 || Punk->stat[8] == 4 )
        Punk->stat[14] = -1;
    if ( Punk->stat[8] == 5 || Punk->stat[8] == 6 || Punk->stat[8] == 7 )
        Punk->stat[14] = -2;
    if ( Punk->stat[8] == 8 || Punk->stat[8] == 9 )
        Punk->stat[14] = -3;
    if ( Punk->stat[8] == 10 )
        Punk->stat[14] = -4;


    Punk->stat[15] = Punk->stat[8] * 5;                     /* Lift        */
    Punk->stat[16] = Punk->stat[8];                         /* Save         */

    Punk->family[0] = fam_rank[die_roll(1,10)-1];       /* Family BG   */

    if ( die_roll(1,10) > 6 ) Punk->family[1] = "Both parents are living";
    else Punk->family[1] = parents_bad[die_roll(1,10)-1];

    if ( die_roll(1,10) > 6 ) Punk->family[2] = "Family status is okay even if parents are missing or dead";
    else Punk->family[2] = fam_tragedy[die_roll(1,10)-1];

    Punk->family[3] = child_envir[die_roll(1,10)-1];

    Punk->sib = die_roll(1,10);                             /* # Siblings  */
    if ( Punk->sib > 7 )    Punk->sib = 0;

    for ( cnt = 0; cnt < Punk->sib; cnt++ ) {
        Punk->SI[cnt].trait[0] = die_roll(1,4)-1;
        Punk->SI[cnt].trait[1] = die_roll(1,10)-1;
        Punk->SI[cnt].trait[2] = die_roll(1,5)-1;
    }
}


/************************************************************************
*   Vampire Punk - Procedure to allow the character to become vampiric.
*   
*   vamp_punk gets the global struct Punk, of type CP.  This follows Night's
*   Edge exactly, and modifies the Punk according to the rules there.  Note
*   that Vampires are automatically PSY capable, and thus Punk->psy is TRUE.
************************************************************************/

void vampire_punk ( struct CP *Punk )

{
    int roll;

    printf("\nIs this character going to be a vampire? (y/n)");

    if ( checkchar( "ny" ) == 'y' ) {
        Punk->vamp = TRUE;
        Punk->psy = TRUE;
        roll = die_roll(1,10);
        if ( roll < 6 ) {
            Punk->VY = die_roll(1,2) - 1;
            Punk->puskills = Punk->puskills + (Punk->VY * 6);
            Punk->stat[1] += 1; Punk->stat[2] += -1;
            Punk->stat[6] += 1; Punk->stat[7] += -1;
            Punk->stat[9] += 1; Punk->feed = 3; Punk->heal = 4;
        }
        if ( roll > 5 && roll < 10 ) {
            Punk->VY = 1 + die_roll(1,19);
            Punk->puskills = Punk->puskills + ((Punk->VY-2) * 3) + 6;
            Punk->stat[1] += 2; Punk->stat[2] += -1;
            Punk->stat[6] += 1; Punk->stat[7] += -2;
            Punk->stat[8] += 1; Punk->stat[9] += 2;
            Punk->feed = 7;     Punk->heal = 5;
        }
        if ( roll == 10 ) {
            Punk->VY = 20 + die_roll(1,80);
            Punk->puskills = Punk->puskills + ((Punk->VY-21) * 2) + 60;
            Punk->stat[1] += 2; Punk->stat[2] += -2;
            Punk->stat[5] += 1; Punk->stat[6] += 2;
            Punk->stat[7] += -2;    Punk->stat[8] += 2;
            Punk->stat[9] += 3; Punk->feed = 14;    Punk->heal = 6;
        }
    }
}


/************************************************************************
*   Wolven Punk - Procedure allows Punk to be wolven and calculate effects.
*
*   wolf_punk gets the global struct Punk, of type CP.  This follows Night's
*   Edge exactly.  Note that were-wolves are automatically PSY capable, and
*   thus Punk->psy is set TRUE.
************************************************************************/

void wolven_punk    ( struct CP *Punk )

{
    if ( Punk->vamp == FALSE) {
        printf("\nIs this character going to be a were-wolf? (y/n)");
        if ( checkchar( "ny" ) == 'y' ) {

            printf("Is this character a lycanthrope, or full were-wolf? (l/w)");
            if ( checkchar( "wl" ) == 'l' ) {
                Punk->wolf = 1;
                Punk->psy = TRUE;
                Punk->stat[7] += -1;
                Punk->stat[2] += 1;
                Punk->stat[9] += 1;
                Punk->QU[0] = "Silver weapons do not cause addition damage.";
                Punk->QU[1] = "During full moon:  REF +2, BOD +2, STUN -4, for 10 turns.";
                Punk->QU[2] = "During full moon:  -1 pt EMP/day until you have killed.";
                Punk->quirks = 3;
            }
            else {
                Punk->wolf = 2;
                Punk->psy = TRUE;
                Punk->VY  = die_roll(1,6) - 1;
                if ( Punk->VY < 2 ) {
                    Punk->puskills = Punk->puskills + (Punk->VY * 6);
                    Punk->stat[8] += 1; Punk->stat[7] += -1;
                    Punk->stat[9] += 1; Punk->heal = 3;
                }
                else {
                    Punk->puskills = Punk->puskills + ((Punk->VY-2) * 4) + 6;
                    Punk->stat[7] += -2;    Punk->stat[8] += 2;
                    Punk->stat[9] += 1; Punk->stat[1] += 1;
                    Punk->heal = 4;
                }
                Punk->SK[Punk->skills].name = 157;
                Punk->SK[Punk->skills].level = 3;
                Punk->SK[Punk->skills].psy = TRUE;
                Punk->skills += 1;
                Punk->puskills += -9;
            }
        }
    }
}


/************************************************************************
*   Psyber Punk - Procedure allows the char to be psychic.  Very Scary.
*   
*   psyber_punk gets the global struct Punk, of type CP.  This follows Night's
*   Edge exactly, and allows the user to either roll or enter the PSY stat.
*   The PSY domains that the char is allowed to use are generated here, for
*   later use in psy_skill().
************************************************************************/

void psyber_punk ( struct CP *Punk )

{
    int roll,doit,tmp;

    if ( Punk->psy == FALSE ) {
        printf("\nIs this character going to have Psychic abilities? (y/n)");
        if ( checkchar( "ny" )  == 'y' ) Punk->psy = TRUE;
        else Punk->psy = FALSE;
    }

    if ( Punk->psy) {
        doit = TRUE;
        tmp = Punk->stat[9];

        while ( doit ) {
            printf("NOTE: Vampire/Were-Wolf PSY bonuses have been calcuated, if applicable.\n");
            printf("Enter or Roll PSY stat?  (e/r) ");
            if (checkchar("re") == 'r' )    Punk->stat[9] = die_roll(2,5);
            else {
                printf("Enter PSY (2 - 10) :");
                Punk->stat[9] += checknum(2,10);
            }
            printf("PSY = %d.  Is this acceptable?  (y/n) ",Punk->stat[9]);
            if ( checkchar("ny") == 'y') doit = FALSE;
        }

        Punk->stat[9] += tmp;

        if ( Punk->stat[9] > 4 ) {
            roll = die_roll(1,10);
            if (roll < 4)                       Punk->psycat = "a";
            if (roll == 4)                      Punk->psycat = "b";
            if (roll > 4 && roll < 9)       Punk->psycat = "c";
            if (roll > 9)                       Punk->psycat = "d";
        }
        else
            Punk->psycat = "x";

        if ( Punk->wolf == 2) Punk->psycat = "d";

        if ( Punk->stat[9] > 7 ) {
            doit = TRUE;
            while (doit) {
                roll = die_roll(1,10);
                if ((roll < 4) && (strrchr(Punk->psycat,'a') == NULL)) {
                    strcat(Punk->psycat,"a");
                    doit = FALSE;
                }
                if ((roll == 4) && (strrchr(Punk->psycat,'b') == NULL)) {
                    strcat(Punk->psycat,"b");
                    doit = FALSE;
                }
                if ((roll > 4 && roll < 9) && (strrchr(Punk->psycat,'c') == NULL)) {
                    strcat(Punk->psycat,"c");
                    doit = FALSE;
                }
                if ((roll > 8) && (strrchr(Punk->psycat,'d') == NULL)) {
                     strcat(Punk->psycat,"d");
                    doit = FALSE;
                }
            }
        }

        if ( Punk->stat[9] > 8 ) {
            doit = TRUE;
            while (doit) {
                roll = die_roll(1,10);
                if ((roll < 4) && (strrchr(Punk->psycat,'a') == NULL)) {
                    strcat(Punk->psycat,"a");
                    doit = FALSE;
                }
                if ((roll == 4) && (strrchr(Punk->psycat,'b') == NULL)) {
                    strcat(Punk->psycat,"b");
                    doit = FALSE;
                }
                if ((roll > 4 && roll < 9) && (strrchr(Punk->psycat,'c') == NULL)) {
                    strcat(Punk->psycat,"c");
                    doit = FALSE;
                }
                if ((roll > 8) && (strrchr(Punk->psycat,'d') == NULL)) {
                    strcat(Punk->psycat,"d");
                    doit = FALSE;
                }
            }
        }

        if ( Punk->stat[9] > 9 ) Punk->psycat = "abcd";
    }
}


/************************************************************************
*   Psy Skill - Procedure gets a Psyche based skill for the character.
*
*   psy_skill gets the global struct Punk, of type CP, and is called from
*   pickup_skills(), if the user chooses skill #110 from that procedure.
*   Three unique possible PSY skills are generated, and the user allowed
*   to choose.  The proc makes sure that the skills offered for choice are
*   unique, and that the char doesn't have them yet.  The proc also gets
*   the number of pickup skill points spent on the PSY skill, legally.
************************************************************************/

void psy_skill ( struct CP *Punk )

{
    int catagory,cnt,roll,tmp,psysk[4],shflag;

    cnt = 0;
    shflag = TRUE;

    printf("\nPSY Skills:\n");
    printf("Do you want short or long skill names? (s/l) ");
    if (checkchar("sl") == 'l' ) shflag = FALSE;

    printf("\n");

    if ( Punk->psy == TRUE && Punk->stat[9] > 4 ) {

        while ( cnt < 3 ){

        catagory = die_roll(1,10);

            if ((catagory < 5) && (strrchr(Punk->psycat,'a') != NULL)) {
                roll = die_roll(1,12);
                if ( roll == 1 ) psysk[cnt] = 130;
                if ( roll == 2 ) psysk[cnt] = 131;
                if ( roll == 3 ) psysk[cnt] = 132;
                if ( roll == 4 ) psysk[cnt] = 133;
                if ( roll == 5 ) psysk[cnt] = 134;
                if ( roll == 6 ) psysk[cnt] = 135;
                if ( roll == 7 ) psysk[cnt] = 136;
                if ( roll == 8 ) psysk[cnt] = 137;
                if ( roll == 9 ) psysk[cnt] = 138;
                if ( roll == 10 ) psysk[cnt] = 139;
                if ( roll == 11 ) psysk[cnt] = 140;
                if ( roll == 12 ) psysk[cnt] = 141;
            }

            if ( (catagory == 5) && (strrchr(Punk->psycat,'b') != NULL) ) {
                roll = die_roll(1,3);
                if ( roll == 1 ) psysk[cnt] = 142;
                if ( roll == 2 ) psysk[cnt] = 143;
                if ( roll == 3 ) psysk[cnt] = 144;
            }

            if ( (catagory > 5 && catagory < 10 ) && (strrchr(Punk->psycat,'c') != NULL) ) {
                roll = die_roll(1,9);
                if ( roll == 1 ) psysk[cnt] = 145;
                if ( roll == 2 ) psysk[cnt] = 146;
                if ( roll == 3 ) psysk[cnt] = 147;
                if ( roll == 4 ) psysk[cnt] = 148;
                if ( roll == 5 ) psysk[cnt] = 149;
                if ( roll == 6 ) psysk[cnt] = 150;
                if ( roll == 7 ) psysk[cnt] = 151;
                if ( roll == 8 ) psysk[cnt] = 152;
                if ( roll == 9 ) psysk[cnt] = 153;
            }

            if ( (catagory > 9) && (strrchr(Punk->psycat,'d') != NULL) ) {
                roll = die_roll(1,5);
                if ( roll == 1 ) psysk[cnt] = 154;
                if ( roll == 2 ) psysk[cnt] = 155;
                if ( roll == 3 ) psysk[cnt] = 156;
                if ( roll == 4 ) psysk[cnt] = 157;
                if ( roll == 5 ) psysk[cnt] = 158;
            }

            if ( psysk[cnt] > 129 && psysk[cnt] < 159 ) {
                if ( cnt == 2 && psysk[0] != psysk[2] && psysk[1] != psysk[2]) cnt = 3;
                if ( cnt == 1 && psysk[0] != psysk[1] ) cnt = 2;
                if ( cnt == 0 ) cnt = 1;
            }

        }

        for (cnt = 0; cnt < 3; cnt++) {
            if ( shflag )   printf("%d %s\n",cnt,short_skills[psysk[cnt]]);
            else                printf("%d %s\n",cnt,long_skills[psysk[cnt]]);

            for ( tmp = 9; tmp < Punk->skills; tmp++ ) 
                if (Punk->SK[tmp].name == psysk[cnt])
                    printf("NOTE: You already have %s\n",short_skills[psysk[cnt]]);
        }

        printf("3 (QUIT) Abort this PSY skill choice and return to pickup skills\n");
        printf("Please choose one of the above (0,1,2,3) :");
        roll = checknum(0,3);
        if ( roll != 3 ) {
            Punk->SK[Punk->skills].name = psysk[roll];
            Punk->SK[Punk->skills].level = 0;
            Punk->SK[Punk->skills].psy = TRUE;
            tmp = TRUE;
        }
        else
            tmp = FALSE;

        while ( tmp ) {
            printf("Enter # points to spend on %s\n",short_skills[Punk->SK[Punk->skills].name]);
            printf("NOTE: The cost per skill level is (skill level) ^ 2\n");
            printf("Available skills points = %d.  Amount to spend : ",Punk->puskills);
            roll = checknum(1,Punk->puskills);

            if ( roll == 1 ) Punk->SK[Punk->skills].level = 1;
            if ( roll == 4 ) Punk->SK[Punk->skills].level = 2;
            if ( roll == 9 ) Punk->SK[Punk->skills].level = 3;
            if ( roll == 16 ) Punk->SK[Punk->skills].level = 4;
            if ( roll == 25 ) Punk->SK[Punk->skills].level = 5;
            if ( roll == 36 ) Punk->SK[Punk->skills].level = 6;
            if ( roll == 49 ) Punk->SK[Punk->skills].level = 7;
            if ( roll == 64 ) Punk->SK[Punk->skills].level = 8;
            if ( roll == 81 ) Punk->SK[Punk->skills].level = 9;
            if ( roll == 100 ) Punk->SK[Punk->skills].level = 10;


            if ( Punk->SK[Punk->skills].level > 0 ) {
                Punk->puskills = Punk->puskills - roll;
                Punk->skills += 1;
                tmp = FALSE;
            }
            else {
                tmp = TRUE;
                printf("Illegal value!  Must be = (skill level) ^ 2\n");
                Punk->SK[Punk->skills].level = 0;
            }
        }
    }
    else {
        printf("You have no psychic abilities, or your PSY stat is low.\n\n");
    }
}


/************************************************************************
*   Pickup Skills - Procedure gets pickup skills for character.
*
*   pickup_skills get the global struct Punk, of type CP.  This proc forces
*   the user to choose unique skills they do not have yet, and purchase them
*   with the pickup skill points generated by INT+REF.  Note that skill #110
*   will allow, if the character has PSY, to choose a PSY based skill legally.
*   Also note that the fbc loop with long/short skill name option is here.
************************************************************************/
#define shortname(n) sortflag ? short_index(n) : short_skills[n]
#define longname(n)  sortflag ? long_index(n)  : long_skills[n]

void pickup_skills ( struct CP *Punk )

{
    int cnt,num,jnk,doit,shflag,sortflag;
    char junk;

    num = 0;
    shflag = TRUE;

    printf("\nPickup Skills:\n");

    while ( Punk->puskills > 0 ) {

        for ( cnt = num; cnt < num + 15; cnt++ )
            if ( shflag )   printf("%d %s\n",cnt,shortname(cnt));
            else                printf("%d %s\n",cnt,longname(cnt));

        printf("Choose a pickup skill.  You have %d point(s) left.\n",Punk->puskills);
        printf("Choose, Forward/Backward, Long/Short names, Alpha/Natural order, Display:\n (c/f/b/l/s/a/c/d) :");
        junk = checkchar( "fbclsdan" );

        if ( junk == 'f' ){
            num = num + 15;
            if ( num > 96 ) num = 96;
        }

        else if ( junk == 'b' ) {
            num = num - 15;
            if ( num < 0 ) num = 0;
        }

        else if ( junk == 'c' ) {
            printf("Skill number :  ");
            jnk = checknum(0,NUMREGSKILLS-1);
            if (sortflag) jnk = skill_index[jnk];
            doit = TRUE;

            for ( cnt = 0; cnt < Punk->skills; cnt++ )
                if ( Punk->SK[cnt].name == jnk ) doit = FALSE;

            if ( doit ) {
                if ( jnk == NUMREGSKILLS-1 ) {
                    psy_skill(Punk);
                }
                else {
                    Punk->SK[Punk->skills].name = jnk;
                    printf("%d skill points.  How many for %s? ", Punk->puskills, short_skills[Punk->SK[Punk->skills].name]);

                    jnk = checknum(1, ( 10 <= Punk->puskills ? 10 : Punk->puskills ));

                    Punk->SK[Punk->skills].level = jnk;
                    Punk->SK[Punk->skills].psy = FALSE;
                    Punk->skills += 1;
                    Punk->puskills = Punk->puskills - jnk;
                }
            }
            else
                printf("You already have that skill.\n");
        }

        else if ( junk == 'l' ) shflag = FALSE;
        else if ( junk == 's' ) shflag = TRUE;
        else if ( junk == 'a' ) sortflag = TRUE;
        else if ( junk == 'n' ) sortflag = FALSE;
        else if ( junk == 'd' ) {
            for ( cnt = 0; cnt < Punk->skills; cnt++ )
                if ( shflag )   printf("%d %s\n",cnt,short_skills[Punk->SK[cnt].name]);
                else                printf("%d %s\n",cnt,long_skills[Punk->SK[cnt].name]);
            printf("\nEnter y to continue :");
            checkchar("y");
        }
    }
}


/************************************************************************
*   Vamp Quirks - Procedure generates vampire quirks for a vampire punk.
*
*   vamp_quirks gets global struct Punk, of type CP.  This one drove me nuts.
*   Forces legal choice of at least three quirks, whose total values add
*   up to be zero.  The fbc loop was used here, again.  This proc is always
*   called, but is only complete executed if Punk->vamp is TRUE.
************************************************************************/

void vamp_quirks ( struct CP *Punk )

{
    int cnt,num,tmp,total,roll,doit;
    char junk;

    cnt = total =  num =  tmp = 0; 
   doit = TRUE;

    if ( Punk->vamp ) {

        printf("\nVampire Quirks:\n");

        while (doit) {

            for ( cnt = num; cnt < num + 15; cnt++ ) printf("%2d = %s\n",cnt,vquirks[cnt]);

            printf("Choose at least 3 quirks, total cost must be zero.\n ");
            printf("Number of quirks = %d, Total cost = %d.\n",tmp,total);
            printf("Choose, Forward, Backward (c/f/b) ");
            junk = checkchar( "fbc" );

            if ( junk == 'f' ){
                num = num + 15;
                if ( num > 7 ) num = 7;
            }

            else if ( junk == 'b' ) {
                num = num - 15;
                if ( num < 0 ) num = 0;
            }

            else if ( junk == 'c' ) {
                printf("Quirk number :  "); 
                roll = checknum(0,22);

                switch ( roll ) {
                case 0: for (roll = 0; roll < 10; roll++)
                                printf("%d %-s %3d\n",roll,statistic[roll],Punk->stat[roll]);
                            printf("Which stat to increase by 1 : "); 
                            roll = checknum(0,9);
                            if (Punk->stat[roll] != 10){
                                Punk->stat[roll] += 1;
                                Punk->QU[tmp] = vquirks[0];
                                total += 3;
                                tmp += 1;
                            }
                            break;
                case 1: for (roll = 0; roll < 10; roll++)
                                printf("%d %-s %3d\n",roll,statistic[roll],Punk->stat[roll]);
                            printf("Which stat to decrease by 1 : ");
                            roll = checknum(0,9);
                            if (Punk->stat[roll] != 1){
                                Punk->stat[roll] += -1;
                                Punk->QU[tmp] = vquirks[1];
                                total += -3;
                                tmp += 1;
                            }
                            break;
                case 2: for (roll = 0; roll < Punk->skills; roll++)
                                printf("%d %-s %3d\n",roll,short_skills[Punk->SK[roll].name],Punk->SK[roll].level);
                            printf("Which skill to increase by 1 : ");
                            roll = checknum(0,Punk->skills);
                            if ((Punk->SK[roll].level < 10) && (Punk->SK[roll].psy == FALSE)){
                                Punk->SK[roll].level += 1;
                                Punk->QU[tmp] = vquirks[2];
                                total += 2;
                                tmp += 1;
                            }
                            break;
                case 3: for (roll = 0; roll < Punk->skills; roll++)
                                printf("%d %-s %3d\n",roll,short_skills[Punk->SK[roll].name],Punk->SK[roll].level);
                            printf("Which skill to decrease by 1 : ");
                            roll = checknum(0,Punk->skills);
                            if ((Punk->SK[roll].level != 1) && (Punk->SK[roll].psy == FALSE)){
                                Punk->SK[roll].level += -1;
                                Punk->QU[tmp] = vquirks[3];
                                total += -2;
                                tmp += 1;
                            }
                            break;
                case 4: Punk->QU[tmp] = vquirks[4]; tmp += 1; total += -4; break;
                case 5: Punk->QU[tmp] = vquirks[5]; tmp += 1; total += -5; break;
                case 6: Punk->QU[tmp] = vquirks[6]; tmp += 1; total += -4; break;
                case 7: Punk->QU[tmp] = vquirks[7]; tmp += 1; total += -6; break;
                case 8: Punk->QU[tmp] = vquirks[8]; tmp += 1; total += -6; break;
                case 9: Punk->QU[tmp] = vquirks[9]; tmp += 1; total += -2; break;
                case 10:    Punk->QU[tmp] = vquirks[10]; tmp += 1; total += -6; break;
                case 11:    Punk->QU[tmp] = vquirks[11]; tmp += 1; total += -3; break;
                case 12:    Punk->QU[tmp] = vquirks[12]; tmp += 1; total += -1; break;
                case 13:    Punk->QU[tmp] = vquirks[13]; tmp += 1; total += -3; break;
                case 14:    Punk->QU[tmp] = vquirks[14]; tmp += 1; total += -2; break;
                case 15:    Punk->QU[tmp] = vquirks[15]; tmp += 1; total += -4; break;
                case 16:    Punk->QU[tmp] = vquirks[16]; tmp += 1; total += -4; break;
                case 17:    Punk->QU[tmp] = vquirks[17]; tmp += 1; total += -2; break;
                case 18:    Punk->QU[tmp] = vquirks[18]; tmp += 1; total +=  2; break;
                case 19:    Punk->QU[tmp] = vquirks[19]; tmp += 1; total += -2; break;
                case 20:    Punk->QU[tmp] = vquirks[20]; tmp += 1; total += -2; break;
                case 21:    Punk->QU[tmp] = vquirks[21]; tmp += 1; total += -4; break;
                case 22:    Punk->QU[tmp] = vquirks[22]; tmp += 1; total += -3; break;
                }

                for (roll = 0; roll < tmp; roll++) printf("%s\n",Punk->QU[roll]);
                printf ("\nEnter y to continue ");
                checkchar("y");

                if (( tmp > 2 ) && ( total == 0 )) {
                    printf("You may stop collecting quirks if you wish.  (y/n)");
                    if ( checkchar ("yn") == 'y') doit = FALSE;
                }
            }
        }
        Punk->quirks = tmp;
    }
}


/************************************************************************
*   Wolf Quirks - Procedure generates were-wolf quirks for a wolven punk.
*
*   wolf_quirks gets the global struct Punk, of type CP.  This was easier
*   than vamp_quirks, but is essentially the same proc, but with different
*   values.  This procedure is always called, but is only completely executed
*   if the character is a complete Were-Wolf, rather than a lycanthrope.
************************************************************************/

void wolf_quirks ( struct CP *Punk )

{
    int cnt,total,roll,doit;

    cnt = 0; total = 0; doit = TRUE;

    if ( Punk->wolf == 2 ) {

        printf("\nWerewolf Quirks:\n");

        while (doit) {

            for ( roll = 0; roll < 9; roll++ ) printf("%2d = %s\n",roll,wquirks[roll]);

            printf("Choose at least 2 quirks.  Total cost must be zero.\n");
            printf("Number of quirks = %d, Total cost = %d.\n",cnt,total);
            roll = checknum(0,9);

            switch ( roll ) {
                case 0: for (roll = 0; roll < 10; roll++)
                                printf("%d %-s %3d\n",roll,statistic[roll],Punk->stat[roll]);
                            printf("Which stat to increase by 1 : ");
                            roll = checknum(0,9);
                            if (Punk->stat[roll] != 10){
                                Punk->stat[roll] += 1;
                                Punk->QU[cnt] = wquirks[0];
                                total += 3;
                                cnt += 1;
                            }
                            break;
                case 1: for (roll = 0; roll < 10; roll++)
                                printf("%d %-s %3d\n",roll,statistic[roll],Punk->stat[roll]);
                            printf("Which stat to decrease by 1 : ");
                            roll = checknum(0,9);
                            if (Punk->stat[roll] != 1){
                                Punk->stat[roll] += -1;
                                Punk->QU[cnt] = wquirks[1];
                                total += -3;
                                cnt += 1;
                            }
                            break;
                case 2: for (roll = 0; roll < Punk->skills; roll++)
                                printf("%d %s %d\n",roll,short_skills[Punk->SK[roll].name],Punk->SK[roll].level);
                            printf("Which skill to increase by 1 : ");
                            roll = checknum(0,Punk->skills);
                            if ((Punk->SK[roll].level < 10) && (Punk->SK[roll].psy == FALSE)){
                                Punk->SK[roll].level += 1;
                                Punk->QU[cnt] = wquirks[2];
                                total += 2;
                                cnt += 1;
                            }
                            break;
                case 3: for (roll = 0; roll < Punk->skills; roll++)
                                    printf("%d %s %d\n",roll,short_skills[Punk->SK[roll].name],Punk->SK[roll].level);
                            printf("Which skill to decrease by 1 : ");
                            roll = checknum(0,Punk->skills);
                            if ((Punk->SK[roll].level != 1) && (Punk->SK[roll].psy == FALSE)){
                                    Punk->SK[roll].level += -1;
                                Punk->QU[cnt] = wquirks[3];
                                total += -2;
                                cnt += 1;
                            }
                        break;
                case 4: Punk->QU[cnt] = wquirks[4]; cnt += 1; total += -4; break;
                case 5: Punk->QU[cnt] = wquirks[5]; cnt += 1; total += -6; break;
                case 6: Punk->QU[cnt] = wquirks[6]; cnt += 1; total += -3; break;
                case 7: Punk->QU[cnt] = wquirks[7]; cnt += 1; total += -4; break;
                case 8: Punk->QU[cnt] = wquirks[8]; cnt += 1; total += -4; break;
                case 9: Punk->QU[cnt] = wquirks[9]; cnt += 1; total += -6; break;
            }

            for (roll = 0; roll < cnt; roll++) printf("%s\n",Punk->QU[roll]);
            printf ("\nEnter y to continue ");
            checkchar("y");

            if (( cnt > 1 ) && ( total == 0 )) {
                printf("You may stop collecting quirks if you wish.  (y/n)");
                if ( checkchar ("yn") == 'y') doit = FALSE;
            }
        }
        Punk->QU[cnt] = "+2 Awareness bonus when dealing with smell or hearing.";
        Punk->QU[cnt+1] = "Silver weapons cause 5 times normal damage.";
        Punk->quirks = cnt + 2;
    }
}


/************************************************************************
*   Make person - Procedure creates a person for charcter & lifepath.
*   
*   make_person gets the struct type PERSON.  This proc rolls up a suggested
*   NPC for events during lifepath (lover, enemy, friend), and
*   for the character.  Note that the person's role, etc. are generated
*   in lifepath, rather than here.  This allowed more flexibility.
************************************************************************/

void make_person ( struct PERSON *Person )

{
    int cnt;
    for ( cnt = 0; cnt < 3; cnt ++ )    Person->style[cnt] = die_roll(1,10) - 1;
    Person->race = die_roll(1,10) - 1;
    for ( cnt = 0; cnt < 5; cnt ++ )    Person->mot[cnt] = die_roll(1,10) - 1;
}


/************************************************************************
*   Life Path - Procedure generates character background based on age.
*   
*   life_path gets the global struct Punk, of type CP.  This proc took the
*   longest amount of time, and was the primary reason for writing this
*   program.  It is a great, big switch statement with lots of nested if's,
*   and is run once for each year the character is over 16.
************************************************************************/

void life_path ( struct CP *Punk )

{
    int roll,cnt,tmp;

    printf("\nGenerate Life Path:\n");
    printf("Please enter an age for this charcter (16 <= age <= 28) :");
    Punk->age = checknum (16,28);

    for ( cnt = 0; cnt < Punk->age - 16; cnt++ ) {

    switch ( die_roll(1,10) ) {
        case 1:
        case 2:
        case 3: if ( die_roll (1,10) % 2 ) {
                        Punk->YE[cnt].num   = 4;
                        Punk->YE[cnt].pflag = FALSE;
                        Punk->YE[cnt].event[0]  = "Disaster Strikes";
                        roll = die_roll(1,10);
                        Punk->YE[cnt].event[1]  = disaster[roll-1];
                        if (roll == 1) {
                            Punk->eb    = Punk->eb - (100 * die_roll(1,10));
                            Punk->YE[cnt].event[2] = "You lost some eb in a bad deal";
                        }
                        if (roll == 2) Punk->YE[cnt].event[2] = "Imprisoned for 6 months";
                        if (roll == 3) {
                            Punk->stat[1] += -1;
                            Punk->YE[cnt].event[2] = "You lost 1 point of Reflexes";
                        }
                        if (roll == 4) {
                            tmp = die_roll(1,10);
                            if ( tmp < 4 ) 
                                Punk->YE[cnt].event[2] = "You are being blackmailed";
                            if ( tmp > 3 && tmp < 8 )
                                Punk->YE[cnt].event[2] = "A secret of yours was exposed";
                            if ( tmp > 7 ) 
                                Punk->YE[cnt].event[2] = "You were betrayed by a close friend";
                        }
                        if (roll == 5) {
                            tmp = die_roll(1,10);
                            if ( tmp < 5 ) {
                                Punk->stat[5] += -5;
                                Punk->YE[cnt].event[2] = "Horribly disfigured from accident";
                            }
                            if ( tmp == 5 || tmp == 6 )
                                Punk->YE[cnt].event[2] = "Hospitalized for 1d10 months";
                            if ( tmp == 7 || tmp == 8 ) 
                                Punk->YE[cnt].event[2] = "Lost memory of 1d10 months of this year";
                            if ( tmp == 9 || tmp == 10 ) 
                                Punk->YE[cnt].event[2] = "80% chance of nightmares about accident";
                        }
                        if (roll == 6) {
                            tmp = die_roll(1,10);
                            if ( tmp < 6 ) 
                                Punk->YE[cnt].event[2] = "They died accidentally";
                            if ( tmp > 5 && tmp < 9 )
                                Punk->YE[cnt].event[2] = "They were murdered by unknown parties";
                            if ( tmp > 8 ) 
                                Punk->YE[cnt].event[2] = "They were murdered and you know who did it";
                        }
                        if (roll == 7) {
                            tmp = die_roll(1,10);
                            if ( tmp < 4 ) 
                                Punk->YE[cnt].event[2] = "The accusation is theft";
                            if ( tmp == 4 || tmp == 5 )
                                Punk->YE[cnt].event[2] = "The accusation is cowardace";
                            if ( tmp >5  && tmp < 9 )
                                Punk->YE[cnt].event[2] = "The accusation is murder";
                            if ( tmp == 9 ) 
                                Punk->YE[cnt].event[2] = "The accusation is rape";
                            if ( tmp == 10 ) 
                                Punk->YE[cnt].event[2] = "The accusation is lying or betrayal";
                        }
                        if (roll == 8) {
                            tmp = die_roll(1,10);
                            if ( tmp < 4 )
                                Punk->YE[cnt].event[2] = "Only a couple of local cops want you";
                            if ( tmp > 3 && tmp < 7 )
                                Punk->YE[cnt].event[2] = "The entire local force wants you";
                            if ( tmp > 6 && tmp < 9 )
                                Punk->YE[cnt].event[2] = "The State Police or Militia wants you";
                            if ( tmp > 8 ) 
                                Punk->YE[cnt].event[2] = "The FBI or equivalent national force wants you";
                        }
                        if (roll == 9) {
                            tmp = die_roll(1,10);
                            if ( tmp < 4 ) 
                                Punk->YE[cnt].event[2] = "A small, local firm";
                            if ( tmp > 3 && tmp < 7 )
                                Punk->YE[cnt].event[2] = "A medium, statewide company";
                            if ( tmp > 6 && tmp < 9 )
                                Punk->YE[cnt].event[2] = "A large, national corporation";
                            if ( tmp > 8 ) 
                                Punk->YE[cnt].event[2] = "A huge, multi-national conglomorate with armies, ninjas, etc";
                        }
                        if (roll == 10) {
                            tmp = die_roll(1,10);
                            if ( tmp < 4 ) {
                                Punk->YE[cnt].event[2] = "A nervous disorder";
                                Punk->stat[1] += -1;
                            }
                            if ( tmp > 3 && tmp < 8 ) {
                                Punk->YE[cnt].event[2] = "A mental problem";
                                Punk->stat[2] += -1;
                            }
                            if ( tmp > 7 ) {
                                Punk->YE[cnt].event[2] = "Major psychosis.  Voices, violent, irrational, etc.";
                                Punk->stat[2] += -1;
                                Punk->stat[1] += -1;
                            }
                        }
                        Punk->YE[cnt].event[3] = gonna_do[die_roll(1,5)-1];
                    }
                    else {
                        Punk->YE[cnt].num   = 3;
                        Punk->YE[cnt].pflag     = FALSE;
                        Punk->YE[cnt].event[0] = "You get lucky";
                        roll = die_roll(1,10);
                        Punk->YE[cnt].event[1] = lucky[roll-1];
                        if (roll == 1) {
                            if ( tmp < 5 )
                                Punk->YE[cnt].event[2] = "It's in the Police force";
                            if ( tmp > 4 && tmp < 8 )
                                Punk->YE[cnt].event[2] = "It's in the Distric Attorney's office";
                            if ( tmp > 7 )
                                Punk->YE[cnt].event[2] = "It's in the Mayor's office";
                        }
                        if (roll == 2 || roll == 3) Punk->eb    += (100 * die_roll(1,10));
                        if (roll == 4) Punk->YE[cnt].event[2] = "Start martial art at +2, or add +1 to existing martial art";
                        if (roll == 5) Punk->YE[cnt].event[2] = "Start INT skill at +2, or add +1 to existing INT skill";
                        if (roll == 6) Punk->YE[cnt].event[2] = "So ya better use it wisely, chombata!";
                        if (roll == 7) Punk->YE[cnt].event[2] = "Call on them once/month, at a Family +2 rating";
                        if (roll == 8) Punk->YE[cnt].event[2] = "Call on him for a tip at Streetwise +2 involving police";
                        if (roll == 9) Punk->YE[cnt].event[2] = "Call on them once/month, at a Family +2 level";
                        if (roll ==10) Punk->YE[cnt].event[2] = "Begin combat skill at +2, or add +1 -- Not m. art or brawling";
                    }
                    break;
        case 4:
        case 5:
        case 6: if ( die_roll(1,10) > 5 ) {
                        Punk->YE[cnt].num   = 5;
                        Punk->YE[cnt].pflag     = TRUE;
                        Punk->YE[cnt].event[0] =  "Made an enemy";
                        Punk->YE[cnt].event[1] = enemy[die_roll(1,10)-1];
                        Punk->YE[cnt].event[2] = cause[die_roll(1,10)-1];
                        Punk->YE[cnt].event[3] = whatcha_do[die_roll(1,5)-1];
                        Punk->YE[cnt].event[4] = resources[die_roll(1,10)-1];
                        make_person ( &Punk->YE[cnt].P );
                    }
                    else {
                        Punk->YE[cnt].num   = 2;
                        Punk->YE[cnt].pflag     = TRUE;
                        Punk->YE[cnt].event[0] = "Made a friend";
                        Punk->YE[cnt].event[1] = buddy[die_roll(1,10)-1];
                        make_person ( &Punk->YE[cnt].P );
                    }
                    break;
        case 7:
        case 8: roll = die_roll(1,10);
                    if ( roll < 5 ) {
                        Punk->YE[cnt].num   = 1;
                        Punk->YE[cnt].pflag     = FALSE;
                        Punk->YE[cnt].event[0] = "Happy love affair";
                    }
                    if ( roll == 5 ) {
                        Punk->YE[cnt].num   = 3;
                        Punk->YE[cnt].pflag     = TRUE;
                        Punk->YE[cnt].event[0] = "Tragic love affair";
                        Punk->YE[cnt].event[1] = tragic_love[die_roll(1,10)-1];
                        Punk->YE[cnt].event[2] = mutual_feel[die_roll(1,10)-1];
                        make_person ( &Punk->YE[cnt].P );
                    }
                    if ( roll == 6 || roll == 7 ) {
                        Punk->YE[cnt].num   = 3;
                        Punk->YE[cnt].pflag     = TRUE;
                        Punk->YE[cnt].event[0] = "Love affair with problems";
                        Punk->YE[cnt].event[1] = problem_love[die_roll(1,10)-1];
                        Punk->YE[cnt].event[2] = mutual_feel[die_roll(1,10)-1];
                        make_person ( &Punk->YE[cnt].P );
                    }
                    if ( roll > 7 ) {
                        Punk->YE[cnt].num   = 1;
                        Punk->YE[cnt].pflag     = FALSE;
                        Punk->YE[cnt].event[0] = "Fast affairs and hot dates";
                    }
                    break;
        case 9:
        case 10:    Punk->YE[cnt].num   = 1;
                    Punk->YE[cnt].pflag     = FALSE;
                    Punk->YE[cnt].event[0] = "Nothing happened this year";
                    break;
        default:    break;
    }
    }
}


/************************************************************************
*   Output - Procedure sends created character to user defined text file.
*   
*   output gets the global struct Punk, of type CP.  This writes the Punk
*   to an ASCII file, for maximum portability between platforms.  Long/short
*   skill names are supported here.  Note that filename may not be longer
*   than 40 characters.  This is left to the file system to handle, after
*   white space has been stripped, to allow for maximum portability.
************************************************************************/

void output ( struct CP *Punk )

{
    FILE *fp1;
    char filename[40];
    int cnt,tmp;

    fflush(stdin);
    printf("\nFile Output:\n");
    printf("Please enter file name: "); fgets(filename, 40, stdin);
   stripspc(filename);

    if ( strlen(filename) <= 1 ) strcpy(filename,"charfile");
    fp1 = fopen(filename,"w");

    for ( cnt = 0; cnt < 17; cnt++ )
        fprintf(fp1,"%s = %2d\n",statistic[cnt],Punk->stat[cnt]);

    fprintf(fp1,"\n");

    fprintf(fp1,"Starting Eurobucks        :  %d\n",Punk->eb);
    if ( die_roll(1,6) > 4 )    fprintf(fp1,"You are unemployed\n");
    else                                fprintf(fp1,"Your monthly wage         :  %d\n",Punk->wage);
    fprintf(fp1,"Your role                 :  %s\n",roles[Punk->role]);

    fprintf(fp1,"\n");

    printf("Output long or short skill names? (l/s) ");

    if ( checkchar("sl") == 'l' ) {
        for ( cnt = 0; cnt < Punk->skills; cnt++ )
            fprintf(fp1,"%s %-2d\n",long_skills[Punk->SK[cnt].name],Punk->SK[cnt].level);
    }
    else {
        for ( cnt = 0; cnt < Punk->skills; cnt++ )
            fprintf(fp1,"%s %-2d\n",short_skills[Punk->SK[cnt].name],Punk->SK[cnt].level);
    }

    fprintf(fp1,"\n");

    fprintf(fp1,"Family Background\n");
    for ( cnt = 0; cnt < 4; cnt++ )
        fprintf(fp1,"%s\n",Punk->family[cnt]);

    if ( Punk->sib == 0 )
         fprintf(fp1,"You are an only child\n");
    else {
        for ( cnt = 0; cnt < Punk->sib; cnt++ ) {
            fprintf(fp1,"Sibling #%d\t",cnt);
            fprintf(fp1,"%-10s %-10s %-s\n",
                gender[Punk->SI[cnt].trait[0]],
                older[Punk->SI[cnt].trait[1]],
                sib_feel[Punk->SI[cnt].trait[2]]);
        }
    }

    fprintf(fp1,"\n");

    if ( Punk->vamp ) {

        fprintf(fp1,"You are a vampire.\n");
        fprintf(fp1,"You have %d Vampire Years.\n",Punk->VY);
        fprintf(fp1,"You must feed every %d days.\n",Punk->feed);
        fprintf(fp1,"You heal %d per day.\n",Punk->heal);
        for ( tmp = 0; tmp < Punk->quirks; tmp++)
            fprintf(fp1,"%s\n",Punk->QU[tmp]);
    }
    else
        fprintf(fp1,"You are not a vampire.\n");

    if ( Punk->wolf == 1 ) {

        fprintf(fp1,"You are a lycanthrope.\n");
        for ( tmp = 0; tmp < Punk->quirks; tmp++)
            fprintf(fp1,"%s\n",Punk->QU[tmp]);
    }
    else
        fprintf(fp1,"You are not a lycanthrope.\n");

    if ( Punk->wolf == 2 ) {

        fprintf(fp1,"You are a were-wolf.\n");
        for ( tmp = 0; tmp < Punk->quirks; tmp++)
            fprintf(fp1,"%s\n",Punk->QU[tmp]);
        fprintf(fp1,"You have %d Were-Wolf Years.\n",Punk->VY);
        fprintf(fp1,"You must kill 3 times each full moon.\n");
        fprintf(fp1,"You heal %d per day.\n",Punk->heal);
    }
    else
        fprintf(fp1,"You are not a were-wolf.\n");

    if ( Punk->psy ) {
        fprintf(fp1,"You have psychic powers.\n");
        fprintf(fp1,"PsyCat = %s\n",Punk->psycat);
    }
    else
        fprintf(fp1,"You do not have psychic powers.\n");

    fprintf(fp1,"\n");

    for ( cnt = 0; cnt < Punk->age - 16  ; cnt++ ) {

        fprintf(fp1,"Lifepath %2d:  ",cnt + 16);

        for ( tmp = 0; tmp < Punk->YE[cnt].num; tmp++ )
            fprintf(fp1,"%s\n",Punk->YE[cnt].event[tmp]);

        if ( Punk->YE[cnt].pflag ) {
            fprintf(fp1,"Sex (Change if desired)   :  %s\n",gender[die_roll(1,4)-1]);
            fprintf(fp1,"Their role                :  %s\n",roles[die_roll(1,45)-1]);
            fprintf(fp1,"Clothing                  :  %s\n",clothes[Punk->YE[cnt].P.style[0]]);
            fprintf(fp1,"Hair                      :  %s\n",hairstyle[Punk->YE[cnt].P.style[1]]);
            fprintf(fp1,"Affectations              :  %s\n",affectations[Punk->YE[cnt].P.style[2]]);
            fprintf(fp1,"Ethnicity                 :  %s\n",ethnicity[Punk->YE[cnt].P.race]);
            fprintf(fp1,"Personality Traits        :  %s\n",per_trait[Punk->YE[cnt].P.mot[0]]);
            fprintf(fp1,"Person valued most        :  %s\n",val_person[Punk->YE[cnt].P.mot[1]]);
            fprintf(fp1,"Idea you value most       :  %s\n",val_idea[Punk->YE[cnt].P.mot[2]]);
            fprintf(fp1,"Feelings about most people:  %s\n",feel_people[Punk->YE[cnt].P.mot[3]]);
            fprintf(fp1,"Most valued possession    :  %s\n",val_thing[Punk->YE[cnt].P.mot[4]]);
        }
        fprintf(fp1,"\n");
    }

    fclose(fp1);
}


/************************************************************************
*   Main
*
*   main calls all the previously defined procedures, in order, and does
*   nothing else.  A structured/object oriented programming approach was
*   taken with this project, to allow for maximum future development.
************************************************************************/

void main()
{

    printf("Cyber 3.2 -- A character creation program for the CyberPunk RPG.\r\n");
    printf("Orignal ANSI C coding by Robert Deck <ind00069@pegasus.cc.ucf.edu>\r\n");
    printf("This is giftware -- if you like it, sent something cool!  See docs...\r\n");
    printf("Copyright and trademarks for the CyberPunk game are held by\r\n");
    printf("R. Talsorian Games.  This software is not approved by R. Talsorian Games.\r\n");

    srand(time(NULL));  /* Seed the random number generator via time() */

    sort_skills();

    roll_stats(&Dude);

    move_stats(&Dude);

    get_role(&Dude);

    dist_points(&Dude);

    calc_punk(&Dude);

    vampire_punk(&Dude);

    wolven_punk(&Dude);

    psyber_punk(&Dude);

    pickup_skills(&Dude);

    vamp_quirks(&Dude);

    wolf_quirks(&Dude);

    life_path(&Dude);

    output(&Dude);

}
